package ru.yandex.travel.api.services.hotels.promo;

import java.time.Instant;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.PreDestroy;

import com.google.common.util.concurrent.MoreExecutors;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.proto.TGetActivePromosRsp;
import ru.yandex.travel.hotels.services.promoservice.PromoServiceClient;

@Slf4j
@Service
@EnableConfigurationProperties(CachedActivePromosServiceProperties.class)
public class CachedActivePromosService {
    private final PromoServiceClient promoServiceClient;
    private final ScheduledExecutorService executor;
    private final boolean enabled;

    TGetActivePromosRsp lastRsp;

    public CachedActivePromosService(PromoServiceClient promoServiceClient, CachedActivePromosServiceProperties properties) {
        this.promoServiceClient = promoServiceClient;
        this.executor = Executors.newScheduledThreadPool(1);
        this.enabled = properties.isEnabled();
        if (this.enabled) {
            executor.scheduleWithFixedDelay(this::UpdateData, 0, properties.getRenewPeriod().toSeconds(), TimeUnit.SECONDS);
        }
    }

    @PreDestroy
    public void shutdown() {
        MoreExecutors.shutdownAndAwaitTermination(this.executor, 100, TimeUnit.MILLISECONDS);
    }

    private void UpdateData() {
        try {
            var rsp = promoServiceClient.getActivePromos(Instant.now()).join();
            synchronized (this) {
                lastRsp = rsp;
            }
        } catch (Exception e) {
            log.warn("Failed to get active promos", e);
        }
    }

    public TGetActivePromosRsp getActivePromos() {
        if (!enabled) {
            throw new RuntimeException("CachedActivePromosService is not enabled");
        }
        synchronized (this) {
            return lastRsp;
        }
    }
}
