package ru.yandex.travel.api.services.hotels.redir;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;

import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.AhcHttpRetryStrategy;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryRateLimiter;

@Component
@EnableConfigurationProperties(HotelsRedirServiceProperties.class)
public class HotelsRedirService {

    private static final AhcHttpRetryStrategy RETRY_STRATEGY = new AhcHttpRetryStrategy(Duration.ZERO, 3);

    private final AsyncHttpClientWrapper httpClient;
    private final Retry retryHelper;
    private final RetryRateLimiter retryRateLimiter;

    private final String redirBaseUrl;

    private final int httpRequestTimeoutMs;
    private final int httpReadTimeoutMs;

    public HotelsRedirService(@Qualifier(value = "hotelRedirAhcWrapper") AsyncHttpClientWrapper httpClient,
                              Retry retryHelper,
                              HotelsRedirServiceProperties hotelsRedirServiceProperties) {
        this.httpClient = httpClient;
        this.retryHelper = retryHelper;
        this.retryRateLimiter = new RetryRateLimiter(hotelsRedirServiceProperties.getRetryRateLimit());

        this.redirBaseUrl = hotelsRedirServiceProperties.getBaseUrl();

        this.httpRequestTimeoutMs = Math.toIntExact(hotelsRedirServiceProperties.getHttpRequestTimeout().toMillis());
        this.httpReadTimeoutMs = Math.toIntExact(hotelsRedirServiceProperties.getHttpReadTimeout().toMillis());
    }

    public CompletableFuture<String> getToursLabelHash(String requestId, String label) {
        final RequestBuilder requestBuilder = new RequestBuilder()
                .setReadTimeout(httpReadTimeoutMs)
                .setRequestTimeout(httpRequestTimeoutMs)
                .setUrl(redirBaseUrl + "/tours/label_to_hash")
                .addQueryParam("LabelParams", label);

        return withRetry(requestBuilder, requestId, "HotelRedirService::ToursLabelToHash")
                .thenApply(Response::getResponseBody);
    }

    private CompletableFuture<Response> withRetry(RequestBuilder requestBuilder, String requestId, String name) {
        return retryHelper.withRetry(
                name,
                rb -> httpClient.executeRequest(rb, "GET", requestId).toCompletableFuture(),
                requestBuilder,
                RETRY_STRATEGY,
                retryRateLimiter
        );
    }
}
