package ru.yandex.travel.api.services.hotels.region_hotel_count;

import java.util.Collections;
import java.util.Optional;
import java.util.OptionalInt;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.IntPoint;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;

@Component
@EnableConfigurationProperties(RegionHotelCountServiceProperties.class)
@Slf4j
public class RegionHotelCountService {
    private static final int INDEX_STRUCTURE_VERSION = 1;

    private static final String FIELD_GEO_ID = "i";
    private static final String FIELD_FILTER_SLUG = "f";
    private static final String FIELD_HOTEL_COUNT = "c";

    private final YtLuceneIndex countersLuceneIndex;

    public RegionHotelCountService(RegionHotelCountServiceProperties params) {
        if (params.isEnabled()) {
            countersLuceneIndex = new YsonYtLuceneIndex(params, "RegionHotelCount", INDEX_STRUCTURE_VERSION, (row) -> {
                Document document = new Document();

                final int geoId = row.getInt("GeoId");
                document.add(new IntPoint(FIELD_GEO_ID, geoId));

                final String filterSlug = row.getStringO("SeoPageFilterSlug").orElse("");
                document.add(new StringField(FIELD_FILTER_SLUG, filterSlug, Field.Store.YES));
                document.add(new StoredField(FIELD_FILTER_SLUG, filterSlug));

                final Optional<Integer> hotelCount = row.getIntO("HotelCountForTitle");
                if (hotelCount.isPresent()) {
                    document.add(new StoredField(FIELD_HOTEL_COUNT, hotelCount.get()));
                }

                return Collections.singletonList(document);
            });
        } else {
            log.warn("Region counters index disabled");
            countersLuceneIndex = null;
        }
    }

    @PostConstruct
    public void init() {
        if (countersLuceneIndex != null) {
            countersLuceneIndex.start();
        }
    }

    @PreDestroy
    public void destroy() {
        if (countersLuceneIndex != null) {
            countersLuceneIndex.stop();
        }
    }

    public boolean isReady() {
        return countersLuceneIndex == null || countersLuceneIndex.isReady();
    }

    public OptionalInt getHotelCount(int geoId) {
        return getHotelCount(geoId, "");
    }

    public OptionalInt getHotelCount(int geoId, String filterSlug) {
        final String filterSlugNotNull = (filterSlug != null) ? filterSlug : "";

        OptionalInt result;
        if (countersLuceneIndex != null) {
            result = countersLuceneIndex.search(searcher -> {
                Query geoIdQuery = IntPoint.newExactQuery(FIELD_GEO_ID, geoId);
                Query filterSlugQuery = new TermQuery(new Term(FIELD_FILTER_SLUG, filterSlugNotNull));
                Query query = new BooleanQuery.Builder()
                        .add(new BooleanClause(geoIdQuery, BooleanClause.Occur.MUST))
                        .add(new BooleanClause(filterSlugQuery, BooleanClause.Occur.MUST))
                        .build();
                TopDocs topDocs = searcher.search(query, 1);
                if (topDocs.totalHits > 0) {
                    Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                    return OptionalInt.of(document.getField(FIELD_HOTEL_COUNT).numericValue().intValue());
                } else {
                    return OptionalInt.empty();
                }
            });
        } else {
            result = OptionalInt.empty();
        }
        return result;
    }
}
