package ru.yandex.travel.api.services.hotels.region_images;

import java.util.function.Function;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterators;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.hotels.proto.TRegionImage;
import ru.yandex.travel.yt_lucene_index.MultiMapPersistentConfig;

@Component
@EnableConfigurationProperties(RegionImagesServiceProperties.class)
@Slf4j
public class RegionImagesService  {
    private final String ORIG_SIZE_SUFFIX = "orig";

    private MultiMapPersistentConfig<Integer, String, TRegionImage> persistentConfig;
    private GeoBase geoBase;

    public RegionImagesService(RegionImagesServiceProperties params, GeoBase geoBase) {
        this.geoBase = geoBase;
        if (params.isEnabled()) {
            persistentConfig = new MultiMapPersistentConfig<>(params, "RegionImages",
                    TRegionImage::newBuilder, TRegionImage::getGeoId, TRegionImage::getUrl, Function.identity());
        } else {
            persistentConfig = null;
        }
    }

    private void ensureEnabled() {
        if (persistentConfig == null) {
            throw new IllegalStateException("RegionImages service disabled");
        }
    }

    private String replaceSize(String url, String size) {
        Preconditions.checkArgument(url.endsWith(ORIG_SIZE_SUFFIX), "Image url %s does not end with orig size suffix", url);
        return url.substring(0, url.length() - ORIG_SIZE_SUFFIX.length()) + size;
    }

    public String getImageExact(int geoId) {
        ensureEnabled();
        var result = persistentConfig.getByKey(geoId);
        if (result != null && !result.isEmpty()) {
            return result.iterator().next();
        }
        return null;
    }

    public String getImageExactWithSize(int geoId, String size) {
        String result = getImageExact(geoId);
        if (result != null) {
            return replaceSize(result, size);
        }
        return null;
    }

    public String getImageUsingTree(int geoId) {
        ensureEnabled();
        for (int parentGeoId: geoBase.getParentsIds(geoId)) {
            var images = persistentConfig.getByKey(parentGeoId);
            if (images != null && !images.isEmpty()) {
                return Iterators.get(images.iterator(), geoId % images.size());
            }
        }
        return null;
    }

    public String getImageUsingTreeWithSize(int geoId, String size) {
        String result = getImageUsingTree(geoId);
        if (result != null) {
            return replaceSize(result, size);
        }
        return null;
    }

    @PostConstruct
    public void init() {
        if (persistentConfig != null) {
            persistentConfig.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (persistentConfig != null) {
            persistentConfig.stop();
        }
    }

    public boolean isReady() {
        return persistentConfig == null || persistentConfig.isReady();
    }
}
