package ru.yandex.travel.api.services.hotels.region_name;

import java.util.Collections;
import java.util.List;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.models.Linguistics;
import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;

@Component
@EnableConfigurationProperties(ru.yandex.travel.api.services.hotels.region_name.RegionNameServiceProperties.class)
@Slf4j
public class RegionNameService {

    private static final int INDEX_STRUCTURE_VERSION = 1;
    private static final String FIELD_NAME_SLUG = "slug";
    private static final String FIELD_NAME_NOMINATIVE = "nominative";
    private static final String FIELD_NAME_GENITIVE = "genitive";
    private static final String FIELD_NAME_DATIVE = "dative";
    private static final String FIELD_NAME_ACCUSATIVE = "accusative";
    private static final String FIELD_NAME_INSTRUMENTAL = "instrumental";
    private static final String FIELD_NAME_PREPOSITIONAL = "prepositional";
    private static final String FIELD_NAME_PREPOSITION = "preposition";
    private final List<String> linguisticFields = List.of(
            FIELD_NAME_NOMINATIVE,
            FIELD_NAME_GENITIVE,
            FIELD_NAME_DATIVE,
            FIELD_NAME_ACCUSATIVE,
            FIELD_NAME_INSTRUMENTAL,
            FIELD_NAME_PREPOSITIONAL,
            FIELD_NAME_PREPOSITION
    );

    private final YtLuceneIndex regionNameIndex;

    public RegionNameService(RegionNameServiceProperties config) {
        if (config.isEnabled()) {
            regionNameIndex = new YsonYtLuceneIndex(config, "RegionName", INDEX_STRUCTURE_VERSION, (row) -> {
                Document document = new Document();

                String slug = row.getString(FIELD_NAME_SLUG);
                document.add(new StringField(FIELD_NAME_SLUG, slug, Field.Store.NO));

                for (String fieldName : linguisticFields) {
                    String fieldValue = row.getString(fieldName);
                    document.add(new StoredField(fieldName, fieldValue));
                }

                return Collections.singletonList(document);
            });
        } else {
            log.warn("Region name index disabled");
            regionNameIndex = null;
        }
    }

    @PostConstruct
    public void init() {
        if (regionNameIndex != null) {
            regionNameIndex.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (regionNameIndex != null) {
            regionNameIndex.stop();
        }
    }

    public boolean isReady() {
        return regionNameIndex == null || regionNameIndex.isReady();
    }

    public Linguistics getLinguisticBySlug(String slug) {
        if (regionNameIndex == null) {
            return null;
        }

        Document document = regionNameIndex.search(searcher -> {
            Query query = new TermQuery(new Term(FIELD_NAME_SLUG, slug));
            TopDocs topDocs = searcher.search(query, 1);
            if (topDocs.totalHits > 0) {
                return searcher.doc(topDocs.scoreDocs[0].doc);
            }
            return null;
        });
        if (document == null) {
            return null;
        }
        Linguistics linguistics = new Linguistics();
        linguistics.setNominativeCase(document.getField(FIELD_NAME_NOMINATIVE).stringValue());
        linguistics.setGenitiveCase(document.getField(FIELD_NAME_GENITIVE).stringValue());
        linguistics.setDativeCase(document.getField(FIELD_NAME_DATIVE).stringValue());
        linguistics.setPrepositionalCase(document.getField(FIELD_NAME_PREPOSITIONAL).stringValue());
        linguistics.setPreposition(document.getField(FIELD_NAME_PREPOSITION).stringValue());
        linguistics.setLocativeCase("");
        linguistics.setDirectionalCase("");
        linguistics.setAblativeCase("");
        linguistics.setAccusativeCase(document.getField(FIELD_NAME_ACCUSATIVE).stringValue());
        linguistics.setInstrumentalCase(document.getField(FIELD_NAME_INSTRUMENTAL).stringValue());
        return linguistics;
    }
}
