package ru.yandex.travel.api.services.hotels.regions;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.io.Resources;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.geobase6.RegionHash;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.travel.api.models.Linguistics;
import ru.yandex.travel.api.models.Region;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.region_name.RegionNameService;
import ru.yandex.travel.api.services.hotels.slug.RegionSlugService;


@Component
@Slf4j
public class RegionsService {
    private final RegionNameService regionNameService;
    private final RegionSlugService regionSlugService;
    private final GeoBase geoBase;
    private final Map<Integer, RegionIndexItem> regions;

    public RegionsService(
            RegionNameService regionNameService,
            RegionSlugService regionSlugService,
            GeoBase geoBase,
            ObjectMapper mapper
    ) throws IOException {
        this.regionNameService = regionNameService;
        this.regionSlugService = regionSlugService;
        this.geoBase = geoBase;
        String json = Resources.toString(Resources.getResource("hotels/regions.json"), Charset.forName("UTF-8"));
        regions = mapper.readValue(json, new TypeReference<Map<Integer, RegionIndexItem>>() {});
    }

    public String getRegionName(int geoId, String domain) {
        RegionIndexItem rii = regions.get(geoId);
        if (rii != null && rii.getName() != null) {
            return rii.getName();
        }
        return GeoBaseHelpers.getRegionName(geoBase, domain, geoId);
    }

    public int getRegionType(int geoId, String domain) {
        RegionHash hash = geoBase.getRegionById(geoId, domain);
        return hash.getAttr("type").getInteger();
    }

    public Region getRegion(int geoId, String domain, String lang) {
        Region reg = new Region();
        reg.setGeoId(geoId);
        RegionHash hash = geoBase.getRegionById(geoId, domain);
        reg.setType(hash.getAttr("type").getInteger());
        reg.setSlug(regionSlugService.getSlugByGeoId(geoId));
        Linguistics regionLinguistics = null;
        if (reg.getSlug() != null) {
            regionLinguistics = regionNameService.getLinguisticBySlug(reg.getSlug());
        }
        if (regionLinguistics == null) {
            regionLinguistics = GeoBaseHelpers.getRegionLinguistics(geoBase, geoId, lang);
        }
        reg.setLinguistics(regionLinguistics);

        Coordinates coordinates = new Coordinates(
            hash.getAttr("latitude").getDouble(),
            hash.getAttr("longitude").getDouble()
        );
        reg.setCoordinates(coordinates);
        return reg;
    }

    public Integer getRegionRoundTo(double latitude, double longitude, int regionType, String domain) {
        return getRegionRoundTo(geoBase.getRegionIdByLocation(latitude, longitude), regionType, domain);
    }

    public Integer getRegionRoundTo(int geoId, int regionType, String domain) {
        return GeoBaseHelpers.getRegionRoundTo(geoBase, geoId, regionType, domain);
    }

    public int getPreferredRegion(double latitude, double longitude, Set<Integer> preferredRegionTypes, String domain) {
        return getPreferredGeoId(geoBase.getRegionIdByLocation(latitude, longitude), preferredRegionTypes, domain);
    }

    public int getPreferredGeoId(int geoId, Set<Integer> preferredRegionTypes, String domain) {
        return GeoBaseHelpers.getPreferredGeoId(geoBase, geoId, preferredRegionTypes, domain);
    }
}
