package ru.yandex.travel.api.services.hotels.search_log;

import java.time.Instant;
import java.util.List;
import java.util.Objects;

import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.infrastucture.ProtobufJsonLogger;
import ru.yandex.travel.api.proto.hotels_portal.THotelsSearchLogRecord;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.hotels.common.Ages;

@Component
@RequiredArgsConstructor
@Slf4j
public class SearchLogger {
    private final ProtobufJsonLogger logger = new ProtobufJsonLogger("ru.yandex.travel.api.services.hotels" +
            ".search_log.SearchLogger");

    @SneakyThrows
    public void log(SearchLogItem item, Instant time, CommonHttpHeaders headers) {
        var logRequestBuilder = THotelsSearchLogRecord.TRequest.newBuilder();
        if (item.getGeoId() != null) {
            logRequestBuilder.setGeoId(item.getGeoId());
        }
        if (item.getPermalink() != null) {
            logRequestBuilder.setPermalink(item.getPermalink());
        }
        if (item.getSearchReason() != null) {
            logRequestBuilder.setSearchReason(item.getSearchReason());
        }

        var offerSearchParams = mapOfferSearchParams(item);
        if (offerSearchParams == null) {
            return;
        }

        var logHeadersBuilder = THotelsSearchLogRecord.TCommonHttpHeaders.newBuilder();
        if (headers.getPassportId() != null) {
            logHeadersBuilder.setPassportUid(headers.getPassportId());
        }
        if (headers.getYandexUid() != null) {
            logHeadersBuilder.setYandexUid(headers.getYandexUid());
        }
        logger.log(THotelsSearchLogRecord.newBuilder()
                .setRequest(logRequestBuilder)
                .setResponse(THotelsSearchLogRecord.TResponse.newBuilder()
                        .setOfferCacheParams(offerSearchParams)
                        .setFoundHotelCount(item.getFoundHotelCount()))
                .setRequestHeaders(logHeadersBuilder)
                .setRequestTimestamp((int)time.getEpochSecond())
                .build());
    }

    private THotelsSearchLogRecord.TOfferSearchParams mapOfferSearchParams(SearchLogItem item) {
        var offerSearchParamsBuilder = THotelsSearchLogRecord.TOfferSearchParams.newBuilder();
        var offerSearchParams = item.getOfferSearchParams();
        if (item.getOfferSearchParams() == null) {
            return null;
        }
        if (offerSearchParams.getCheckinDate() != null) {
            offerSearchParamsBuilder.setCheckInDate(offerSearchParams.getCheckinDate().toString());
        }
        if (offerSearchParams.getCheckoutDate() != null) {
            offerSearchParamsBuilder.setCheckOutDate(offerSearchParams.getCheckoutDate().toString());
        }
        if (offerSearchParams.getAdults() != null) {
            offerSearchParamsBuilder.setAges(
                    Ages.build(
                            offerSearchParams.getAdults(),
                            Objects.requireNonNullElse(
                                    offerSearchParams.getChildrenAges(),
                                    List.of()
                            )
                    ).toString()
            );
        }

        return offerSearchParamsBuilder.build();
    }


}
