package ru.yandex.travel.api.services.hotels.searcher;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Preconditions;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.ERequestClass;
import ru.yandex.travel.hotels.proto.OfferSearchServiceV1Grpc;
import ru.yandex.travel.hotels.proto.THotelId;
import ru.yandex.travel.hotels.proto.THotelTestContext;
import ru.yandex.travel.hotels.proto.TSearchOffersReq;
import ru.yandex.travel.hotels.proto.TSearchOffersRpcReq;
import ru.yandex.travel.hotels.proto.TSearchOffersRpcRsp;
import ru.yandex.travel.hotels.proto.TSearchOffersRsp;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;

@RequiredArgsConstructor
@Getter
@Setter
public class SearcherClient {
    private final HAGrpcChannelFactory searcherGrpcChannelFactory;

    public CompletableFuture<List<TSearchOffersRsp>> findOffers(EPartnerId partnerId, String hotelId, long permalink,
                                                                LocalDate checkinDate, LocalDate checkoutDate,
                                                                String occupancyString, THotelTestContext testContext) {
        Preconditions.checkNotNull(partnerId);
        Preconditions.checkNotNull(hotelId);
        Preconditions.checkNotNull(checkinDate);
        Preconditions.checkNotNull(checkoutDate);
        Preconditions.checkNotNull(occupancyString);

        var stub = OfferSearchServiceV1Grpc.newFutureStub(searcherGrpcChannelFactory.getAnyChannel());
        TSearchOffersReq requset = TSearchOffersReq.newBuilder()
                .setHotelId(THotelId.newBuilder()
                        .setPartnerId(partnerId)
                        .setOriginalId(hotelId)
                        .build())
                .setCheckInDate(checkinDate.format(DateTimeFormatter.ISO_DATE))
                .setCheckOutDate(checkoutDate.format(DateTimeFormatter.ISO_DATE))
                .setOccupancy(occupancyString)
                .setCurrency(ECurrency.C_RUB)
                .setRequestClass(ERequestClass.RC_INTERACTIVE)
                .setId(ProtoUtils.randomId())
                .setPermalink(permalink)
                .build();
        TSearchOffersRpcReq rpcRequest = TSearchOffersRpcReq.newBuilder()
                .addSubrequest(requset)
                .setSync(true)
                .setTestContext(testContext)
                .build();
        return FutureUtils.buildCompletableFuture(stub.searchOffers(rpcRequest)).thenApply(TSearchOffersRpcRsp::getSubresponseList);
    }
}
