package ru.yandex.travel.api.services.hotels.serp_popular_destinations;

import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.geobase6.LookupException;
import ru.yandex.geobase6.RegionHash;
import ru.yandex.travel.api.models.hotels.SerpDestinationResponseItem;
import ru.yandex.travel.api.proto.serp.TSerpPopularDestination;
import ru.yandex.travel.api.services.cache.SimpleSyncCache;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.region_images.RegionImagesService;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.yt_lucene_index.MultiMapPersistentConfig;


@Component
@EnableConfigurationProperties(SerpPopularDestinationsServiceProperties.class)
@Slf4j
public class SerpPopularDestinationsService {
    private static class UnknownSourceIdException extends RuntimeException {
    }

    private final String DOMAIN = "ru";
    private final String LANG = "ru";
    private final int RESULTS_MAX = 15;
    private final int RESULTS_MIN = 5;
    private final int INVALIDATE_ATTEMPS = 3;

    private final RegionImagesService regionImagesService;
    private final RegionsService regionsService;
    private final GeoBase geoBase;

    private final boolean isEnabled;
    private final String portalPrefix;
    private final String title;
    private final String imageSize;
    private final MultiMapPersistentConfig<Integer, TSerpPopularDestination, TSerpPopularDestination> persistentConfig;
    private final SimpleSyncCache<Integer, Integer> sourceGeoIdCache;
    private final SimpleSyncCache<Integer, List<SerpDestinationResponseItem>> resultCache;

    public SerpPopularDestinationsService(SerpPopularDestinationsServiceProperties params,
                                          RegionImagesService regionImagesService, RegionsService regionsService,
                                          GeoBase geoBase) {
        this.regionImagesService = regionImagesService;
        this.regionsService = regionsService;
        this.geoBase = geoBase;
        this.isEnabled = params.isEnabled();
        this.portalPrefix = params.getPortalPrefix();
        this.title = params.getTitle();
        this.imageSize = params.getImageSize();
        if (isEnabled) {
            persistentConfig = new MultiMapPersistentConfig<>(params, "SerpPopularDestinations",
                    TSerpPopularDestination::newBuilder,
                    TSerpPopularDestination::getSourceGeoId, (d) -> d,
                    (items) -> items.stream().sorted(Comparator.comparingInt((x) -> -x.getCnt())).collect(Collectors.toList())
            );
        } else {
            persistentConfig = null;
            log.warn("SerpPopularDestinationService is disabled");
        }
        this.sourceGeoIdCache = new SimpleSyncCache<>(null, params.getCacheDuration(), "serpPopularDestinations",
                "sourceGeoIdCache");
        this.resultCache = new SimpleSyncCache<>(null, params.getCacheDuration(), "serpPopularDestinations",
                "resultCache");
    }

    @PostConstruct
    public void init() {
        if (persistentConfig != null) {
            persistentConfig.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (persistentConfig != null) {
            persistentConfig.stop();
        }
    }

    public boolean isReady() {
        return persistentConfig == null || persistentConfig.isReady();
    }

    private void ensureEnabled() {
        if (!isEnabled) {
            throw new RuntimeException("DestinationService is not enabled in config");
        }
    }

    private int getSourceId(int currentId) {
        try {
            while (currentId != GeoBaseHelpers.WORLD_REGION) {
                RegionHash hash = geoBase.getRegionById(currentId, DOMAIN);
                Integer type = hash.getAttr("type").getInteger();
                if (type == GeoBaseHelpers.CITY_REGION_TYPE && persistentConfig.containsKey(currentId)) {
                    return currentId;
                }
                Integer capitalId = hash.getAttr("capital_id").getInteger();
                if (capitalId != 0 && persistentConfig.containsKey(capitalId)) {
                    return capitalId;
                }
                if (type == GeoBaseHelpers.REGION_REGION_TYPE) {
                    break;
                }
                currentId = geoBase.getParentId(currentId, DOMAIN);
            }
        } catch (LookupException ex) {
            log.warn("Unknown region", ex);
        }
        return GeoBaseHelpers.WORLD_REGION;
    }

    public String getTitle() {
        return title;
    }

    public List<SerpDestinationResponseItem> getPopularDestinations(int userGeoId, String domain) {
        ensureEnabled();
        if (!DOMAIN.equals(domain)) {
            throw new NotImplementedException(String.format("Only domain == %s is supported", DOMAIN));
        }
        for (int att = 0; att < INVALIDATE_ATTEMPS; ++att) {
            int sourceId = sourceGeoIdCache.getOrCompute(userGeoId, this::getSourceId);
            try {
                return resultCache.getOrCompute(sourceId, (x) -> calculatePopularDestinationsForKey(sourceId, domain));
            } catch (UnknownSourceIdException exc) {
                log.warn("Invalid source geoId: {}", sourceId);
                sourceGeoIdCache.invalidate(sourceId);
                resultCache.invalidate(sourceId);
            }
        }
        throw new IllegalStateException(String.format("Cannot generate response for %s/%s", userGeoId, domain));
    }

    private List<SerpDestinationResponseItem> calculatePopularDestinationsForKey(int geoId, String domain) {
        Collection<TSerpPopularDestination> pds = persistentConfig.getByKey(geoId);
        if (pds == null) {
            // This may happen if persistent config is updated, and sourceGeoIdCache contains outdated data
            throw new UnknownSourceIdException();
        }
        List<SerpDestinationResponseItem> items = pds.stream()
                .map(pd -> new SerpDestinationResponseItem(
                        pd.getTargetGeoId(),
                        regionsService.getRegionName(pd.getTargetGeoId(), domain),
                        regionImagesService.getImageExactWithSize(pd.getTargetGeoId(), imageSize),
                        GeoBaseHelpers.getRegionDescription(
                                geoBase,
                                domain,
                                pd.getTargetGeoId(),
                                false,
                                false,
                                GeoBaseHelpers.DEFAULT_ALLOWED_REGION_TYPES,
                                new HashSet<Integer>(),
                                false
                        ),
                        pd.getMedianMinPricePerNight(), pd.getMedianNights(),
                        generateUrl(pd)
                ))
                .filter(ri -> Strings.isNotBlank(ri.getImage()))
                .limit(RESULTS_MAX)
                .collect(Collectors.toList());
        if (items.size() < RESULTS_MIN && geoId != GeoBaseHelpers.WORLD_REGION) {
            return calculatePopularDestinationsForKey(GeoBaseHelpers.WORLD_REGION, domain);
        }
        return items;
    }

    private String generateUrl(TSerpPopularDestination pd) {
        return portalPrefix + "&geoId=" + pd.getTargetGeoId();
    }

}
