package ru.yandex.travel.api.services.hotels.slug;

import java.util.Collections;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.LongPoint;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.infrastucture.TravelPreconditions;
import ru.yandex.travel.api.models.hotels.interfaces.HotelIdentifierProvider;
import ru.yandex.travel.hotels.common.HotelNotFoundException;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;

@Component
@EnableConfigurationProperties(HotelSlugServiceProperties.class)
@Slf4j
public class HotelSlugService {
    private static final int INDEX_STRUCTURE_VERSION = 2;
    private static final String PERMALINK_SLUG_PREFIX = "permalink/";
    private static final String SHOW_PERMALINK_PREFIX = "~";

    private static final String FIELD_CLUSTER_PERMALINK = "cp";
    private static final String FIELD_ANY_PERMALINK = "p";
    private static final String FIELD_SLUG = "s";

    private final boolean showPermalinkEnabled;
    private final YtLuceneIndex slugLuceneIndex;

    // Slug -> Permalink
    private final Counter slug2PermalinkFoundCounter;
    private final Counter slug2PermalinkNotFoundCounter;

    // Permalink -> slug
    private final Counter permalink2SlugFoundCounter;
    private final Counter permalink2SlugNotFoundCounter;

    public HotelSlugService(HotelSlugServiceProperties params) {
        if (params.isEnabled()) {
            slugLuceneIndex = new YsonYtLuceneIndex(params, "HotelSlugs", INDEX_STRUCTURE_VERSION, (row) -> {
                Document document = new Document();
                long clusterPermalink = row.getLong("cluster_permalink");
                document.add(new StoredField(FIELD_CLUSTER_PERMALINK, clusterPermalink));
                document.add(new LongPoint(FIELD_ANY_PERMALINK, clusterPermalink));
                for (YTreeNode listNode : row.getList("other_permalinks")) {
                    long permalink = listNode.longValue();
                    if (permalink != clusterPermalink) { // Was explicitly added before
                        document.add(new LongPoint(FIELD_ANY_PERMALINK, permalink));
                    }
                }
                String mainSlug = row.getString("main_slug");
                document.add(new StringField(FIELD_SLUG, mainSlug, Field.Store.YES));
                for (YTreeNode listNode : row.getList("other_slugs")) {
                    String slug = listNode.stringValue();
                    if (!slug.equals(mainSlug)) {// Was explicitly added before
                        document.add(new StringField(FIELD_SLUG, slug, Field.Store.NO));
                    }
                }
                return Collections.singletonList(document);
            });
        } else {
            log.warn("Hotel slug index disabled");
            slugLuceneIndex = null;
        }
        showPermalinkEnabled = params.isShowPermalinkEnabled();
        slug2PermalinkFoundCounter = Counter.builder("hotels.slug.slug2permalink.found").register(Metrics.globalRegistry);
        slug2PermalinkNotFoundCounter = Counter.builder("hotels.slug.slug2permalink.notFound").register(Metrics.globalRegistry);
        permalink2SlugFoundCounter = Counter.builder("hotels.slug.permalink2slug.found").register(Metrics.globalRegistry);
        permalink2SlugNotFoundCounter = Counter.builder("hotels.slug.permalink2slug.notFound").register(Metrics.globalRegistry);
    }

    @PostConstruct
    public void init() {
        if (slugLuceneIndex != null) {
            slugLuceneIndex.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (slugLuceneIndex != null) {
            slugLuceneIndex.stop();
        }
    }

    public boolean isReady() {
        return slugLuceneIndex == null || slugLuceneIndex.isReady();
    }

    public Permalink findPermalinkByHotelIdentifier(HotelIdentifierProvider hotelId) {
        TravelPreconditions.checkRequestArgument(hotelId.getPermalink() != null || hotelId.getHotelSlug() != null,
                "permalink or hotelSlug should be given");
        TravelPreconditions.checkRequestArgument(hotelId.getPermalink() == null || hotelId.getHotelSlug() == null,
                "permalink and hotelSlug should not be given simultaneously");
        if (hotelId.getPermalink() != null) {
            return hotelId.getPermalink();
        }
        // Search by slug
        String slug;
        if (hotelId.getHotelSlug().startsWith(SHOW_PERMALINK_PREFIX)) {
            slug = hotelId.getHotelSlug().substring(SHOW_PERMALINK_PREFIX.length());
        } else {
            slug = hotelId.getHotelSlug();
        }
        if (slug.startsWith(PERMALINK_SLUG_PREFIX)) {
            slug2PermalinkFoundCounter.increment();
            try {
                return Permalink.of(slug.substring(PERMALINK_SLUG_PREFIX.length()));
            } catch (IllegalArgumentException e) {
                throw new TravelApiBadRequestException(String.format("Invalid slug: %s", slug), e);
            }
        }
        Permalink permalink = null;
        if (slugLuceneIndex != null) {
            permalink = slugLuceneIndex.search(searcher -> {
                    Query query = new TermQuery(new Term(FIELD_SLUG, slug));
                    TopDocs topDocs = searcher.search(query, 1);
                    if (topDocs.totalHits > 0) {
                        Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                        return Permalink.of(document.getField(FIELD_CLUSTER_PERMALINK).numericValue().longValue());
                    }
                    return null;
                });
        }
        if (permalink != null) {
            slug2PermalinkFoundCounter.increment();
            return permalink;
        }
        slug2PermalinkNotFoundCounter.increment();
        throw new HotelNotFoundException(String.format("Permalink not found by slug '%s'", slug));
    }

    public String findMainSlug(HotelIdentifierProvider hotelId, Permalink permalink) {
        if (showPermalinkEnabled && hotelId != null && hotelId.getHotelSlug() != null && hotelId.getHotelSlug().startsWith(SHOW_PERMALINK_PREFIX)) {
            return SHOW_PERMALINK_PREFIX + PERMALINK_SLUG_PREFIX + permalink.toString();
        }
        return findMainSlugByPermalink(permalink);
    }

    public String findMainSlugByPermalink(Permalink permalink) {
        String slug = null;
        if (slugLuceneIndex != null) {
            slug = slugLuceneIndex.search(searcher -> {
                Query query = LongPoint.newExactQuery(FIELD_ANY_PERMALINK, permalink.asLong());
                TopDocs topDocs = searcher.search(query, 1);
                if (topDocs.totalHits > 0) {
                    Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                    return document.getField(FIELD_SLUG).stringValue();
                }
                return null;
            });
        }
        if (slug != null) {
            permalink2SlugFoundCounter.increment();
            return slug;
        }
        log.warn("Slug not found by permalink {}", permalink);
        permalink2SlugNotFoundCounter.increment();
        return PERMALINK_SLUG_PREFIX + permalink.toString();
    }
}
