package ru.yandex.travel.api.services.hotels.slug;

import java.util.Collections;
import java.util.Optional;
import java.util.OptionalInt;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.IntPoint;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.api.infrastucture.TravelPreconditions;
import ru.yandex.travel.hotels.common.HotelNotFoundException;
import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;

@Component
@EnableConfigurationProperties(RegionSlugServiceProperties.class)
@Slf4j
public class RegionSlugService {
    private static final int INDEX_STRUCTURE_VERSION = 4;

    private static final String FIELD_SLUG = "s";
    private static final String FIELD_GEO_ID = "i";

    private static final String SHOW_GEOID_PREFIX = "~";

    private final boolean showGeoIdEnabled;
    private final YtLuceneIndex slugLuceneIndex;

    // Slug -> GeoId
    private final Counter slug2GeoIdFoundCounter;
    private final Counter slug2GeoIdNotFoundCounter;

    public RegionSlugService(RegionSlugServiceProperties params) {
        if (params.isEnabled()) {
            slugLuceneIndex = new YsonYtLuceneIndex(params, "RegionSlugs", INDEX_STRUCTURE_VERSION, (row) -> {
                Document document = new Document();

                String mainSlug = row.getString("main_slug");
                document.add(new StringField(FIELD_SLUG, mainSlug, Field.Store.YES));
                for (YTreeNode listNode : row.getList("other_slugs")) {
                    String slug = listNode.stringValue();
                    if (!slug.equals(mainSlug)) {// Was explicitly added before
                        document.add(new StringField(FIELD_SLUG, slug, Field.Store.NO));
                    }
                }
                int geoId = row.getInt("geo_id");
                document.add(new IntPoint(FIELD_GEO_ID, geoId));
                document.add(new StoredField(FIELD_GEO_ID, geoId));
                return Collections.singletonList(document);
            });
        } else {
            log.warn("Region slug index disabled");
            slugLuceneIndex = null;
        }
        showGeoIdEnabled = params.isShowGeoIdEnabled();
        slug2GeoIdFoundCounter = Counter.builder("regions.slug.slug2GeoId.found").register(Metrics.globalRegistry);
        slug2GeoIdNotFoundCounter =
                Counter.builder("regions.slug.slug2GeoId.notFound").register(Metrics.globalRegistry);
    }

    @PostConstruct
    public void init() {
        if (slugLuceneIndex != null) {
            slugLuceneIndex.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (slugLuceneIndex != null) {
            slugLuceneIndex.stop();
        }
    }

    public boolean isReady() {
        return slugLuceneIndex == null || slugLuceneIndex.isReady();
    }

    public int getGeoIdBySlug(String slug) {
        TravelPreconditions.checkRequestArgument(slug != null, "Region slug cannot be null");
        if (slug.startsWith(SHOW_GEOID_PREFIX)) {
            slug = slug.substring(SHOW_GEOID_PREFIX.length());
        }
        if (StringUtils.isNumericArabic(slug)) {
            return Integer.valueOf(slug);
        }
        return findGeoIdBySlug(slug);
    }

    private int findGeoIdBySlug(String slug) {
        OptionalInt result;
        if (slugLuceneIndex != null) {
            result = slugLuceneIndex.search(searcher -> {
                Query query = new TermQuery(new Term(FIELD_SLUG, slug));
                TopDocs topDocs = searcher.search(query, 1);
                if (topDocs.totalHits > 0) {
                    Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                    return OptionalInt.of(document.getField(FIELD_GEO_ID).numericValue().intValue());
                } else {
                    return OptionalInt.empty();
                }
            });
        } else {
            result = OptionalInt.empty();
        }
        if (result.isPresent()) {
            slug2GeoIdFoundCounter.increment();
            return result.getAsInt();
        } else {
            slug2GeoIdNotFoundCounter.increment();
            // TODO: Use correct exception.
            throw new HotelNotFoundException(String.format("Region not found by slug '%s'", slug));
        }
    }

    public String getSlug(String initialSlug, int geoId) {
        if (showGeoIdEnabled && initialSlug.startsWith(SHOW_GEOID_PREFIX)) {
            return SHOW_GEOID_PREFIX + geoId;
        }
        return getSlugByGeoId(geoId);
    }

    public String getSlugByGeoId(int geoId) {
        Optional<String> result;
        if (slugLuceneIndex != null) {
            result = slugLuceneIndex.search(searcher -> {
                Query query = IntPoint.newExactQuery(FIELD_GEO_ID, geoId);
                TopDocs topDocs = searcher.search(query, 1);
                if (topDocs.totalHits > 0) {
                    Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                    return Optional.of(document.getField(FIELD_SLUG).stringValue());
                } else {
                    return Optional.empty();
                }
            });
        } else {
            result = Optional.empty();
        }
        return result.orElse(null);
    }
}
