package ru.yandex.travel.api.services.hotels.static_pages;

import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.OptionalInt;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.hotels_portal.Experiments;
import ru.yandex.travel.api.endpoints.hotels_portal.HotelsFilteringService;
import ru.yandex.travel.api.endpoints.hotels_portal.HotelsPortalProperties;
import ru.yandex.travel.api.endpoints.hotels_portal.HotelsPortalUtils;
import ru.yandex.travel.api.models.hotels.Badge;
import ru.yandex.travel.api.models.hotels.DummyRequestAttribution;
import ru.yandex.travel.api.models.hotels.HotelFilterGroup;
import ru.yandex.travel.api.models.hotels.OfferCacheMetadata;
import ru.yandex.travel.api.models.hotels.Price;
import ru.yandex.travel.api.models.hotels.RegionSearchHotelsRequestData;
import ru.yandex.travel.api.services.hotels.amenities.AmenityService;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.hotel_images.HotelImagesService;
import ru.yandex.travel.api.services.hotels.hotel_images.ImageWhitelistDataProvider;
import ru.yandex.travel.api.services.hotels.min_prices.MinMaxPricesService;
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.UaasSearchExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.PartnerConfigService;
import ru.yandex.travel.hotels.geosearch.GeoSearchService;
import ru.yandex.travel.hotels.geosearch.model.GeoOriginEnum;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchReq;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchRsp;
import ru.yandex.travel.hotels.geosearch.model.OfferCacheRequestParams;
import ru.yandex.travel.hotels.offercache.api.ERespMode;
import ru.yandex.travel.hotels.offercache.api.TReadReq;
import ru.yandex.travel.hotels.proto.TPartner;
import ru.yandex.travel.hotels.proto.region_pages.EGeoSearchSortType;

@Component
@AllArgsConstructor
@Slf4j
public class RegionHotelsSearcher implements IRegionHotelSearcher {
    @Data
    @AllArgsConstructor
    private static class SearchRequest {
        private String logId;
        private GeoSearchReq geoSearchReq;
        private Experiments experiments;
        private UaasSearchExperiments uaasSearchExperiments;
    }

    private final GeoBase geoBase;
    private final HotelsPortalProperties hotelsPortalProperties;
    private final PartnerConfigService partnerConfigService;
    private final HotelsFilteringService hotelsFilteringService;
    private final GeoSearchService geoSearchService;
    private final AmenityService amenityService;
    private final HotelSlugService hotelSlugService;
    private final MinMaxPricesService minPricesService;
    private final ExperimentDataProvider uaasExperimentDataProvider;
    private final HotelImagesService hotelImagesService;
    private final ImageWhitelistDataProvider imageWhitelistDataProvider;

    public CompletableFuture<SearchResultWithMinPrices> searchHotelsWithMinPrices(int geoId,
                                                                                  RegionSearchHotelsRequestData requestData,
                                                                                  CommonHttpHeaders headers,
                                                                                  UserCredentials userCredentials,
                                                                                  String logPrefix) {
        return searchHotels(geoId, requestData, headers, userCredentials, logPrefix).thenApply(searchResult ->
                new SearchResultWithMinPrices(
                        searchResult.getHotels().stream()
                                .map(this::tryEnrichWithMinPrice)
                                .filter(Objects::nonNull)
                                .collect(Collectors.toUnmodifiableList()),
                        searchResult.getTotalHotelsFound()
                ));
    }

    public CompletableFuture<SearchResult> searchHotels(int geoId, RegionSearchHotelsRequestData requestData,
                                                        CommonHttpHeaders headers, UserCredentials userCredentials,
                                                        String logPrefix) {

        var searchReq = getSearchRequest(geoId, requestData, headers, userCredentials, logPrefix);

        log.info("{}: Do GeoSearch request", searchReq.logId);
        return geoSearchService.query(searchReq.geoSearchReq)
                .thenApply(geoRsp -> {
                    OfferCacheMetadata ocMeta = OfferCacheMetadata.extract(geoRsp.getOfferCacheResponseMetadata(),
                            requestData, searchReq.geoSearchReq.isUseProdOfferCache());
                    return new SearchResult(geoRsp.getResponseMetadata().getFound(), ocMeta,
                            extractHotels(geoRsp, ocMeta, searchReq.logId, searchReq.geoSearchReq.getFilterCategories(),
                                    hotelsPortalProperties.getHotelCategoriesToShow(), searchReq.experiments,
                                    searchReq.uaasSearchExperiments));
                })
                .exceptionally(t -> {
                    log.error("{}: Unable to complete GeoSearch request, returning empty list", searchReq.logId, t);
                    return new SearchResult(0, OfferCacheMetadata.createFake(requestData), Collections.emptyList());
                });
    }

    public SearchRequest getSearchRequest(int geoId, RegionSearchHotelsRequestData requestData,
                                          CommonHttpHeaders headers, UserCredentials userCredentials,
                                          String logPrefix) {
        var logId = String.format("%s::HotelsBlock(%s)", logPrefix, UUID.randomUUID().toString());

        var origin = HotelsPortalUtils.selectGeoOriginByDevice(headers, GeoOriginEnum.CITY_PAGE_DESKTOP,
                GeoOriginEnum.CITY_PAGE_TOUCH);
        if (geoId != requestData.getGeoId()) {
            log.warn("Ignoring geoId from geoSearchRequestData ({}), using passed geoId ({})",
                    requestData.getGeoId(), geoId);
        }
        var regionLinguistics = GeoBaseHelpers.getRegionLinguistics(geoBase, geoId, "ru");
        var filterAtoms = requestData.getFilters();
        var bbox = requestData.getBBoxString();
        if (bbox.isEmpty()) {
            bbox = null;
        }

        String where = regionLinguistics.getPrepositionalCase();
        if (Strings.isNullOrEmpty(where)) {
            where = regionLinguistics.getNominativeCase();
        }
        var effectiveQuery = String.format("%s %s", regionLinguistics.getPreposition(), where);
        var txt = hotelsPortalProperties.getSearchPrefix() + effectiveQuery;

        var geoReq = GeoSearchReq.byText(origin, txt, headers);
        HotelsPortalUtils.setAdditionalGeoSearchParams(geoReq, hotelsPortalProperties);
        geoReq.setFilterCategories(hotelsPortalProperties.getHotelCategoriesToSearch());
        geoReq.setLr(geoId);
        geoReq.setSupressTextCorrection(true);

        var sortType = requestData.getSortType();
        if (sortType == EGeoSearchSortType.CHEAP_FIRST) {
            geoReq.setSortType(GeoSearchReq.SortType.CHEAP_FIRST);
        } else if (sortType == EGeoSearchSortType.EXPENSIVE_FIRST) {
            geoReq.setSortType(GeoSearchReq.SortType.EXPENSIVE_FIRST);
        }

        if (bbox != null) {
            // Чтобы bbox был конкретный даже без контекста
            geoReq.setRspn(true);
            geoReq.setBoundingBox(bbox);
        }
        geoReq.setFilterHotelProviders(partnerConfigService.getAll().values().stream().map(TPartner::getCode).collect(Collectors.toList()));
        geoReq.setOffset(0);
        geoReq.setLimit(requestData.getLimit());

        List<HotelFilterGroup> filtersGroups = hotelsFilteringService.prepareFilters(filterAtoms, Instant.now(), null, geoId, headers);
        HotelsPortalUtils.addGeoSearchFilters(geoReq, filtersGroups);

        var attribution = new DummyRequestAttribution();
        var experiments = new Experiments(attribution, hotelsPortalProperties);
        var uaasSearchExperiments = uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers);
        var userRegion = HotelsPortalUtils.determineUserRegion(geoBase, headers, attribution);
        TReadReq.Builder ocReqBuilder = HotelsPortalUtils.prepareOfferCacheRequestParams(
                attribution, headers, userCredentials, userRegion, requestData, null, null,
                null ,true, experiments, uaasSearchExperiments,
                "RegionHotelsSearcher/" + logId,
                true,
                hotelsPortalProperties.isSortOffersUsingPlus());
        ocReqBuilder.setRespMode(ERespMode.RM_MultiHotel);
        ocReqBuilder.setCompactResponseForSearch(true);
        geoReq.setOfferCacheRequestParams(OfferCacheRequestParams.build(ocReqBuilder));
        geoReq.setIncludeOfferCache(true);
        geoReq.setIncludeSpravPhotos(true);
        geoReq.setIncludePhotos(true);
        geoReq.setIncludeRating(true);
        geoReq.setIncludeNearByStops(true);
        geoReq.setRequestLogId(logId);

        return new SearchRequest(logId, geoReq, experiments, uaasSearchExperiments);
    }

    private List<SearchResultHotel> extractHotels(GeoSearchRsp geoRsp, OfferCacheMetadata ocMeta, String logId,
                                                          List<String> categoryIdsFromFilters,
                                                          List<String> hotelCategoryIds, Experiments experiments,
                                                          UaasSearchExperiments uaasSearchExperiments) {
        String reqId = geoRsp.getResponseInfo() != null ? geoRsp.getResponseInfo().getReqid() : "---";
        log.info("{}: GeoSearch request finished in {} + {} ms, ReqId: {}", logId, geoRsp.getResponseTime().toMillis(),
                geoRsp.getParseTime().toMillis(), reqId);

        validateGeoSearchResponse(geoRsp, logId);
        var imageParamsProvider = new HotelsPortalUtils.FixedImageParamsProvider();
        return geoRsp.getHotels()
                .stream()
                .filter(hotel -> hotel.getPermalink() != null)
                .filter(hotel -> {
                    if (hotel.getGeoObjectMetadata() == null) {
                        log.error("{}: Failed to find GeoObjectMetadata for hotel {} (Permalink: {})", logId,
                                hotel.getGeoObject().getName(), hotel.getPermalink());
                        return false;
                    }
                    return true;
                })
                .map(hotel -> {
                    var hotelInfo = HotelsPortalUtils.extractHotelWithOffers(
                            amenityService, hotelSlugService, hotelImagesService, ocMeta, hotel,
                            imageParamsProvider, true, hotelsPortalProperties.getHotelBadgePriorities(), hotelsPortalProperties.getOfferBadgePriorities(),
                            categoryIdsFromFilters, experiments, hotelCategoryIds, false, false, false,
                            uaasSearchExperiments, imageWhitelistDataProvider,
                            hotelsPortalProperties.isShowNearestStationOnSnippet());
                    List<Badge> badges = hotel.getOfferCacheResponse() != null
                            && hotel.getOfferCacheResponse().getMirPromoAvailable()
                            ? List.of(Badge.builder().id("mir_cashback").text("Возврат 20%").build())
                            : Collections.emptyList();
                    return new SearchResultHotel(hotelInfo, badges, hotel.getOfferCacheResponse().getIsPlusAvailable());
                })
                .collect(Collectors.toUnmodifiableList());
    }

    private void validateGeoSearchResponse(GeoSearchRsp geoRsp, String logId) {
        if (geoRsp.isNotImplementedError()) {
            log.info("{}: GeoSearch returned 'NotImplemented'", logId);
            throw new RuntimeException("Unexpected Not implemented error in geosearch");
        }
        if (geoRsp.getResponseMetadata() == null) {
            log.error("{}: failed, No ResponseMetadata in GeoSearch response", logId);
            throw new RuntimeException("No ResponseMetadata in GeoSearch response");
        }
        if (geoRsp.getResponseInfo() == null) {
            log.error("{}: failed, No ResponseInfo in GeoSearch response", logId);
            throw new RuntimeException("No ResponseInfo in GeoSearch response");
        }
    }

    private HotelWithMinPrice tryEnrichWithMinPrice(SearchResultHotel resultHotel) {
        var hotel = resultHotel.getHotelWithOffers().getHotel();
        final OptionalInt priceByPermalink = minPricesService.getMinPriceByPermalink(hotel.getPermalink().asLong());
        if (priceByPermalink.isEmpty()) {
            return null;
        } else {
            final Price price = new Price();
            price.setCurrency(Price.Currency.RUB);
            price.setValue(priceByPermalink.getAsInt());
            return new HotelWithMinPrice(hotel, price, resultHotel.getBadges(), resultHotel.getIsPlusAvailable());
        }
    }
}
