package ru.yandex.travel.api.services.hotels.static_pages;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.OptionalInt;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.hotels_portal.Experiments;
import ru.yandex.travel.api.endpoints.hotels_portal.HotelSearchUtils;
import ru.yandex.travel.api.endpoints.hotels_portal.HotelsFilteringService;
import ru.yandex.travel.api.endpoints.hotels_portal.HotelsPortalProperties;
import ru.yandex.travel.api.endpoints.hotels_portal.HotelsPortalUtils;
import ru.yandex.travel.api.models.hotels.Badge;
import ru.yandex.travel.api.models.hotels.BoundingBox;
import ru.yandex.travel.api.models.hotels.DummyRequestAttribution;
import ru.yandex.travel.api.models.hotels.Hotel;
import ru.yandex.travel.api.models.hotels.Price;
import ru.yandex.travel.api.models.hotels.RegionSearchHotelsRequestData;
import ru.yandex.travel.api.services.hotels.amenities.AmenityService;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geocounter.GeoCounterService;
import ru.yandex.travel.api.services.hotels.geocounter.model.GeoCounterGetHotelsReq;
import ru.yandex.travel.api.services.hotels.geocounter.model.GeoCounterGetHotelsRsp;
import ru.yandex.travel.api.services.hotels.hotel_images.HotelImagesService;
import ru.yandex.travel.api.services.hotels.hotel_images.ImageWhitelistDataProvider;
import ru.yandex.travel.api.services.hotels.min_prices.MinMaxPricesService;
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.UaasSearchExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.geosearch.GeoSearchService;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.geosearch.model.GeoOriginEnum;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchReq;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchRsp;
import ru.yandex.travel.hotels.proto.geocounter_service.ESortType;
import ru.yandex.travel.hotels.proto.region_pages.EGeoSearchSortType;

@Component
@AllArgsConstructor
@Slf4j
public class RegionHotelsViaGeocounterSearcher implements IRegionHotelSearcher {
    @Data
    @AllArgsConstructor
    private static class SearchExecutionContext {
        String reqId;
        String logId;
        Experiments experiments;
        UaasSearchExperiments uaasSearchExperiments;
    }

    @Data
    @NoArgsConstructor
    private static class SearchHotel {
        private Hotel hotel;
        private List<Badge> badges;
        private Boolean isPlusAvailable = false;
    }

    private static final Map<EGeoSearchSortType, ESortType> GEO_SEARCH_TO_COUNTER_SORT_TYPE_MAP = Map.of(
            EGeoSearchSortType.RELEVANT_FIRST, ESortType.ST_ByRank,
            EGeoSearchSortType.CHEAP_FIRST, ESortType.ST_ByPriceAsc,
            EGeoSearchSortType.EXPENSIVE_FIRST, ESortType.ST_ByPriceDesc
    );

    private final GeoBase geoBase;
    private final GeoCounterService geoCounter;
    private final HotelsPortalProperties hotelsPortalProperties;
    private final HotelsFilteringService hotelsFilteringService;
    private final GeoSearchService geoSearchService;
    private final AmenityService amenityService;
    private final HotelSlugService hotelSlugService;
    private final MinMaxPricesService minPricesService;
    private final ExperimentDataProvider uaasExperimentDataProvider;
    private final HotelImagesService hotelImagesService;
    private final ImageWhitelistDataProvider imageWhitelistDataProvider;
    private final RegionHotelsSearcher regionHotelsSearcher;

    public CompletableFuture<SearchResult> searchHotels(int geoId, RegionSearchHotelsRequestData requestData,
                                                        CommonHttpHeaders headers, UserCredentials userCredentials,
                                                        String logPrefix) {
        // TODO: can be implemented with geocounter search (use searchRawHotels)
        return regionHotelsSearcher.searchHotels(geoId, requestData, headers, userCredentials, logPrefix);
    }

    public CompletableFuture<SearchResultWithMinPrices> searchHotelsWithMinPrices(int geoId,
                                                                                  RegionSearchHotelsRequestData requestData,
                                                                                  CommonHttpHeaders headers,
                                                                                  UserCredentials userCredentials,
                                                                                  String logPrefix) {
        var execContext = createExecutionContext(headers, logPrefix);
        return searchRawHotels(geoId, requestData, headers, userCredentials, execContext).thenApply(searchResult ->
                new SearchResultWithMinPrices(
                        searchResult.stream()
                                .map(this::tryEnrichWithMinPrice)
                                .filter(Objects::nonNull)
                                .collect(Collectors.toUnmodifiableList()),
                        searchResult.size()
                ));
    }

    private SearchExecutionContext createExecutionContext(CommonHttpHeaders headers, String logPrefix) {
        var reqId = UUID.randomUUID().toString().replace("-", "");
        var attribution = new DummyRequestAttribution();
        return new SearchExecutionContext(
                reqId,
                String.format("%s::HotelsBlock(%s)", logPrefix, reqId),
                new Experiments(attribution, hotelsPortalProperties),
                uaasExperimentDataProvider.getInstance(UaasSearchExperiments.class, headers)
        );
    }

    public CompletableFuture<List<SearchHotel>> searchRawHotels(int geoId,
                                                                RegionSearchHotelsRequestData requestData,
                                                                CommonHttpHeaders headers,
                                                                UserCredentials userCredentials,
                                                                SearchExecutionContext execContext) {

        var getHotelsFuture = doGeoCounterGetHotelsRequest(
                geoId, requestData, headers, userCredentials, execContext
        );

        var geoSearchFuture = getHotelsFuture.thenCompose(hotelsRsp -> {
            var permalinks = hotelsRsp.getHotels().stream()
                    .map(x -> Permalink.of(x.getPermalink()))
                    .collect(Collectors.toUnmodifiableList());
            var origin = HotelsPortalUtils.selectGeoOriginByDevice(headers, GeoOriginEnum.CITY_PAGE_DESKTOP,
                    GeoOriginEnum.CITY_PAGE_TOUCH);
            var geoReq = GeoSearchReq.byPermalinks(origin, permalinks, headers);
            HotelsPortalUtils.setAdditionalGeoSearchParams(geoReq, hotelsPortalProperties);
            geoReq.setLimit(permalinks.size());
            geoReq.setIncludeOfferCache(false);
            geoReq.setIncludeSpravPhotos(true);
            geoReq.setIncludePhotos(true);
            geoReq.setIncludeRating(true);
            geoReq.setIncludeNearByStops(true);
            geoReq.setRequestLogId(execContext.logId);

            return geoSearchService.query(geoReq).thenApply(geoRsp -> {
                log.info("{}: GeoSearch request finished in {} + {} ms", execContext.logId,
                        geoRsp.getResponseTime().toMillis(),
                        geoRsp.getParseTime().toMillis());
                return geoRsp;
            });
        });

        return CompletableFuture.allOf(getHotelsFuture, geoSearchFuture).handle((ignored, ignoredT) -> {
                    GeoCounterGetHotelsRsp hotelsRsp;
                    try {
                        hotelsRsp = getHotelsFuture.join();
                    } catch (Exception exc) {
                        log.error("{}: Unable to complete geoCounter hotels request. Returning empty list.", execContext.logId, exc);
                        return CompletableFuture.completedFuture(Collections.<SearchHotel>emptyList());
                    }

                    GeoSearchRsp geoSearchRsp;
                    try {
                        geoSearchRsp = geoSearchFuture.join();
                    } catch (Exception exc) {
                        log.error("{}: Unable to complete geoSearch request", execContext.logId, exc);
                        geoSearchRsp = null;
                    }

                    return CompletableFuture.completedFuture(extractHotels(hotelsRsp, geoSearchRsp, execContext));
                }).thenCompose(x -> x);
    }

    private CompletableFuture<GeoCounterGetHotelsRsp> doGeoCounterGetHotelsRequest(int geoId,
                                                                                   RegionSearchHotelsRequestData requestData,
                                                                                   CommonHttpHeaders headers,
                                                                                   UserCredentials userCredentials,
                                                                                   SearchExecutionContext execContext) {
        log.info("{}: Doing getHotels request to geoCounter", execContext.logId);

        BoundingBox bbox = null;
        if (requestData.getBBoxString() != null && !requestData.getBBoxString().isEmpty()) {
            bbox = BoundingBox.of(requestData.getBBoxString());
        }
        var origin = HotelsPortalUtils.selectGeoOriginByDevice(headers, GeoOriginEnum.CITY_PAGE_DESKTOP,
                GeoOriginEnum.CITY_PAGE_TOUCH);
        var attribution = new DummyRequestAttribution();
        var getHotelsReqProto = HotelSearchUtils.prepareGeoCounterGetHotelsReq(
                geoId, requestData, requestData, bbox, Instant.now(), null,
                GEO_SEARCH_TO_COUNTER_SORT_TYPE_MAP.get(requestData.getSortType()), null,
                null, false, 0, 0, requestData.getLimit(),
                execContext.reqId, false, origin, null,
                execContext.uaasSearchExperiments, attribution, headers, geoBase,
                userCredentials.isLoggedIn(), hotelsPortalProperties.isSortOffersUsingPlus(),
                hotelsFilteringService, null, true
        );

        return geoCounter.getHotels(
                new GeoCounterGetHotelsReq(getHotelsReqProto, execContext.logId, false)
        ).whenComplete((gcRsp, t) -> {
            if (t == null) {
                log.info("{}: Processed getHotels response from geoCounter in {} ms", execContext.logId,
                        gcRsp.getResponseTime().toMillis());
            } else {
                log.info("{}: Finished failed getHotels request to geoCounter in {} ms", execContext.logId,
                        gcRsp.getResponseTime().toMillis());
            }
            log.info("{}: GeoCounter reqId: {}, sessionId: {}", execContext.logId, gcRsp.getReqId(), gcRsp.getSessionId());
        });
    }

    private List<SearchHotel> extractHotels(GeoCounterGetHotelsRsp getHotelsResult, GeoSearchRsp geoSearchRsp,
                                            SearchExecutionContext execContext) {
        List<SearchHotel> rspAllHotels = new ArrayList<>();

        var geoHotels = geoSearchRsp == null
                ? Map.<Permalink, GeoHotel>of()
                : geoSearchRsp.getHotels().stream().collect(Collectors.toUnmodifiableMap(x -> x.getPermalink(), x -> x));

        var imageParamsProvider = new HotelsPortalUtils.FixedImageParamsProvider();
        for (var gcHotel : getHotelsResult.getHotels()) {
            var geoHotel = geoHotels.get(Permalink.of(gcHotel.getPermalink()));

            var hotel = HotelSearchUtils.extractHotel(
                    gcHotel, geoHotel, null, imageParamsProvider, imageWhitelistDataProvider,
                    hotelSlugService, amenityService, hotelImagesService, hotelsPortalProperties,
                    execContext.experiments, execContext.uaasSearchExperiments);

            var rspHotel = new SearchHotel();
            rspHotel.setHotel(hotel);
            rspHotel.setIsPlusAvailable(gcHotel.getIsPlusAvailable());
            rspHotel.setBadges(HotelsPortalUtils.toBadges(gcHotel.getBadgesList(), hotelsPortalProperties));
            rspAllHotels.add(rspHotel);
        }

        return rspAllHotels;
    }

    private HotelWithMinPrice tryEnrichWithMinPrice(SearchHotel searchHotel) {
        final OptionalInt priceByPermalink = minPricesService.getMinPriceByPermalink(
                searchHotel.getHotel().getPermalink().asLong()
        );
        if (priceByPermalink.isEmpty()) {
            return null;
        } else {
            final Price price = new Price();
            price.setCurrency(Price.Currency.RUB);
            price.setValue(priceByPermalink.getAsInt());
            return new HotelWithMinPrice(searchHotel.getHotel(), price, searchHotel.getBadges(), searchHotel.getIsPlusAvailable());
        }
    }
}
