package ru.yandex.travel.api.services.hotels.static_pages;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import ma.glasnost.orika.CustomConverter;
import ma.glasnost.orika.MapperFacade;
import ma.glasnost.orika.MappingContext;
import ma.glasnost.orika.MappingContextFactory;
import ma.glasnost.orika.converter.ConverterFactory;
import ma.glasnost.orika.impl.DefaultMapperFactory;
import ma.glasnost.orika.metadata.Type;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.hotels_portal.ExtraVisitAndUserParamsUtils;
import ru.yandex.travel.api.endpoints.hotels_portal.HotelsFilteringService;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCityStaticPageRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.SearchHotelsRspV1;
import ru.yandex.travel.api.models.Region;
import ru.yandex.travel.api.models.crosslinks.CrosslinksHotelsBlockData;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.api.models.hotels.HotelWithMinPrice;
import ru.yandex.travel.api.models.hotels.Price;
import ru.yandex.travel.api.models.hotels.SortId;
import ru.yandex.travel.api.models.hotels.breadcrumbs.Breadcrumbs;
import ru.yandex.travel.api.models.hotels.seo.GeoRegionSchemaOrgInfo;
import ru.yandex.travel.api.models.hotels.seo.OpenGraphInfo;
import ru.yandex.travel.api.models.hotels.seo.SchemaOrgInfo;
import ru.yandex.travel.api.models.hotels.seo.SeoInfo;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.TPrice;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.proto.region_pages.EGeoSearchSortType;
import ru.yandex.travel.hotels.proto.region_pages.EPlainTextBlockStyle;
import ru.yandex.travel.hotels.proto.region_pages.TBlock;
import ru.yandex.travel.hotels.proto.region_pages.TExternalLinkBlock;
import ru.yandex.travel.hotels.proto.region_pages.TFaqSchemaMarkupItem;
import ru.yandex.travel.hotels.proto.region_pages.TGeoLinkGroupBlock;
import ru.yandex.travel.hotels.proto.region_pages.TGeoRegionSchemaOrgInfo;
import ru.yandex.travel.hotels.proto.region_pages.TGeoSearchRequestData;
import ru.yandex.travel.hotels.proto.region_pages.THeadingBlock;
import ru.yandex.travel.hotels.proto.region_pages.THotelExtraItems;
import ru.yandex.travel.hotels.proto.region_pages.THotelLinkBlock;
import ru.yandex.travel.hotels.proto.region_pages.THotelListBlock;
import ru.yandex.travel.hotels.proto.region_pages.TOpenGraphInfo;
import ru.yandex.travel.hotels.proto.region_pages.TParagraph;
import ru.yandex.travel.hotels.proto.region_pages.TParagraphBlock;
import ru.yandex.travel.hotels.proto.region_pages.TPlainTextBlock;
import ru.yandex.travel.hotels.proto.region_pages.TPriceTextBlock;
import ru.yandex.travel.hotels.proto.region_pages.TRegionLinkBlock;
import ru.yandex.travel.hotels.proto.region_pages.TRegionLinkSetBlock;
import ru.yandex.travel.hotels.proto.region_pages.TRegionLinkSubSetBlock;
import ru.yandex.travel.hotels.proto.region_pages.TRegionPage;
import ru.yandex.travel.hotels.proto.region_pages.TSchemaOrgInfo;
import ru.yandex.travel.hotels.proto.region_pages.TSearchHotelsLinkBlock;
import ru.yandex.travel.hotels.proto.region_pages.TSectionTextBlock;
import ru.yandex.travel.hotels.proto.region_pages.TSectionTextBlockContent;
import ru.yandex.travel.hotels.proto.region_pages.TSeoBlock;
import ru.yandex.travel.hotels.proto.region_pages.TSpoilerTextBlock;
import ru.yandex.travel.hotels.proto.region_pages.TSubSectionTextBlock;

@Component
@Slf4j
public class RegionPagesMapper {

    @AllArgsConstructor
    private static class MapperTools {
        private final MapperFacade mapperFacade;
        private final MappingContextFactory mappingContextFactory;
    }

    interface Properties {
        String GEO_ID = "GEO_ID";
        String SLUG = "SLUG";
        String HOTEL_SEARCH_REQUEST_MAP = "rda";
        String IMAGE_URL = "IMAGE_URL";
        String GEO_REGION = "GEO_REGION";
        String BREADCRUMBS_BLOCK = "BREADCRUMBS_BLOCK";
        String CROSS_SALE_BLOCK = "CROSS_SALE_BLOCK";
        String FILTERS_BLOCK = "FILTERS_BLOCK";
    }

    interface Converters {
        String CONTENT_TO_BLOCK = "content2block";
        String SECTION_TEXT_BLOCK_CHILDREN = "sectionTextBlockChildrenConverter";
        String PARAGRAPH_CHILDREN = "paragraphChildrenConverter";
        String SCHEMA_ORG = "schemaOrg";
        String PRICE = "price";
        String PLAIN_TEXT_STYLE = "plainTextStyle";
        String GEO_SEARCH_SORT_TYPE = "geoSearchSortType";
    }

    private final static Map<Integer, String> regionPrefix = Map.of(
        GeoBaseHelpers.CITY_REGION_TYPE, "г.",
        GeoBaseHelpers.VILLAGE_REGION_TYPE, "с."
    );
    private final HotelsFilteringService hotelsFilteringService;
    private final MapperFacade mapperFacade;
    private final MappingContextFactory mappingContextFactory;
    private final RegionsService regionsService;

    public RegionPagesMapper(RegionsService regionsService, HotelsFilteringService hotelsFilteringService) {
        this.hotelsFilteringService = hotelsFilteringService;
        this.regionsService = regionsService;
        final MapperTools mapperTools = initMapper();
        this.mapperFacade = mapperTools.mapperFacade;
        this.mappingContextFactory = mapperTools.mappingContextFactory;
    }

    public GetCityStaticPageRspV1 map(
            TRegionPage page,
            List<THotelListBlock> hotelListBlocks,
            List<CompletableFuture<HotelBlockBuildResult>> hotelListFutures,
            CompletableFuture<CrosslinksHotelsBlockData> crosslinksHotelsFuture,
            int geoId,
            String finalSlug,
            String imageUrl,
            Breadcrumbs breadcrumbs,
            Region region,
            String domain,
            CommonHttpHeaders commonHttpHeaders
    ) {
        try {
            final Map<TGeoSearchRequestData, HotelBlockBuildResult> requestsMap = new HashMap<>();
            Set<Permalink> usedPermalinks = new HashSet<>();
            boolean hotelsExists = false;
            // Every subsequent hotel list should not contain hotels from previous one
            for (int idx = 0; idx < hotelListBlocks.size(); ++idx) {
                TGeoSearchRequestData request = hotelListBlocks.get(idx).getGeoSearchRequestData();
                CompletableFuture<HotelBlockBuildResult> future = hotelListFutures.get(idx);
                if (future.isCompletedExceptionally()) {
                    log.warn("One of GeoSearch requests has failed: key is {}", request);
                    continue;
                }
                HotelBlockBuildResult origResult = future.get();
                List<HotelWithMinPrice> filteredHotels = new ArrayList<>();
                for (HotelWithMinPrice hotel : origResult.getHotelsWithMeanPrice()) {
                    if (filteredHotels.size() >= request.getLimit()) {
                        break;
                    }
                    if (usedPermalinks.add(hotel.getHotel().getPermalink())) {
                        updateHotel(hotel, origResult.getHotelListBlock(), domain);
                        filteredHotels.add(hotel);
                        hotelsExists = true;
                    }
                }
                requestsMap.put(
                        request,
                        new HotelBlockBuildResult(
                            origResult.getHotelListBlock(),
                            filteredHotels
                        )
                );
            }

            CrosslinksHotelsBlockData crosslinksBlockData = crosslinksHotelsFuture.get();

            MappingContext context = mappingContextFactory.getContext();
            context.setProperty(Properties.HOTEL_SEARCH_REQUEST_MAP, requestsMap);
            context.setProperty(Properties.GEO_ID, geoId);
            context.setProperty(Properties.SLUG, finalSlug);
            context.setProperty(Properties.IMAGE_URL, imageUrl);
            context.setProperty(Properties.GEO_REGION, region);
            context.setProperty(Properties.BREADCRUMBS_BLOCK, buildBreadcrumbsBlock(breadcrumbs));
            context.setProperty(Properties.FILTERS_BLOCK, hotelsExists ? buildFiltersBlock(geoId, commonHttpHeaders) : null);
            context.setProperty(Properties.CROSS_SALE_BLOCK, buildCrossSaleHotelsBlock(crosslinksBlockData));

            try {
                var result = mapperFacade.map(page, GetCityStaticPageRspV1.class, context);
                postProcess(result, page, imageUrl, geoId, context);
                return result;
            } finally {
                mappingContextFactory.release(context);
            }
        } catch (Exception e) {
            throw new RuntimeException("Exception on static page mapping", e);
        }
    }

    private void updateHotel(HotelWithMinPrice hotel, THotelListBlock block, String domain) {
        boolean showCityName = false;
        boolean showLocation = false;
        boolean showNearestStations = false;
        if (block.hasHotelExtraItems()) {
            THotelExtraItems hotelExtraItems = block.getHotelExtraItems();
            showCityName = hotelExtraItems.getCityName();
            showLocation = hotelExtraItems.getLocation();
            showNearestStations = hotelExtraItems.getNearestStations();
        }
        if (showCityName) { // TODO: remove after TRAVELBACK-1637 related tasks finished
            Coordinates coordinates = hotel.getHotel().getCoordinates();
            Integer cityGeoId = regionsService.getRegionRoundTo(
                    coordinates.getLat(),
                    coordinates.getLon(),
                    GeoBaseHelpers.CITY_REGION_TYPE,
                    domain
            );
            if (cityGeoId != null) {
                String cityName = regionsService.getRegionName(cityGeoId, domain);
                hotel.getHotel().setCityName(cityName);
            }
        }
        if (showLocation) {
            Coordinates hotelCoordinates = hotel.getHotel().getCoordinates();
            int locationGeoId = regionsService.getPreferredRegion(
                    hotelCoordinates.getLat(),
                    hotelCoordinates.getLon(),
                    Set.of(GeoBaseHelpers.CITY_REGION_TYPE, GeoBaseHelpers.VILLAGE_REGION_TYPE),
                    domain
            );
            String location = regionsService.getRegionName(locationGeoId, domain);
            int regionType = regionsService.getRegionType(locationGeoId, domain);
            String prefix = regionPrefix.get(regionType);
            if (prefix != null) {
                location = String.format("%s %s", prefix, location);
            }
            hotel.getHotel().setLocation(location);
        }
        if (!showNearestStations) {
            hotel.getHotel().setNearestStations(null);
        }
    }

    private GetCityStaticPageRspV1.HotelsFilterBlock buildFiltersBlock(int geoId, CommonHttpHeaders commonHttpHeaders) {
        SearchHotelsRspV1.FilterInfo filterInfo = hotelsFilteringService.composeDefaultFilterInfo(
                null, Instant.now(), null, geoId, false, commonHttpHeaders);

        var searchParams = new GetCityStaticPageRspV1.SearchPageIdentifyParams();
        searchParams.setGeoId(geoId);

        var hotelsFilterData = new GetCityStaticPageRspV1.HotelsFilterBlockData();
        hotelsFilterData.setFilterInfo(filterInfo);
        hotelsFilterData.setSearchParams(searchParams);

        var hotelsFilterBlock = new GetCityStaticPageRspV1.HotelsFilterBlock();
        hotelsFilterBlock.setData(hotelsFilterData);
        return hotelsFilterBlock;
    }

    private GetCityStaticPageRspV1.CrossSaleHotelsBlock buildCrossSaleHotelsBlock(CrosslinksHotelsBlockData crosslinksHotelsBlockData) {
        var crossSaleHotelsBlock = new GetCityStaticPageRspV1.CrossSaleHotelsBlock();
        crossSaleHotelsBlock.setData(crosslinksHotelsBlockData);
        return crossSaleHotelsBlock;
    }

    private GetCityStaticPageRspV1.BreadcrumbsBlock buildBreadcrumbsBlock(Breadcrumbs breadcrumbs) {
        GetCityStaticPageRspV1.BreadcrumbsData breadcrumbsData = new GetCityStaticPageRspV1.BreadcrumbsData();
        breadcrumbsData.setBreadcrumbs(breadcrumbs);
        GetCityStaticPageRspV1.BreadcrumbsBlock breadcrumbsBlock = new GetCityStaticPageRspV1.BreadcrumbsBlock();
        breadcrumbsBlock.setData(breadcrumbsData);
        return breadcrumbsBlock;
    }

    private void postProcess(GetCityStaticPageRspV1 rsp, TRegionPage page, String imageUrl, int geoId, MappingContext context) {
        var pageParams = ExtraVisitAndUserParamsUtils.initParamsMap();
        ExtraVisitAndUserParamsUtils.fillSeoStaticPageParams(pageParams, geoId, page.getPageType());
        rsp.setExtraVisitAndUserParams(ExtraVisitAndUserParamsUtils.createForSeoRegionPage(pageParams));

        rsp.getSeoInfo().getOpenGraph().setImage(imageUrl);

        rsp.setBlocks(rsp.getBlocks().stream().filter(Objects::nonNull).collect(Collectors.toList()));
    }

    private MapperTools initMapper() {
        final MappingContextFactory mappingContextFactory = new MappingContext.Factory();
        final DefaultMapperFactory mapperFactory = new DefaultMapperFactory.Builder()
                .mappingContextFactory(mappingContextFactory)
                .build();
        final ConverterFactory converterFactory = mapperFactory.getConverterFactory();

        converterFactory.registerConverter(Converters.CONTENT_TO_BLOCK, new CustomConverter<TBlock,
                GetCityStaticPageRspV1.GeoRegionBlock>() {
            @Override
            public GetCityStaticPageRspV1.GeoRegionBlock convert(
                    TBlock source,
                    Type<? extends GetCityStaticPageRspV1.GeoRegionBlock> destinationType,
                    MappingContext mappingContext
            ) {
                if (source.hasHeadingBlock()) {
                    var heading = mapperFacade.map(source.getHeadingBlock(),
                            GetCityStaticPageRspV1.SearchFormBlock.class);
                    heading.getData().getSearchFormParams().setRegionSlug((String) mappingContext.getProperty(Properties.SLUG));
                    heading.getData().setImageUrl((String) mappingContext.getProperty(Properties.IMAGE_URL));
                    return heading;
                } else if (source.hasHotelListBlock()) {
                    final var rda =
                            (Map<TGeoSearchRequestData, HotelBlockBuildResult>) mappingContext.getProperty(Properties.HOTEL_SEARCH_REQUEST_MAP);
                    final int geoId = (int) mappingContext.getProperty(Properties.GEO_ID);
                    final Region region = (Region) mappingContext.getProperty(Properties.GEO_REGION);

                    final TGeoSearchRequestData key = source.getHotelListBlock().getGeoSearchRequestData();
                    final HotelBlockBuildResult hotelBlockBuildResult = rda.get(key);
                    if (hotelBlockBuildResult == null) {
                        return null; // will be filtered at postprocessing
                    }
                    if (hotelBlockBuildResult.getHotelsWithMeanPrice().isEmpty()) {
                        return null; // will be filtered at postprocessing
                    }

                    final GetCityStaticPageRspV1.HotelsBlock res = mapperFacade.map(source.getHotelListBlock(),
                            GetCityStaticPageRspV1.HotelsBlock.class);
                    res.getData().getSearchParams().setGeoId(geoId);
                    res.getData().setRegion(region);
                    res.getData().setHotels(hotelBlockBuildResult.getHotelsWithMeanPrice());

                    return res;
                } else if (source.hasSectionTextBlock()) {
                    var x = mapperFacade.map(source.getSectionTextBlock(),
                            GetCityStaticPageRspV1.SectionTextBlock.class);
                    return x;
                } else if (source.hasRegionLinkSetBlock()) {
                    var x = mapperFacade.map(source.getRegionLinkSetBlock(),
                            GetCityStaticPageRspV1.RegionLinkSetBlock.class);
                    return x;
                } else if (source.hasFiltersBlock()) {
                    return (GetCityStaticPageRspV1.HotelsFilterBlock) mappingContext.getProperty(Properties.FILTERS_BLOCK);
                } else if (source.hasBreadcrumbsBlock()) {
                    return (GetCityStaticPageRspV1.BreadcrumbsBlock) mappingContext.getProperty(Properties.BREADCRUMBS_BLOCK);
                } else if (source.hasMapBlock()) {
                    return (GetCityStaticPageRspV1.CrossSaleHotelsBlock) mappingContext.getProperty(Properties.CROSS_SALE_BLOCK);
                } else {
                    throw new RuntimeException(String.format("Unknown class %s", source.getInstanceCase()));
                }
            }
        });
        converterFactory.registerConverter(Converters.SECTION_TEXT_BLOCK_CHILDREN,
                new CustomConverter<List<TSectionTextBlockContent>,
                        List<GetCityStaticPageRspV1.SectionTextChildBlock>>() {
                    @Override
                    public List<GetCityStaticPageRspV1.SectionTextChildBlock> convert(
                            List<TSectionTextBlockContent> sourcea,
                            Type<? extends List<GetCityStaticPageRspV1.SectionTextChildBlock>> destinationType,
                            MappingContext mappingContext
                    ) {
                        return sourcea.stream().map(source -> {
                            if (source.hasSubSectionTextBlock()) {
                                return mapperFacade.map(source.getSubSectionTextBlock(),
                                        GetCityStaticPageRspV1.SubSectionTextBlock.class);
                            } else if (source.hasSpoilerTextBlock()) {
                                return mapperFacade.map(source.getSpoilerTextBlock(),
                                        GetCityStaticPageRspV1.SpoilerBlock.class);
                            } else if (source.hasParagraph()) {
                                return mapperFacade.map(source.getParagraph(), GetCityStaticPageRspV1.Paragraph.class);
                            } else if (source.hasGeoLinkGroupBlock()) {
                                return mapperFacade.map(source.getGeoLinkGroupBlock(),
                                        GetCityStaticPageRspV1.GeoLinkGroupBlock.class);
                            } else {
                                throw new RuntimeException(String.format("Unknown class %s", source.getInstanceCase()));
                            }
                        }).collect(Collectors.toList());
                    }
                });
        converterFactory.registerConverter(Converters.PARAGRAPH_CHILDREN, new CustomConverter<List<TParagraphBlock>,
                List<GetCityStaticPageRspV1.ParagraphBlock>>() {
            @Override
            public List<GetCityStaticPageRspV1.ParagraphBlock> convert(
                    List<TParagraphBlock> sourcea,
                    Type<? extends List<GetCityStaticPageRspV1.ParagraphBlock>> destinationType,
                    MappingContext mappingContext
            ) {
                return sourcea.stream().map(source -> {
                    if (source.hasExternalLinkBlock()) {
                        return mapperFacade.map(source.getExternalLinkBlock(),
                                GetCityStaticPageRspV1.ExternalLinkBlock.class);
                    } else if (source.hasHotelLinkBlock()) {
                        return mapperFacade.map(source.getHotelLinkBlock(),
                                GetCityStaticPageRspV1.HotelLinkBlock.class);
                    } else if (source.hasRegionLinkBlock()) {
                        return mapperFacade.map(source.getRegionLinkBlock(),
                                GetCityStaticPageRspV1.RegionLinkBlock.class);
                    } else if (source.hasSearchHotelsLinkBlock()) {
                        return mapperFacade.map(source.getSearchHotelsLinkBlock(),
                                GetCityStaticPageRspV1.SearchHotelsLinkBlock.class);
                    } else if (source.hasPriceTextBlock()) {
                        return mapperFacade.map(source.getPriceTextBlock(),
                                GetCityStaticPageRspV1.PriceTextBlock.class);
                    } else if (source.hasPlainTextBlock()) {
                        return mapperFacade.map(source.getPlainTextBlock(),
                                GetCityStaticPageRspV1.PlainTextBlock.class);
                    } else {
                        throw new RuntimeException(String.format("Unknown class %s", source.getInstanceCase()));
                    }
                }).collect(Collectors.toList());
            }
        });
        converterFactory.registerConverter(Converters.SCHEMA_ORG, new CustomConverter<TSchemaOrgInfo, SchemaOrgInfo>() {
            @Override
            public SchemaOrgInfo convert(
                    TSchemaOrgInfo source,
                    Type<? extends SchemaOrgInfo> destinationType,
                    MappingContext mappingContext
            ) {
                if (source.hasGeoRegionSchemaOrgInfo()) {
                    return mapperFacade.map(source.getGeoRegionSchemaOrgInfo(), GeoRegionSchemaOrgInfo.class);
                } else {
                    throw new RuntimeException(String.format("Unknown class %s", source.getInstanceCase()));
                }
            }
        });
        converterFactory.registerConverter(Converters.PRICE, new CustomConverter<TPrice, Price>() {
            private final Map<ECurrency, Price.Currency> currencyMap = Collections.unmodifiableMap(Map.of(
                    ECurrency.C_RUB, Price.Currency.RUB
            ));

            @Override
            public Price convert(TPrice source, Type<? extends Price> destinationType, MappingContext mappingContext) {
                final var result = new Price();
                int precision = source.getPrecision();
                Preconditions.checkArgument(precision >= 0, "Negative precision");
                Preconditions.checkArgument(precision < 30, "Too big precision");
                long value = source.getAmount();
                while (precision-- > 0) {
                    value /= 10;
                }
                result.setValue((int) value);
                final Price.Currency currency = currencyMap.get(source.getCurrency());
                if (Objects.isNull(currency)) {
                    throw new RuntimeException(String.format("Currency %s not found", source.getCurrency()));
                }
                result.setCurrency(currency);
                return result;
            }
        });
        converterFactory.registerConverter(Converters.PLAIN_TEXT_STYLE, new CustomConverter<EPlainTextBlockStyle,
                GetCityStaticPageRspV1.PlainTextBlockData.Style>() {
            @Override
            public GetCityStaticPageRspV1.PlainTextBlockData.Style convert(
                    EPlainTextBlockStyle source,
                    Type<? extends GetCityStaticPageRspV1.PlainTextBlockData.Style> destinationType,
                    MappingContext mappingContext
            ) {
                switch (source) {
                    case BOLD:
                        return GetCityStaticPageRspV1.PlainTextBlockData.Style.BOLD;
                    case ITALIC:
                        return GetCityStaticPageRspV1.PlainTextBlockData.Style.ITALIC;
                    default:
                        throw new RuntimeException(String.format("Unknown style enum %s", source));
                }
            }
        });
        converterFactory.registerConverter(Converters.GEO_SEARCH_SORT_TYPE, new CustomConverter<EGeoSearchSortType,
                SortId>() {
            @Override
            public SortId convert(
                    EGeoSearchSortType source,
                    Type<? extends SortId> destinationType,
                    MappingContext mappingContext
            ) {
                switch (source) {
                    case RELEVANT_FIRST:
                        return SortId.RELEVANT_FIRST;
                    case CHEAP_FIRST:
                        return SortId.CHEAP_FIRST;
                    case EXPENSIVE_FIRST:
                        return SortId.EXPENSIVE_FIRST;
                    default:
                        throw new RuntimeException(String.format("Unknown sort type %s", source));
                }
            }
        });

        mapperFactory.classMap(TRegionPage.class, GetCityStaticPageRspV1.class)
                .fieldAToB("seoBlock", "seoInfo")
                .fieldMap("contentList", "blocks").aToB().converter(Converters.CONTENT_TO_BLOCK).add()
                .register();

        mapperFactory.classMap(TSeoBlock.class, SeoInfo.class)
                .fieldAToB("title", "title")
                .fieldAToB("description", "description")
                .fieldAToB("openGraphInfo", "openGraph")
                .fieldMap("schemaOrgInfo", "schemaOrg").aToB().converter(Converters.SCHEMA_ORG).add()
                .register();

        mapperFactory.classMap(TOpenGraphInfo.class, OpenGraphInfo.class)
                .fieldAToB("title", "title")
                .fieldAToB("description", "description")
                .register();

        mapperFactory.classMap(TGeoRegionSchemaOrgInfo.class, GeoRegionSchemaOrgInfo.class)
                .fieldAToB("faqSchemaMarkupItemList", "faqItems")
                .register();

        mapperFactory.classMap(TFaqSchemaMarkupItem.class, GeoRegionSchemaOrgInfo.IFAQSchemaMarkupItem.class)
                .fieldAToB("question", "question")
                .fieldAToB("answer", "answer")
                .register();

        mapperFactory.classMap(THeadingBlock.class, GetCityStaticPageRspV1.SearchFormBlock.class)
                .fieldAToB("title", "data.title")
                .fieldAToB("isCalendarOpen", "data.searchFormParams.isCalendarOpen")
                .fieldAToB("geoSearchFiltersList", "data.filterParams.filterAtoms")
                .register();

        mapperFactory.classMap(THotelListBlock.class, GetCityStaticPageRspV1.HotelsBlock.class)
                .fieldAToB("title", "data.title")
                .fieldAToB("buttonText", "data.searchButtonText")
                .fieldAToB("geoSearchRequestData.filtersList", "data.searchParams.filterParams.filterAtoms")
                .fieldAToB("geoSearchRequestData.sortType", "data.searchParams.selectedSortId")
                .register();

        mapperFactory.classMap(TSectionTextBlock.class, GetCityStaticPageRspV1.SectionTextBlock.class)
                .fieldAToB("title", "data.title")
                .fieldMap("childrenList", "children").aToB().converter(Converters.SECTION_TEXT_BLOCK_CHILDREN).add()
                .register();

        mapperFactory.classMap(TSubSectionTextBlock.class, GetCityStaticPageRspV1.SubSectionTextBlock.class)
                .fieldAToB("title", "data.title")
                .fieldAToB("paragraphsList", "children")
                .register();

        mapperFactory.classMap(TSpoilerTextBlock.class, GetCityStaticPageRspV1.SpoilerBlock.class)
                .fieldAToB("title", "data.title")
                .fieldAToB("description", "data.description")
                .register();

        mapperFactory.classMap(TParagraph.class, GetCityStaticPageRspV1.Paragraph.class)
                .fieldMap("childrenList", "children").aToB().converter(Converters.PARAGRAPH_CHILDREN).add()
                .register();

        mapperFactory.classMap(TExternalLinkBlock.class, GetCityStaticPageRspV1.ExternalLinkBlock.class)
                .fieldAToB("text", "data.text")
                .fieldAToB("url", "data.url")
                .register();

        mapperFactory.classMap(THotelLinkBlock.class, GetCityStaticPageRspV1.HotelLinkBlock.class)
                .fieldAToB("text", "data.text")
                .fieldAToB("slug", "data.urlParams.hotelSlug")
                .register();

        mapperFactory.classMap(TRegionLinkBlock.class, GetCityStaticPageRspV1.RegionLinkBlock.class)
                .fieldAToB("text", "data.text")
                .fieldAToB("regionSlug", "data.urlParams.regionSlug")
                .fieldAToB("filterSlug", "data.urlParams.filterSlug")
                .register();

        mapperFactory.classMap(TSearchHotelsLinkBlock.class, GetCityStaticPageRspV1.SearchHotelsLinkBlock.class)
                .fieldAToB("text", "data.text")
                .fieldAToB("urlParams.geoId", "data.urlParams.geoId")
                .fieldMap("urlParams.sortType", "data.urlParams.selectedSortId").aToB().converter(Converters.GEO_SEARCH_SORT_TYPE).add()
                .fieldAToB("urlParams.filterAtomsList", "data.urlParams.filterParams.filterAtoms")
                .fieldAToB("urlParams.priceFrom", "data.urlParams.filterParams.filterPriceFrom")
                .fieldAToB("urlParams.priceTo", "data.urlParams.filterParams.filterPriceTo")
                .register();

        mapperFactory.classMap(TPriceTextBlock.class, GetCityStaticPageRspV1.PriceTextBlock.class)
                .fieldMap("price", "data.price").aToB().converter(Converters.PRICE).add()
                .register();

        mapperFactory.classMap(TPlainTextBlock.class, GetCityStaticPageRspV1.PlainTextBlock.class)
                .fieldAToB("text", "data.text")
                .fieldMap("styleList", "data.styles").aToB().converter(Converters.PLAIN_TEXT_STYLE).add()
                .register();

        mapperFactory.classMap(TGeoLinkGroupBlock.class, GetCityStaticPageRspV1.GeoLinkGroupBlock.class)
                .fieldAToB("main", "data.mainLink")
                .fieldAToB("additional", "data.additionalLink")
                .register();

        mapperFactory.classMap(TRegionLinkSetBlock.class, GetCityStaticPageRspV1.RegionLinkSetBlock.class)
                .fieldAToB("title", "data.title")
                .fieldAToB("subsetsList", "data.subsets")
                .register();

        mapperFactory.classMap(TRegionLinkSubSetBlock.class, GetCityStaticPageRspV1.RegionLinkSubSetBlock.class)
                .fieldAToB("title", "data.title")
                .fieldAToB("linksList", "data.links")
                .register();

        return new MapperTools(mapperFactory.getMapperFacade(), mappingContextFactory);
    }
}
