package ru.yandex.travel.api.services.hotels.static_pages;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.hotels_portal.Experiments;
import ru.yandex.travel.api.endpoints.hotels_portal.HotelsPortalProperties;
import ru.yandex.travel.api.endpoints.hotels_portal.breadcrumbs.BreadcrumbsService;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DoesCityStaticPageExistReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.DoesCityStaticPageExistRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCityStaticPageReqV1;
import ru.yandex.travel.api.endpoints.hotels_portal.req_rsp.GetCityStaticPageRspV1;
import ru.yandex.travel.api.infrastucture.TravelPreconditions;
import ru.yandex.travel.api.models.Region;
import ru.yandex.travel.api.models.hotels.HotelWithMinPrice;
import ru.yandex.travel.api.models.hotels.RegionSearchHotelsRequestData;
import ru.yandex.travel.api.models.hotels.breadcrumbs.Breadcrumbs;
import ru.yandex.travel.api.services.crosslinks.CrosslinksService;
import ru.yandex.travel.api.services.hotels.region_images.RegionImagesService;
import ru.yandex.travel.api.services.hotels.regions.RegionsService;
import ru.yandex.travel.api.services.hotels.slug.RegionSlugService;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.proto.region_pages.TBlock;
import ru.yandex.travel.hotels.proto.region_pages.TGeoSearchRequestData;
import ru.yandex.travel.hotels.proto.region_pages.THotelListBlock;
import ru.yandex.travel.hotels.proto.region_pages.TRegionPage;


@Component
@RequiredArgsConstructor
@EnableConfigurationProperties(RegionPagesServiceProperties.class)
@Slf4j
public class RegionPagesService {
    private final static String RANDOM_REGION_SLUG = "~idkfa";
    private final static int SEARCH_LIMIT = 50;
    private final static String IMAGE_SIZE = "region-desktop";

    private final RegionPagesServiceProperties config;
    private final BreadcrumbsService breadcrumbsService;
    private final CrosslinksService crosslinksService;
    private final HotelsPortalProperties hotelsPortalProperties;
    private final RegionPagesStorage regionPagesStorage;
    private final RegionSlugService regionSlugService;
    private final RegionPagesMapper regionPagesMapper;
    private final RegionHotelsSearcher regionHotelsSearcher;
    private final RegionHotelsViaGeocounterSearcher regionHotelsViaGeocounterSearcher;
    private final RegionImagesService regionImagesService;
    private final RegionsService regionsService;

    public CompletableFuture<DoesCityStaticPageExistRspV1> doesCityStaticPageExist(DoesCityStaticPageExistReqV1 req) {
        boolean result = false;
        Optional<TRegionPage> res = regionPagesStorage.tryGetRegionPage(req.getGeoId(), "");
        if (res.isPresent()) {
            result = true;
        }
        return CompletableFuture.completedFuture(new DoesCityStaticPageExistRspV1(result));
    }

    public CompletableFuture<GetCityStaticPageRspV1> getRegionStaticPage(GetCityStaticPageReqV1 req,
                                                                         CommonHttpHeaders headers,
                                                                         UserCredentials userCredentials) {
        TravelPreconditions.checkRequestArgument(hotelsPortalProperties.isRegionStaticPageEnabled(), "Region static page is disabled");
        final int geoId = getGeoId(req.getRegionSlug());
        final String finalSlug = getFinalSlug(req.getRegionSlug(), geoId);
        final TRegionPage page = regionPagesStorage.getRegionPage(geoId, req.getFilterSlug());
        final Breadcrumbs breadcrumbs = breadcrumbsService.getStaticPageBreadcrumbs(
                req.getDomain(), geoId, page.getFilterName(), page.getFilterSlug()
        );

        final var requestsData = page.getContentList().stream()
                .filter(TBlock::hasHotelListBlock)
                .map(TBlock::getHotelListBlock)
                .collect(Collectors.toList());

        final var hotelListFutures = requestsData.stream()
                .map(block -> queryHotelsBlock(geoId, block, headers, userCredentials))
                .collect(Collectors.toList());

        final var crosslinksHotelsFuture = crosslinksService.getHotelsBlockData(
                geoId, req.getDomain(), headers, userCredentials
        );

        List<CompletableFuture<?>> futures = new ArrayList<>(hotelListFutures);
        futures.add(crosslinksHotelsFuture);

        Region region = regionsService.getRegion(geoId, req.getDomain(), "ru");

        return CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new))
                .thenApply(ignored -> regionPagesMapper.map(page, requestsData, hotelListFutures, crosslinksHotelsFuture,
                        geoId, finalSlug, regionImagesService.getImageUsingTreeWithSize(geoId, IMAGE_SIZE),
                        breadcrumbs, region, req.getDomain(), headers));
    }

    private boolean isRandomRegion(String slug) {
        return config.isRandomRegionEnabled() && RANDOM_REGION_SLUG.equalsIgnoreCase(slug);
    }

    private int getGeoId(String slug) {
        if (isRandomRegion(slug)) {
            return regionPagesStorage.getRandomGeoId();
        } else {
            return regionSlugService.getGeoIdBySlug(slug);
        }
    }

    private String getFinalSlug(String initialSlug, int geoId) {
        if (isRandomRegion(initialSlug)) {
            return RANDOM_REGION_SLUG;
        } else {
            return regionSlugService.getSlug(initialSlug, geoId);
        }
    }

    public CompletableFuture<HotelBlockBuildResult> queryHotelsBlock(
            int geoId,
            THotelListBlock block,
            CommonHttpHeaders headers,
            UserCredentials userCredentials) {

        RegionSearchHotelsRequestData blockRequest = composeRegionSearchHotelsRequestData(block.getGeoSearchRequestData());

        var experiments = new Experiments(null, hotelsPortalProperties);
        if (experiments.isExp("region-pages-search-via-geocounter")) {
            return regionHotelsViaGeocounterSearcher.searchHotelsWithMinPrices(geoId, blockRequest,
                    headers, userCredentials, "CityPages").thenApply(result ->
                    new HotelBlockBuildResult(
                            block,
                            result.getHotelsWithMinPrice().stream()
                                    .map(this::mapHotelWithMinPrice)
                                    .collect(Collectors.toUnmodifiableList())
                    )
            );
        } else {
            return regionHotelsSearcher.searchHotelsWithMinPrices(geoId, blockRequest,
                    headers, userCredentials, "CityPages").thenApply(result ->
                    new HotelBlockBuildResult(
                            block,
                            result.getHotelsWithMinPrice().stream()
                                    .map(this::mapHotelWithMinPrice)
                                    .collect(Collectors.toUnmodifiableList())
                    )
            );
        }
    }

    private RegionSearchHotelsRequestData composeRegionSearchHotelsRequestData(TGeoSearchRequestData block) {
        return new RegionSearchHotelsRequestData(
                SEARCH_LIMIT,
                block.getGeoId(),
                block.getBBoxString(),
                block.getFiltersList(),
                block.getSortType(),
                null,
                null,
                null,
                null
        );
    }

    private HotelWithMinPrice mapHotelWithMinPrice(IRegionHotelSearcher.HotelWithMinPrice hotelWithMinPrice) {
        return new HotelWithMinPrice(
                hotelWithMinPrice.getHotel(),
                hotelWithMinPrice.getMinPrice(),
                hotelWithMinPrice.getBadges(),
                hotelWithMinPrice.getHasOfferYandexPlus()
        );
    }
}
