package ru.yandex.travel.api.services.hotels.static_pages;

import java.util.Collections;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.IntPoint;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.hotels.common.HotelNotFoundException;
import ru.yandex.travel.hotels.proto.region_pages.TRegionPage;
import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;


@Component
@EnableConfigurationProperties(RegionPagesStorageProperties.class)
@Slf4j
public class RegionPagesStorage {
    private static final int INDEX_STRUCTURE_VERSION = 4;

    private static final String FIELD_GEO_ID = "i";
    private static final String FIELD_FILTER_SLUG = "f";
    private static final String FIELD_PROTO_DATA = "p";

    private final YtLuceneIndex cityPagesLuceneIndex;

    private final Counter staticCityPageFoundCounter;
    private final Counter staticCityPageNotFoundCounter;

    public RegionPagesStorage(RegionPagesStorageProperties params) {
        if (params.isEnabled()) {
            cityPagesLuceneIndex = new YsonYtLuceneIndex(params, "CityStaticPages", INDEX_STRUCTURE_VERSION, (row) -> {
                Document document = new Document();

                final int geoId = row.getInt("geo_id");
                document.add(new IntPoint(FIELD_GEO_ID, geoId));
                document.add(new StoredField(FIELD_GEO_ID, geoId));

                final String filterSlug = row.getStringO("filter_slug").orElse("");
                document.add(new StringField(FIELD_FILTER_SLUG, filterSlug, Field.Store.YES));
                document.add(new StoredField(FIELD_FILTER_SLUG, filterSlug));

                final byte[] proto = row.getBytes("proto");
                document.add(new StoredField(FIELD_PROTO_DATA, proto));

                return Collections.singletonList(document);
            });
        } else {
            log.warn("City static pages index disabled");
            cityPagesLuceneIndex = null;
        }
        staticCityPageFoundCounter = Counter.builder("static_pages.city.found").register(Metrics.globalRegistry);
        staticCityPageNotFoundCounter = Counter.builder("static_pages.city.notFound").register(Metrics.globalRegistry);
    }

    @PostConstruct
    public void init() {
        if (cityPagesLuceneIndex != null) {
            cityPagesLuceneIndex.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (cityPagesLuceneIndex != null) {
            cityPagesLuceneIndex.stop();
        }
    }

    public boolean isReady() {
        return cityPagesLuceneIndex == null || cityPagesLuceneIndex.isReady();
    }

    public int getRandomGeoId() {
        if (cityPagesLuceneIndex == null) {
            // TODO: Use correct exception.
            throw new HotelNotFoundException("Cannot generate random geoId");
        }
        return cityPagesLuceneIndex.search(searcher -> {
            var indexReader = searcher.getIndexReader();
            while (true) {
                int docNo = ThreadLocalRandom.current().nextInt(0, indexReader.maxDoc());
                var doc = searcher.doc(docNo);
                var geoIdField = doc.getField(FIELD_GEO_ID);
                if (geoIdField != null) {// May be missing at system doc
                    return geoIdField.numericValue().intValue();
                }
            }
        });
    }

    public TRegionPage getRegionPage(int geoId, String filterSlug) {
        if (filterSlug == null) {
            filterSlug = "";
        }
        Optional<TRegionPage> result = tryGetRegionPage(geoId, filterSlug);
        if (result.isPresent()) {
            staticCityPageFoundCounter.increment();
            return result.get();
        } else {
            staticCityPageNotFoundCounter.increment();
            // TODO: Use correct exception.
            throw new HotelNotFoundException(
                    String.format("Static city page not found by (geoId=%d, filterSlug='%s')", geoId, filterSlug)
            );
        }
    }

    public Optional<TRegionPage> tryGetRegionPage(int geoId) {
        return tryGetRegionPage(geoId, "");
    }

    public Optional<TRegionPage> tryGetRegionPage(int geoId, String filterSlug) {
        Optional<TRegionPage> result;
        if (cityPagesLuceneIndex != null) {
            result = cityPagesLuceneIndex.search(searcher -> {
                Query geoIdQuery = IntPoint.newExactQuery(FIELD_GEO_ID, geoId);
                Query filterSlugQuery = new TermQuery(new Term(FIELD_FILTER_SLUG, filterSlug));
                Query query = new BooleanQuery.Builder()
                        .add(new BooleanClause(geoIdQuery, BooleanClause.Occur.MUST))
                        .add(new BooleanClause(filterSlugQuery, BooleanClause.Occur.MUST))
                        .build();
                TopDocs topDocs = searcher.search(query, 1);
                if (topDocs.totalHits > 0) {
                    Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                    return Optional.of(TRegionPage.parseFrom(document.getField(FIELD_PROTO_DATA).binaryValue().bytes));
                } else {
                    return Optional.empty();
                }
            });
        } else {
            result = Optional.empty();
        }
        return result;
    }
}
