package ru.yandex.travel.api.services.hotels.suggest;

import java.time.Duration;
import java.util.Base64;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.AhcHttpRetryStrategy;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryRateLimiter;
import ru.yandex.travel.hotels.proto.suggest.Suggest.THotelSuggest;
import ru.yandex.travel.hotels.proto.suggest.Suggest.TRegionSuggest;

@Component
@EnableConfigurationProperties(HotelSuggestServiceProperties.class)
public class HotelSuggestService {

    private static final AhcHttpRetryStrategy RETRY_STRATEGY = new AhcHttpRetryStrategy(Duration.ZERO, 3);

    private final AsyncHttpClientWrapper httpClient;
    private final Retry retryHelper;
    private final RetryRateLimiter retryRateLimiter;
    private final ObjectMapper objectMapper = new ObjectMapper();

    private final String suggestUrl;

    private final int httpRequestTimeoutMs;
    private final int httpReadTimeoutMs;

    public HotelSuggestService(
            @Qualifier(value = "hotelSuggestAhcWrapper") AsyncHttpClientWrapper httpClient,
            Retry retryHelper,
            HotelSuggestServiceProperties hotelSuggestServiceProperties
    ) {
        this.httpClient = httpClient;
        this.retryHelper = retryHelper;
        this.retryRateLimiter = new RetryRateLimiter(hotelSuggestServiceProperties.getRetryRateLimit());

        this.suggestUrl = hotelSuggestServiceProperties.getBaseUrl() + "/hotels/suggest";

        this.httpRequestTimeoutMs = Math.toIntExact(hotelSuggestServiceProperties.getHttpRequestTimeout().toMillis());
        this.httpReadTimeoutMs = Math.toIntExact(hotelSuggestServiceProperties.getHttpReadTimeout().toMillis());
    }

    public CompletableFuture<Suggest> getSuggest(String requestId, String query, int limit) {
        final RequestBuilder requestBuilder = new RequestBuilder()
                .setReadTimeout(httpReadTimeoutMs)
                .setRequestTimeout(httpRequestTimeoutMs)
                .setUrl(suggestUrl)
                .addQueryParam("query", query)
                .addQueryParam("limit", String.valueOf(limit));

        return withRetry(requestBuilder, requestId, "HotelSuggestService::Suggest")
                .thenApply(this::readJson);
    }

    private CompletableFuture<Response> withRetry(RequestBuilder requestBuilder, String requestId, String name) {
        return retryHelper.withRetry(
                name,
                rb -> httpClient.executeRequest(rb, "GET", requestId).toCompletableFuture(),
                requestBuilder,
                RETRY_STRATEGY,
                retryRateLimiter
        );
    }

    private Suggest readJson(Response response) {
        final EncodedSuggest encodedSuggest = wrap(() -> objectMapper.readValue(response.getResponseBody(), new TypeReference<EncodedSuggest>() {}));
        return new Suggest(
                encodedSuggest.getHotelSuggests().stream().map(HotelSuggestService::toHotelSuggest).collect(Collectors.toList()),
                encodedSuggest.getRegionSuggests().stream().map(HotelSuggestService::toRegionSuggest).collect(Collectors.toList())
        );
    }

    private static THotelSuggest toHotelSuggest(String s) {
        return wrap(() -> THotelSuggest.parseFrom(Base64.getUrlDecoder().decode(s)));
    }

    private static TRegionSuggest toRegionSuggest(String s) {
        return wrap(() -> TRegionSuggest.parseFrom(Base64.getUrlDecoder().decode(s)));
    }

    @FunctionalInterface
    public interface ThrowingSupplier<T> {
        T get() throws Exception;
    }

    private static <T> T wrap(ThrowingSupplier<T> supplier) {
        try {
            return supplier.get();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}
