package ru.yandex.travel.api.services.hotels.tugc;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Objects;
import java.util.Set;

import io.grpc.ManagedChannelBuilder;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TvmHeaderClientHelper;
import ru.yandex.travel.hotels.proto.TPingRpcReq;
import ru.yandex.travel.hotels.proto.tugc_service.PingServiceV1Grpc;
import ru.yandex.travel.orders.client.ChannelState;
import ru.yandex.travel.orders.client.ChannelSupplier;
import ru.yandex.travel.orders.client.GrpcChannelSupplierFactory;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;
import ru.yandex.travel.orders.client.LabeledChannel;
import ru.yandex.travel.tvm.TvmWrapper;

import static ru.yandex.travel.commons.concurrent.FutureUtils.buildCompletableFuture;

@Slf4j
@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(TugcProperties.class)
public class TugcConfiguration {
    private final TugcProperties properties;

    @Bean("TugcSupplier")
    public ChannelSupplier getChannelSupplier() {
        GrpcChannelSupplierFactory grpcChannelSupplierFactory = new GrpcChannelSupplierFactory(properties);

        return grpcChannelSupplierFactory.getChannelSupplier();
    }

    public TvmHeaderClientHelper getTvmGrpcTvmHelper(@Autowired(required = false) TvmWrapper tvm) {
        if (tvm != null && properties.getTvm().isEnabled()) {
            return new TvmHeaderClientHelper(tvm);
        }

        return null;
    }

    @Bean("TugcGrpcChannelFactory")
    public HAGrpcChannelFactory haGrpcChannelFactory(@Qualifier("TugcSupplier") ChannelSupplier channelSupplier,
                                                     @Autowired(required = false) TvmWrapper tvm) {
        return HAGrpcChannelFactory.Builder.newBuilder().withPingProducer(
                channel -> buildCompletableFuture(PingServiceV1Grpc
                        .newFutureStub(channel)
                        .ping(TPingRpcReq.newBuilder().build())
                ).thenApply((rsp) -> rsp.getIsReady() ? ChannelState.READY : ChannelState.NOT_READY))
                .withFailureDetectorProperties(properties.getFailureDetection())
                .withChannelSupplier(channelSupplier)
                .withChannelBuilder((String target) -> createChannel(target, getTvmGrpcTvmHelper(tvm)))
                .build();
    }

    @SneakyThrows
    private LabeledChannel createChannel(String target, TvmHeaderClientHelper tvmHelper) {
        var clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
        var loggingClientInterceptor = new LoggingClientInterceptor(clientFqdn, target,
                Set.of(PingServiceV1Grpc.getPingMethod().getFullMethodName()));
        var defaultTimeoutClientInterceptor = new DefaultTimeoutClientInterceptor(properties.getTimeout());
        var intercepts = new ArrayList<>(Arrays.asList(loggingClientInterceptor, defaultTimeoutClientInterceptor));

        if (tvmHelper != null) {
            intercepts.add(tvmHelper.getInterceptor(properties.getTvm().getDestinationAlias()));
        }

        return new LabeledChannel(target,
                ManagedChannelBuilder
                        .forTarget(target)
                        .intercept(intercepts)
                        .usePlaintext()
                        .maxInboundMessageSize(properties.getMaxMessageSize())
                        .build());
    }
}
