package ru.yandex.travel.api.services.hotels.tugc;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import com.google.common.util.concurrent.ListenableFuture;
import io.grpc.Metadata;
import io.grpc.Status;
import io.grpc.StatusException;
import io.grpc.StatusRuntimeException;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.services.hotels.tugc.model.AddFavoriteHotelRsp;
import ru.yandex.travel.api.services.hotels.tugc.model.EmptyRsp;
import ru.yandex.travel.api.services.hotels.tugc.model.FavoriteGeoIdsRsp;
import ru.yandex.travel.api.services.hotels.tugc.model.FavoriteHotelsRsp;
import ru.yandex.travel.api.services.hotels.tugc.model.HotelFavoriteInfosRsp;
import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryRateLimiter;
import ru.yandex.travel.commons.retry.SpeculativeRetryStrategy;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.grpc.interceptors.ExplicitUserCredentialsClientInterceptor;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.proto.tugc_service.FavoriteInterfaceV1Grpc;
import ru.yandex.travel.hotels.proto.tugc_service.TAddFavoriteHotelReq;
import ru.yandex.travel.hotels.proto.tugc_service.TGetFavoriteHotelsReq;
import ru.yandex.travel.hotels.proto.tugc_service.TGetGeoIdsReq;
import ru.yandex.travel.hotels.proto.tugc_service.TGetHotelFavoriteInfosReq;
import ru.yandex.travel.hotels.proto.tugc_service.TRemoveFavoriteHotelsReq;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;

import static ru.yandex.travel.commons.concurrent.FutureUtils.buildCompletableFuture;

@Slf4j
@Component
public class TugcService {

    private final HAGrpcChannelFactory haGrpcChannelFactory;
    private final RetryRateLimiter rateLimiter;
    private final Retry retryHelper;

    public TugcService(@Qualifier("TugcGrpcChannelFactory") HAGrpcChannelFactory haGrpcChannelFactory,
                             Retry retryHelper) {
        this.haGrpcChannelFactory = haGrpcChannelFactory;
        this.rateLimiter = new RetryRateLimiter(0.3);
        this.retryHelper = retryHelper;
    }

    private FavoriteInterfaceV1Grpc.FavoriteInterfaceV1FutureStub getStub(UserCredentials userCredentials) {
        return FavoriteInterfaceV1Grpc
                .newFutureStub(haGrpcChannelFactory.getRoundRobinChannel())
                .withInterceptors(new ExplicitUserCredentialsClientInterceptor(userCredentials));
    }

    private <TReq, TRsp> CompletableFuture<TRsp> doRequest(String logId,
                                                           UserCredentials userCredentials,
                                                           String requestName,
                                                           BiFunction<FavoriteInterfaceV1Grpc.FavoriteInterfaceV1FutureStub, TReq, ListenableFuture<TRsp>> startRequest,
                                                           TReq request) {
        return retryHelper.withSpeculativeRetry("TugcService::" + requestName + "/" + logId,
                req -> buildCompletableFuture(startRequest.apply(getStub(userCredentials), request)),
                request, getSpeculativeRetryStrategy(), rateLimiter)
                .whenComplete((r, t) -> {
                    if (t != null) {
                        log.error("Failed tugc request '{}' ({})", requestName, logId, t);
                        Counter.builder("tugc.failedRequests").tag("method", requestName).register(Metrics.globalRegistry).increment();
                        Counter.builder("tugc.failedRequests").tag("method", "_TOTAL_").register(Metrics.globalRegistry).increment();
                    }
                });
    }

    public CompletableFuture<AddFavoriteHotelRsp> addFavoriteHotel(String logId, UserCredentials userCredentials, Permalink permalink, int geoId) {
        return doRequest(logId, userCredentials, "addFavoriteHotel",
                FavoriteInterfaceV1Grpc.FavoriteInterfaceV1FutureStub::addFavoriteHotel,
                TAddFavoriteHotelReq.newBuilder().setPermalink(permalink.asLong()).setGeoId(geoId).build())
                .handle((protoRsp, exception) -> {
                    if (exception == null) {
                        return new AddFavoriteHotelRsp(false);
                    }

                    Throwable actual = exception.getCause();

                    if (actual instanceof StatusException || actual instanceof StatusRuntimeException) {
                        Metadata metadata = Status.trailersFromThrowable(actual);
                        TError error = metadata.get(ServerUtils.METADATA_ERROR_KEY);

                        if (error == null) {
                            throw new IllegalStateException(String.format("%s: Unexpected grpc error, message=%s",
                                    logId,
                                    exception.getMessage()));
                        }

                        if (error.getCode().equals(EErrorCode.EC_HOTEL_LIMIT_EXCEEDED)) {
                            return new AddFavoriteHotelRsp(true);
                        } else {
                            throw new IllegalStateException(String.format("%s: Unexpected grpc error, code=%s, message=%s",
                                    logId,
                                    error.getCode(),
                                    error.getMessage()));
                        }
                    }

                    throw new IllegalStateException(String.format("%s: Unexpected grpc error, message=%s",
                            logId,
                            exception.getMessage()));
                });
    }

    public CompletableFuture<EmptyRsp> removeFavoriteHotel(String logId, UserCredentials userCredentials, Permalink permalink) {
        return removeFavoriteHotelsImpl(logId, userCredentials,
                TRemoveFavoriteHotelsReq.newBuilder().setPermalink(permalink.asLong()).build());
    }

    public CompletableFuture<EmptyRsp> removeFavoriteHotels(String logId, UserCredentials userCredentials, int geoId) {
        return removeFavoriteHotelsImpl(logId, userCredentials,
                TRemoveFavoriteHotelsReq.newBuilder().setGeoId(geoId).build());
    }

    public CompletableFuture<EmptyRsp> removeFavoriteHotelsImpl(String logId, UserCredentials userCredentials, TRemoveFavoriteHotelsReq req) {
        return doRequest(logId, userCredentials, "removeFavoriteHotels",
                FavoriteInterfaceV1Grpc.FavoriteInterfaceV1FutureStub::removeFavoriteHotels, req)
                .thenApply(protoRsp -> new EmptyRsp());
    }

    public CompletableFuture<FavoriteHotelsRsp> getFavoriteHotels(String logId, UserCredentials userCredentials) {
        return getFavoriteHotelsImpl(logId, userCredentials, TGetFavoriteHotelsReq.newBuilder().build());
    }

    public CompletableFuture<FavoriteHotelsRsp> getFavoriteHotels(String logId, UserCredentials userCredentials, int geoId) {
        return getFavoriteHotelsImpl(logId, userCredentials, TGetFavoriteHotelsReq.newBuilder().setGeoId(geoId).build());
    }

    private CompletableFuture<FavoriteHotelsRsp> getFavoriteHotelsImpl(String logId, UserCredentials userCredentials, TGetFavoriteHotelsReq req) {
        return doRequest(logId, userCredentials, "getFavoriteHotels",
                FavoriteInterfaceV1Grpc.FavoriteInterfaceV1FutureStub::getFavoriteHotels,
                req)
                .thenApply(protoRsp -> new FavoriteHotelsRsp(protoRsp.getPermalinksList()
                        .stream().map(Permalink::of).collect(Collectors.toUnmodifiableList())));
    }

    public CompletableFuture<HotelFavoriteInfosRsp> getHotelFavoriteInfos(String logId, UserCredentials userCredentials, List<Permalink> permalinks) {
        return doRequest(logId, userCredentials, "getHotelFavoriteInfos",
                FavoriteInterfaceV1Grpc.FavoriteInterfaceV1FutureStub::getHotelFavoriteInfos,
                TGetHotelFavoriteInfosReq.newBuilder().addAllPermalinks(
                        permalinks.stream().map(Permalink::asLong).collect(Collectors.toUnmodifiableList())
                ).build())
                .thenApply(protoRsp -> new HotelFavoriteInfosRsp(protoRsp.getPermalinkInfosList()
                        .stream()
                        .collect(Collectors.toUnmodifiableMap(x -> Permalink.of(x.getPermalink()), x -> x.getIsFavorite())))
                );
    }

    public CompletableFuture<FavoriteGeoIdsRsp> getFavoriteGeoIds(String logId, UserCredentials userCredentials) {
        return doRequest(logId, userCredentials, "getFavoriteGeoIds",
                FavoriteInterfaceV1Grpc.FavoriteInterfaceV1FutureStub::getGeoIds,
                TGetGeoIdsReq.newBuilder().build())
                .thenApply(protoRsp -> new FavoriteGeoIdsRsp(protoRsp.getGeoIdInfosList().stream()
                            .map(x -> new FavoriteGeoIdsRsp.GeoIdInfo(x.getGeoId(), x.getPermalinkCount()))
                            .collect(Collectors.toUnmodifiableList())));
    }

    public <T> SpeculativeRetryStrategy<T> getSpeculativeRetryStrategy() {
        return SpeculativeRetryStrategy.<T>builder()
                .shouldRetryOnException(e -> {
                    var status = Status.fromThrowable(e);
                    if (List.of(Status.Code.INVALID_ARGUMENT, Status.Code.NOT_FOUND, Status.Code.ALREADY_EXISTS,
                            Status.Code.PERMISSION_DENIED, Status.Code.FAILED_PRECONDITION, Status.Code.OUT_OF_RANGE,
                            Status.Code.UNIMPLEMENTED, Status.Code.UNAUTHENTICATED).contains(status.getCode())) {
                        return false;
                    }

                    Metadata metadata = Status.trailersFromThrowable(e);
                    if (metadata == null) {
                        return true;
                    }

                    TError error = metadata.get(ServerUtils.METADATA_ERROR_KEY);
                    if (error == null) {
                        return true;
                    }

                    return !error.getCode().equals(EErrorCode.EC_HOTEL_LIMIT_EXCEEDED) &&
                            !error.getCode().equals(EErrorCode.EC_PERMISSION_DENIED);
                })
                .validateResult(resp -> {})
                .timeout(Duration.ofSeconds(3))
                .retryDelay(Duration.ofMillis(400))
                .numRetries(5)
                .build();
    }
}
