package ru.yandex.travel.api.services.hotels.ugc;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.asynchttpclient.request.body.multipart.ByteArrayPart;
import org.asynchttpclient.request.body.multipart.StringPart;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcAddReviewReq;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcAttribution;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcChangeReviewRsp;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcDeletePhotoReq;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcDeleteReviewReq;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcDigestReq;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcDigestRsp;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcEditReviewReq;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcGetCurrentUserReviewReq;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcGetCurrentUserReviewRsp;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcSetReviewReactionReq;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcSetReviewReactionRsp;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcUploadImageHttpRsp;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcUploadImageReq;
import ru.yandex.travel.api.services.hotels.ugc.model.UgcUploadImageRsp;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.AhcHttpRetryStrategy;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryRateLimiter;
import ru.yandex.travel.credentials.UnauthenticatedUserException;
import ru.yandex.travel.hotels.common.HotelNotFoundException;
import ru.yandex.travel.tvm.TvmWrapper;


@Component
@EnableConfigurationProperties(UgcServiceProperties.class)
@Slf4j
public class UgcService {
    private final UgcServiceProperties config;
    private final AsyncHttpClientWrapper client;
    private final TvmWrapper tvm;
    private final UgcParser parser;
    private final UgcSerializer serializer;
    private final Retry retryHelper;
    private final RetryRateLimiter retryRateLimiter;

    public UgcService(UgcServiceProperties config,
                      @Qualifier(value = "ugcAhcWrapper") AsyncHttpClientWrapper client, Retry retryHelper,
                      ObjectMapper mapper, @Autowired(required = false) TvmWrapper tvm) {
        this.config = config;
        this.client = client;
        this.tvm = tvm;
        this.retryHelper = retryHelper;
        this.retryRateLimiter = new RetryRateLimiter(config.getRetryRateLimit());
        this.parser = new UgcParser(mapper);
        this.serializer = new UgcSerializer();
        if (tvm != null && config.getTvmDestinationAlias() != null) {
            tvm.validateAlias(config.getTvmDestinationAlias());
        }
    }

    private void checkStatus(String prefix, Response response) {
        if (response.getStatusCode() == HttpStatus.SC_200_OK) {
            return;
        }

        String msg = String.format("UGC returned %s, message '%s'", response.getStatusCode(), response.getResponseBody());
        if (response.getStatusCode() == HttpStatus.SC_400_BAD_REQUEST) {
            log.error("{}: Illegal argument: {}", prefix, msg);
            throw new TravelApiBadRequestException(msg); // Give up on trying to reproduce UGC validation
        }
        if (response.getStatusCode() == HttpStatus.SC_404_NOT_FOUND) {
            log.error("{}: Hotel, review or photo not found: {}", prefix, msg);
            throw new HotelNotFoundException(msg);
        }
        if (response.getStatusCode() == HttpStatus.SC_409_CONFLICT) {
            log.error("{}: Conflict: {}", prefix, msg);
            throw new TravelApiBadRequestException(msg);
        }
        if (response.getStatusCode() == HttpStatus.SC_401_UNAUTHORIZED) {
            log.error("{}: User unauthorized: {}", prefix, msg);
            throw new UnauthenticatedUserException(msg);
        }
        log.error("{}: Bad HTTP status: {}", prefix, msg);
        throw new HttpException(response.getStatusCode(), msg);
    }

    public CompletableFuture<UgcDigestRsp> getDigest(UgcDigestReq req) {
        RequestBuilder httpReq = prepareDigestRequest(req);
        String prefix = String.format("UgcService::getDigest/%s/(Permalink %s)",
                req.getAttribution().getReqId(), req.getPermalink());
        log.debug("{}: Sending, full url: '{}'", prefix, httpReq.build().getUrl());
        return retryHelper.withRetry(prefix, this::runRequest, httpReq,
                new AhcHttpRetryStrategy(), retryRateLimiter).thenApply(httpResponse -> {
            checkStatus(prefix, httpResponse);
            try {
                UgcDigestRsp rsp = parser.parseDigestRsp(httpResponse.getResponseBodyAsBytes());
                log.debug("{}: Finished", prefix);
                return rsp;
            } catch (InvalidProtocolBufferException e) {
                log.error("{}: Unable to deserialize UGC get-digest proto response", prefix, e);
                throw new RuntimeException(e);
            }
        });
    }

    public CompletableFuture<UgcSetReviewReactionRsp> setReviewReaction(UgcSetReviewReactionReq req) {
        RequestBuilder httpReq = prepareSetReviewReactionReq(req);
        String prefix = String.format("UgcService::setReviewReaction/%s/(Permalink %s)",
                req.getAttribution().getReqId(), req.getPermalink());
        log.debug("Sending {}, full url: '{}'", prefix, httpReq.build().getUrl());
        return retryHelper.withRetry(prefix, this::runRequest, httpReq,
                new AhcHttpRetryStrategy(), retryRateLimiter).thenApply(httpResponse -> {
            checkStatus(prefix, httpResponse);
            log.debug("{}: Finished", prefix);
            return new UgcSetReviewReactionRsp();
        });
    }

    public CompletableFuture<UgcUploadImageRsp> uploadPhoto(UgcUploadImageReq req) throws IOException {
        RequestBuilder httpReq = prepareUploadImageReq(req);
        String prefix = String.format("UgcService::uploadPhoto/%s/(Permalink %s)",
                req.getAttribution().getReqId(), req.getPermalink());
        log.debug("Sending {}, full url: '{}'", prefix, httpReq.build().getUrl());

        return retryHelper.withRetry(prefix, this::runRequest, httpReq,
                new AhcHttpRetryStrategy(), retryRateLimiter).thenApply(httpResponse -> {
            checkStatus(prefix, httpResponse);
            log.debug("{}: Finished", prefix);

            UgcUploadImageHttpRsp rsp = parser.parseUploadImageRsp(httpResponse);
            String status = rsp.getResponse().getStatus();

            if (!status.equals("ok")) {
                throw new HttpException(500, status);
            }

            return rsp.getResponse().getImage();
        });
    }

    public CompletableFuture<Void> deletePhoto(UgcDeletePhotoReq req) {
        RequestBuilder httpReq = prepareDeleteImageReq(req);
        String prefix = String.format("UgcService::deletePhoto/%s/(Permalink %s)",
                req.getAttribution().getReqId(), req.getPermalink());
        log.debug("Sending {}, full url: '{}'", prefix, httpReq.build().getUrl());

        return retryHelper.withRetry(prefix, this::runRequest, httpReq,
                new AhcHttpRetryStrategy(), retryRateLimiter).thenApply(httpResponse -> {
           checkStatus(prefix, httpResponse);
           log.debug("{}: Finished", prefix);

           return null;
        });
    }

    public CompletableFuture<UgcChangeReviewRsp> addReview(UgcAddReviewReq req) {
        RequestBuilder httpReq = prepareAddReviewReq(req);
        String prefix = String.format("UgcService::addReview/%s/(Permalink %s)",
                req.getAttribution().getReqId(), req.getPermalink());
        log.debug("Sending {}, full url: '{}'", prefix, httpReq.build().getUrl());

        return retryHelper.withRetry(prefix, this::runRequest, httpReq,
                new AhcHttpRetryStrategy(), retryRateLimiter).thenApply(httpResponse -> {
            checkStatus(prefix, httpResponse);

            try {
                UgcChangeReviewRsp rsp = parser.parseChangeReviewRsp(httpResponse.getResponseBodyAsBytes());
                log.debug("{}: Finished", prefix);

                return rsp;
            } catch (InvalidProtocolBufferException e) {
                log.error("{}: Unable to deserialize UGC add-review proto response", prefix, e);

                throw new RuntimeException(e);
            }
        });
    }

    public CompletableFuture<UgcChangeReviewRsp> editReview(UgcEditReviewReq req) {
        RequestBuilder httpReq = prepareEditReviewReq(req);
        String prefix = String.format("UgcService::editReview/%s/(Permalink %s)",
                req.getAttribution().getReqId(), req.getPermalink());
        log.debug("Sending {}, full url: '{}'", prefix, httpReq.build().getUrl());

        return retryHelper.withRetry(prefix, this::runRequest, httpReq,
                new AhcHttpRetryStrategy(), retryRateLimiter).thenApply(httpResponse -> {
            checkStatus(prefix, httpResponse);

            try {
                UgcChangeReviewRsp rsp = parser.parseChangeReviewRsp(httpResponse.getResponseBodyAsBytes());
                log.debug("{}: Finished", prefix);

                return rsp;
            } catch (InvalidProtocolBufferException e) {
                log.error("{}: Unable to deserialize UGC edit-review proto response", prefix, e);

                throw new RuntimeException(e);
            }
        });
    }

    public CompletableFuture<Void> deleteReview(UgcDeleteReviewReq req) {
        RequestBuilder httpReq = prepareDeleteReviewReq(req);
        String prefix = String.format("UgcService::deleteReview/%s/(Permalink %s)",
                req.getAttribution().getReqId(), req.getPermalink());
        log.debug("Sending {}, full url: '{}'", prefix, httpReq.build().getUrl());

        return retryHelper.withRetry(prefix, this::runRequest, httpReq,
                new AhcHttpRetryStrategy(), retryRateLimiter).thenApply(httpResponse -> {
            checkStatus(prefix, httpResponse);

            return null;
        });
    }

    public CompletableFuture<UgcGetCurrentUserReviewRsp> getCurrentUserReview(UgcGetCurrentUserReviewReq req) {
        RequestBuilder httpReq = prepareGetCurrentUserReviewReq(req);
        String prefix = String.format("UgcService::getCurrentUserReview/%s/(Permalink %s)",
            req.getAttribution().getReqId(), req.getPermalink());
        log.debug("Sending {}, full url: '{}'", prefix, httpReq.build().getUrl());

        return retryHelper.withRetry(prefix, this::runRequest, httpReq,
                new AhcHttpRetryStrategy(), retryRateLimiter).thenApply(httpResponse -> {
            checkStatus(prefix, httpResponse);

            try {
                UgcGetCurrentUserReviewRsp rsp = parser.parseGetCurrentUserReviewRsp(httpResponse.getResponseBodyAsBytes());
                log.debug("{}: Finished", prefix);

                return rsp;
            } catch (InvalidProtocolBufferException e) {
                log.error("{}: Unable to deserialize UGC get-my-review proto response", prefix, e);

                throw new RuntimeException(e);
            }
        });
    }

    private CompletableFuture<Response> runRequest(RequestBuilder request) {
        return client.executeRequest(request);
    }

    private String booleanToString(boolean v) {
        return v ? "true" : "false";
    }

    private void setCommonFields(RequestBuilder builder, UgcAttribution attribution, boolean isEnabledTestUgc) {
        builder.setReadTimeout(Math.toIntExact(config.getHttpReadTimeout().toMillis()));
        builder.setRequestTimeout(Math.toIntExact(config.getHttpRequestTimeout().toMillis()));
        builder.setHeader("Accept", "application/octet-stream");
        builder.setHeader("X-App", config.getAppId());
        if (!Strings.isNullOrEmpty(attribution.getPassportUid())) {
            builder.addHeader("X-Passport-UID", attribution.getPassportUid());
        }
        if (!Strings.isNullOrEmpty(attribution.getYandexUid())) {
            builder.addHeader("X-Yandex-UID", attribution.getYandexUid());
        }
        if (!Strings.isNullOrEmpty(attribution.getUserIp())) {
            builder.addHeader("X-User-IP", attribution.getUserIp());
        }
        if (!Strings.isNullOrEmpty(attribution.getReqId())) {
            builder.addHeader("X-Req-Id", attribution.getReqId());
        }
        if (tvm != null && config.getTvmDestinationAlias() != null && !isEnabledTestUgc) {
            String serviceTicket = tvm.getServiceTicket(config.getTvmDestinationAlias());
            builder.setHeader(CommonHttpHeaders.HeaderType.SERVICE_TICKET.getHeader(), serviceTicket);
        }
    }

    private RequestBuilder prepareDigestRequest(UgcDigestReq req) {
        String url = getOrgUrl(req.isEnabledTestUgc()) + "/" + req.getPermalink() + "/get-reviews";
        //strange address for GetDigest handle, isn't it?

        RequestBuilder builder = new RequestBuilder()
                .setUrl(url)
                .addQueryParam("limit", Integer.toString(req.getLimit()))
                .addQueryParam("offset", Integer.toString(req.getOffset()))
                .addQueryParam("my_review", booleanToString(req.isIncludeMyReview()))
                .addQueryParam("my_reactions", booleanToString(req.isIncludeMyReactions()))
                .addQueryParam("key_phrases", booleanToString(req.isIncludeKeyPhrases()))
                .addQueryParam("ranking", req.getRanking().toString())
                .addQueryParam("snippet", booleanToString(req.isIncludeSnippet()));
        if (!Strings.isNullOrEmpty(req.getKeyPhrase())) {
            builder.addQueryParam("key_phrase", req.getKeyPhrase());
        }
        setCommonFields(builder, req.getAttribution(), req.isEnabledTestUgc());
        return builder;
    }

    private RequestBuilder prepareSetReviewReactionReq(UgcSetReviewReactionReq req) {
        String url = getOrgUrl(req.isEnabledTestUgc()) + "/" + req.getPermalink() + "/react-review";
        RequestBuilder builder = new RequestBuilder()
                .setUrl(url)
                .setMethod("POST")
                .addQueryParam("review_id", req.getReviewId())
                .addQueryParam("react", req.getType().name());
        setCommonFields(builder, req.getAttribution(), req.isEnabledTestUgc());
        return builder;
    }

    private RequestBuilder prepareUploadImageReq(UgcUploadImageReq req) throws IOException {
        String url = getBaseUrl(req.isEnabledTestUgc()) + "/upload2-internal";

        return new RequestBuilder()
                .setUrl(url)
                .setMethod("GET")
                .setHeader("Accept", "application/json")
                .setReadTimeout(Math.toIntExact(config.getHttpReadTimeout().toMillis()))
                .setRequestTimeout(Math.toIntExact(config.getUploadRequestTimeout().toMillis()))
                .addQueryParam("review_photo", booleanToString(true))
                .addQueryParam("appId", config.getAppId())
                .addQueryParam("uid", req.getAttribution().getPassportUid())
                .addBodyPart(new ByteArrayPart("image_source", req.getImage().getBytes(), req.getImage().getContentType()))
                .addBodyPart(new StringPart("address", req.getPermalink().toString()));
    }

    private RequestBuilder prepareDeleteImageReq(UgcDeletePhotoReq req) {
        String url = getUserUrl(req.isEnabledTestUgc()) + "/" + req.getAttribution().getPassportUid() + "/orgs/" + req.getPermalink() +
                "/photos/" + req.getImageId();

        return new RequestBuilder()
                .setUrl(url)
                .setMethod("DELETE")
                .setReadTimeout(Math.toIntExact(config.getHttpReadTimeout().toMillis()))
                .setRequestTimeout(Math.toIntExact(config.getHttpRequestTimeout().toMillis()))
                .addQueryParam("appId", config.getAppId());
    }

    private RequestBuilder prepareAddReviewReq(UgcAddReviewReq req) {
        String url = getOrgUrl(req.isEnabledTestUgc()) + "/" + req.getPermalink() + "/add-my-review";
        byte[] body = serializer.serializeAddReviewBody(req);

        RequestBuilder builder = new RequestBuilder()
                .setUrl(url)
                .setMethod("POST")
                .setHeader("Content-Type", "application/octet-stream")
                .setBody(body);

        setCommonFields(builder, req.getAttribution(), req.isEnabledTestUgc());

        return builder;
    }

    private RequestBuilder prepareEditReviewReq(UgcEditReviewReq req) {
        String url = getOrgUrl(req.isEnabledTestUgc()) + "/" + req.getPermalink() + "/edit-my-review";
        byte[] body = serializer.serializeEditReviewBody(req);

        RequestBuilder builder = new RequestBuilder()
                .setUrl(url)
                .setMethod("POST")
                .setHeader("Content-Type", "application/octet-stream")
                .setBody(body)
                .addQueryParam("review_id", req.getReviewId());

        setCommonFields(builder, req.getAttribution(), req.isEnabledTestUgc());

        return builder;
    }

    private RequestBuilder prepareDeleteReviewReq(UgcDeleteReviewReq req) {
        String url = getOrgUrl(req.isEnabledTestUgc()) + "/" + req.getPermalink() + "/delete-my-review";

        RequestBuilder builder = new RequestBuilder()
                .setUrl(url)
                .setMethod("POST")
                .addQueryParam("review_id", req.getReviewId());

        setCommonFields(builder, req.getAttribution(), req.isEnabledTestUgc());

        return builder;
    }

    private RequestBuilder prepareGetCurrentUserReviewReq(UgcGetCurrentUserReviewReq req) {
        String url = getOrgUrl(req.isEnabledTestUgc()) + "/" + req.getPermalink() + "/get-my-review";

        RequestBuilder builder = new RequestBuilder()
                .setUrl(url)
                .setMethod("GET");

        setCommonFields(builder, req.getAttribution(), req.isEnabledTestUgc());

        return builder;
    }

    private String getBaseUrl(Boolean isEnableTestUgc) {
        if (isEnableTestUgc && config.isEnabledTestUgc()) {
            return config.getTestBaseUrl();
        }

        return config.getBaseUrl();
    }

    private String getOrgUrl(Boolean isEnableTestUgc) {
        return getBaseUrl(isEnableTestUgc) + config.getUrlPrefixes().getOrgs();
    }

    private String getUserUrl(Boolean isEnableTestUgc) {
        return getBaseUrl(isEnableTestUgc) + config.getUrlPrefixes().getUsers();
    }
}
