package ru.yandex.travel.api.services.hotels_booking_flow;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.protobuf.Message;
import io.micrometer.core.instrument.Counter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.javamoney.moneta.Money;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import ru.yandex.travel.api.config.hotels.HotelOfferConfigurationProperties;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.AppliedPromoCampaignsDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.YandexPlusApplicationDto;
import ru.yandex.travel.api.exceptions.HotelInfoNotFoundException;
import ru.yandex.travel.api.services.hotels_booking_flow.promo.HotelPromoCampaignsService;
import ru.yandex.travel.api.services.orders.Meters;
import ru.yandex.travel.api.services.orders.user_info.UserInfoGrpcService;
import ru.yandex.travel.clients.promo_service_booking_flow.PromoServiceBookingFlowUtils;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.health.HealthCheckedSupplier;
import ru.yandex.travel.commons.health.UnhealthyServiceException;
import ru.yandex.travel.commons.messaging.KeyValueStorage;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.common.token.TravelToken;
import ru.yandex.travel.hotels.models.booking_flow.BedGroupInfo;
import ru.yandex.travel.hotels.models.booking_flow.DiscountInfo;
import ru.yandex.travel.hotels.models.booking_flow.HotelInfo;
import ru.yandex.travel.hotels.models.booking_flow.LegalInfo;
import ru.yandex.travel.hotels.models.booking_flow.MetaInfo;
import ru.yandex.travel.hotels.models.booking_flow.Offer;
import ru.yandex.travel.hotels.models.booking_flow.RateInfo;
import ru.yandex.travel.hotels.models.booking_flow.SearchInfo;
import ru.yandex.travel.hotels.models.booking_flow.promo.Mir2020PromoCampaign;
import ru.yandex.travel.hotels.models.booking_flow.promo.PromoCampaignsInfo;
import ru.yandex.travel.hotels.models.booking_flow.promo.YandexPlusPromoCampaign;
import ru.yandex.travel.hotels.proto.EMirEligibility;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TExperimentInfo;
import ru.yandex.travel.hotels.proto.THotelTestContext;
import ru.yandex.travel.hotels.proto.TOfferData;
import ru.yandex.travel.hotels.proto.TOfferInfo;
import ru.yandex.travel.hotels.proto.TUserInfo;
import ru.yandex.travel.hotels.services.promoservice.PromoServiceClient;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;

@Slf4j
@EnableConfigurationProperties({OfferServiceConfigurationProperties.class,HotelOfferConfigurationProperties.class})
public abstract class AbstractPartnerBookingProvider<T extends Message> implements PartnerBookingProvider {

    // Usually we insert service dependency via constructor (by declaring them as final and adding
    // @RequiredArgsConstructor annotation, but here it is not possible since this class gets inherited - and
    // lombok does not support combination of inheritance and @RequiredArgsConstructor.
    @Autowired
    OfferServiceConfigurationProperties offerConfig;
    @Autowired
    GeoSearchHotelContentService geoSearchHotelContentService;
    @Autowired
    HealthCheckedSupplier<KeyValueStorage> storageSupplier;
    @Autowired
    Meters meters;
    @Autowired
    ChecksumService checksumService;
    @Autowired
    HotelPromoCampaignsService promoCampaignsService;
    @Autowired
    PaymentScheduleService paymentScheduleService;
    @Autowired
    PromoServiceClient promoServiceClient;
    @Autowired
    HotelOfferConfigurationProperties offerProperties;
    @Autowired
    UserInfoGrpcService userInfoGrpcService;
    @Autowired
    ExperimentDataProvider experimentDataProvider;

    @Override
    public CompletableFuture<Offer> getOffer(BookingFlowContext context, Integer yandexPlusBalance) {
        try {
            var offerDataFuture = fetchOfferData(context.getDecodedToken(), context.getStage(), context);
            context.setOfferDataFuture(offerDataFuture.exceptionally(t -> null));
            var testContextFuture = getTestContextFuture(offerDataFuture);
            context.setTestContextFuture(testContextFuture);
            var geoDataFutures = geoSearchHotelContentService.getFutures(context);
            var partnerFutures = getPartnerFutures(context, offerDataFuture.thenApply(this::getPartnerOffer));
            context.setGeoSearchFutures(geoDataFutures);
            context.setPartnerFutures(partnerFutures);
            var hotelInfoFuture = wrapFailureToNull(geoDataFutures.getHotelInfoFuture(), context,
                    BookingProviderMethod.HOTEL_INFO);
            var partnerHotelInfoFuture = wrapFailureToNull(partnerFutures.getPartnerHotelInfo(), context,
                    BookingProviderMethod.PARTNER_HOTEL_INFO);
            var rateInfoFuture = wrapFailureToNull(partnerFutures.getRateInfo(), context,
                    BookingProviderMethod.RATE_INFO);
            var stayInfoFuture = wrapFailureToNull(partnerFutures.getStayInfo(), context,
                    BookingProviderMethod.STAY_INFO);
            var roomInfoFuture = wrapFailureToNull(partnerFutures.getRoomInfo(), context,
                    BookingProviderMethod.ROOM_INFO);
            var refundRulesFuture =
                    wrapFailureToNull(partnerFutures.getRefundRules(), context, BookingProviderMethod.REFUND_RULES);
            var legalInfoFuture = combineLegalInfoItems(
                    wrapFailureToNull(geoDataFutures.getHotelLegalItemFuture(), context,
                            BookingProviderMethod.HOTEL_LEGAL),
                    wrapFailureToNull(partnerFutures.getHotelLegalInfoItem(), context,
                            BookingProviderMethod.HOTEL_LEGAL),
                    wrapFailureToNull(partnerFutures.getPartnerLegalInfoItem(), context,
                            BookingProviderMethod.PARTNER_LEGAL),
                    context);
            var discountInfoFuture = wrapFailureToNull(
                    calculateDiscountForOffer(context, rateInfoFuture, testContextFuture), context,
                    BookingProviderMethod.DISCOUNT_INFO);

            return CompletableFuture.allOf(context.getOfferDataFuture(), hotelInfoFuture, partnerHotelInfoFuture,
                            rateInfoFuture, roomInfoFuture, stayInfoFuture, refundRulesFuture, legalInfoFuture,
                            testContextFuture, discountInfoFuture)
                    .thenCompose(theVoid -> {
                        var offerData = context.getOfferDataFuture().join();
                        RateInfo rateInfo = rateInfoFuture.join();
                        RefundRules refundRules = refundRulesFuture.join();
                        HotelInfo hotelInfo = hotelInfoFuture.join();

                        if (hotelInfo == null) {
                            throwGeosearchHotelNotFoundError(context);
                        }

                        THotelTestContext testContext = testContextFuture.join();
                        final DiscountInfo discountInfo = discountInfoFuture.join();
                        var builder = Offer.builder()
                                .hotelInfo(hotelInfo)
                                .partnerHotelInfo(partnerHotelInfoFuture.join())
                                .rateInfo(rateInfo)
                                .roomInfo(roomInfoFuture.join())
                                .stayInfo(stayInfoFuture.join())
                                .refundRules(refundRules)
                                .legalInfo(legalInfoFuture.join())
                                .discountInfo(discountInfo)
                                .allGuestsRequired(offerProperties.getPartnersWithAllGuests().stream()
                                        .anyMatch(partner -> Objects.equals(partner, getPartnerId().toString())));

                        builder.allowPostPay(isPostPayAllowed(offerData, context, hotelInfo));
                        Money priceBeforePromos = builder.build().calculateActualPrice();

                        CompletableFuture<PromoCampaignsInfo> promoCampaignsInfoFuture;
                        if (priceBeforePromos == null) {
                            // not a proper booking offer, no need to calculate promo campaigns
                            promoCampaignsInfoFuture = CompletableFuture.completedFuture(PromoCampaignsInfo.builder().build());
                        } else {
                            CheckParamsRequest request = CheckParamsRequest.buildFromContext(
                                    rateInfoOrNull(rateInfo),
                                    priceBeforePromos,
                                    context);

                            promoCampaignsInfoFuture = userInfoGrpcService.getUserExistingOrderTypes(context.getUserCredentials())
                                    .thenCompose(userExistingOrderTypes -> promoCampaignsService.getCommonPromoCampaignsInfo(request, userExistingOrderTypes, context.getHeaders()))
                                    .thenApply(commonCampaigns -> {
                                        YandexPlusPromoCampaign yandexPlus = promoCampaignsService.getYandexPlusCampaign(commonCampaigns, request, yandexPlusBalance);

                                        Money discountedAmount;
                                        if (discountInfo != null && discountInfo.isApplied()) {
                                            discountedAmount = discountInfo.getDiscountedPrice();
                                        } else {
                                            discountedAmount = rateInfo.getTotalMoney();
                                        }

                                        boolean needWithdraw = YandexPlusApplicationDto.DtoMode.WITHDRAW == Optional.ofNullable(context.getAppliedPromoCampaigns())
                                                .map(AppliedPromoCampaignsDto::getYandexPlus)
                                                .map(YandexPlusApplicationDto::getMode)
                                                .orElse(YandexPlusApplicationDto.DtoMode.TOPUP);

                                        Money priceAfterPlusWithdraw = discountedAmount;
                                        if (rateInfo != null && priceAfterPlusWithdraw != null && yandexPlus != null && yandexPlus.getWithdrawPoints() != null) {
                                            Money plusWithdrawPoints = Money.of(yandexPlus.getWithdrawPoints(), ProtoCurrencyUnit.RUB);
                                            priceAfterPlusWithdraw = priceAfterPlusWithdraw.subtract(plusWithdrawPoints);
                                            rateInfo.setPriceAfterPlusWithdraw(priceAfterPlusWithdraw);
                                        }

                                        return promoCampaignsService.calculatePromoCampaignsInfo(
                                                commonCampaigns,
                                                request,
                                                yandexPlus,
                                                needWithdraw ? priceAfterPlusWithdraw : discountedAmount
                                        );
                                    });
                        }

                        return promoCampaignsInfoFuture.thenApply(promoCampaignsInfo -> {
                                Mir2020PromoCampaign mir2020 = promoCampaignsInfo.getMir2020();
                                if (rateInfo != null && rateInfo.getTotalRate() != null
                                        && (mir2020 == null || mir2020.getEligibility() != EMirEligibility.ME_ELIGIBLE)) {
                                    builder.deferredPaymentSchedule(paymentScheduleService.getDeferredSchedule(
                                            priceBeforePromos, priceBeforePromos, refundRules, hotelInfo, context, testContext));
                                }
                                builder.promoCampaignsInfo(promoCampaignsInfo);
                                return builder.metaInfo(buildMetaInfo(context, builder.build())).build();
                            });

                    });
        } catch (Exception ex) {
            return CompletableFuture.failedFuture(ex);
        }
    }

    protected BedGroupInfo createOneBedGroupInfo(List<BedGroupInfo> beds) {
        String newDescription = beds.stream().map(bg -> bg.getDescription()).collect(Collectors.joining(" или "));
        return new BedGroupInfo(0, newDescription);
    }

    private CompletableFuture<DiscountInfo> calculateDiscountForOffer(
            BookingFlowContext context,
            CompletableFuture<RateInfo> rateInfoFuture,
            CompletableFuture<THotelTestContext> testContextFuture) {
        return CompletableFuture.allOf(rateInfoFuture, testContextFuture)
                .thenCompose(ignored -> {
                    final var testContext = FutureUtils.joinCompleted(testContextFuture);
                    final var rateInfo = FutureUtils.joinCompleted(rateInfoFuture);
                    final Money priceFromPartnerOffer = rateInfoOrNull(rateInfo);

                    if (testContext != null && testContext.hasDiscountAmount()) {
                        Money discountAmount = Money.of(testContext.getDiscountAmount().getValue(),
                                priceFromPartnerOffer.getCurrency());
                        return CompletableFuture.completedFuture(DiscountInfo.builder()
                                .applied(true)
                                .discountAmount(discountAmount)
                                .discountedPrice(priceFromPartnerOffer.subtract(discountAmount))
                                .build());
                    } else {
                        try {
                            final CheckParamsRequest params = CheckParamsRequest.buildFromContext(
                                    priceFromPartnerOffer, null, context);
                            TOfferInfo offerInfo = PromoServiceBookingFlowUtils.buildOfferInfo(
                                    params.getPartnerId(),
                                    params.getHotelId(),
                                    params.getCheckIn().toString(),
                                    params.getCheckOut().toString(),
                                    params.getPriceFromPartnerOffer(),
                                    null,
                                    null
                            );
                            TUserInfo userInfo = PromoServiceBookingFlowUtils.buildUserInfo(
                                    params.getPassportId(),
                                    params.getIsPlusUser()
                            );
                            TExperimentInfo experimentInfo = PromoServiceBookingFlowUtils.buildExperimentInfo(experimentDataProvider, context.getHeaders());
                            return promoServiceClient.calculateDiscountForOffer(offerInfo, userInfo, experimentInfo)
                                    .thenApply(resp -> DiscountInfo.builder()
                                            .applied(resp.getDiscountInfo().getDiscountApplied())
                                            .discountedPrice(ProtoUtils.fromTPrice(resp.getDiscountInfo().getPriceAfterDiscount()))
                                            .discountAmount(params.getPriceFromPartnerOffer().subtract(ProtoUtils.fromTPrice(resp.getDiscountInfo().getPriceAfterDiscount())))
                                            .build());
                        } catch (Exception e) {
                            return CompletableFuture.failedFuture(e);
                        }
                    }
                });
    }

    private Money rateInfoOrNull(RateInfo rateInfo) {
        return rateInfo != null ? rateInfo.getTotalMoney() : null;
    }

    private MetaInfo buildMetaInfo(BookingFlowContext context, Offer offer) {
        EPartnerId partnerId = context.getDecodedToken().getPartnerId();
        var metaInfoBuilder = MetaInfo.builder()
                .search(SearchInfo.builder()
                        .checkIn(context.getDecodedToken().getCheckInDate())
                        .checkOut(context.getDecodedToken().getCheckOutDate())
                        .adults(context.getDecodedToken().getOccupancy() != null ?  // workaround for old tokens
                                context.getDecodedToken().getOccupancy().getAdults() : 0)
                        .children(context.getDecodedToken().getOccupancy() != null ?  // workaround for old tokens
                                context.getDecodedToken().getOccupancy().getChildren() : null)
                        .permalink(context.getDecodedToken().getPermalink())
                        .partnerId(partnerId)
                        .originalId(context.getDecodedToken().getOriginalId())
                        .build())
                .token(context.getToken())
                .label(context.getOfferLabel())
                .offerGeneratedAt(context.getDecodedToken().getGeneratedAt())
                .tokenId(context.getDecodedToken().getTokenId())
                .deduplicationKey(context.getDeduplicationKey())
                .checkSum(checksumService.buildCheckSum(offer));
        if (context.getOrderCreationData() != null && context.getStage() == BookingFlowContext.Stage.CREATE_ORDER) {
            metaInfoBuilder.selectedBedGroupIndex(context.getOrderCreationData().getSelectedBedGroupIndex());
        }
        return metaInfoBuilder.build();
    }

    private CompletableFuture<LegalInfo> combineLegalInfoItems(CompletableFuture<LegalInfo.LegalInfoItem> geoHotelFuture,
                                                               CompletableFuture<LegalInfo.LegalInfoItem> partnerHotelFuture,
                                                               CompletableFuture<LegalInfo.LegalInfoItem> partnerFuture,
                                                               BookingFlowContext context) {
        return CompletableFuture.allOf(geoHotelFuture, partnerHotelFuture, partnerFuture).thenApply(ignored -> {
            LegalInfo.LegalInfoItem geoHotelLegalInfo = geoHotelFuture.join();

            if (geoHotelLegalInfo == null) {
                throwGeosearchHotelNotFoundError(context);
            }
            LegalInfo.LegalInfoItem partnerHotelLegalInfo = partnerHotelFuture.join();
            return LegalInfo.builder()
                    .yandex(LegalInfo.LegalInfoItem.builder()
                            .name(offerConfig.getYandexLegalData().getName())
                            .legalAddress(offerConfig.getYandexLegalData().getAddress())
                            .ogrn(offerConfig.getYandexLegalData().getOgrn())
                            .workingHours(offerConfig.getYandexLegalData().getWorkingHours())
                            .build())
                    .partner(partnerFuture.join())
                    .hotel(mergeLegalInfoItems(geoHotelLegalInfo, partnerHotelLegalInfo))
                    .build();
        });
    }

    private LegalInfo.LegalInfoItem mergeLegalInfoItems(LegalInfo.LegalInfoItem geoHotelLegalInfo,
                                                        LegalInfo.LegalInfoItem partnerHotelLegalInfo) {
        if (partnerHotelLegalInfo == null) {
            return geoHotelLegalInfo;
        }
        LegalInfo.LegalInfoItem.LegalInfoItemBuilder legalInfoItemBuilder = LegalInfo.LegalInfoItem.builder()
                .name(partnerHotelLegalInfo.getName())
                .actualAddress(geoHotelLegalInfo.getActualAddress())
                .workingHours(partnerHotelLegalInfo.getWorkingHours())
                .legalAddress(partnerHotelLegalInfo.getLegalAddress());
        if (StringUtils.isNotBlank(partnerHotelLegalInfo.getOgrn())) {
            legalInfoItemBuilder.ogrn(partnerHotelLegalInfo.getOgrn());
        }
        return legalInfoItemBuilder.build();
    }

    private String getRequestId(BookingFlowContext context) {
        String reqId = "";
        if (context.getHeaders() != null) {
            reqId = context.getHeaders().getRequestId();
            reqId = reqId == null ? "" : reqId;
        }
        return reqId;
    }

    private CompletableFuture<TOfferData> fetchOfferData(TravelToken travelToken, BookingFlowContext.Stage stage, BookingFlowContext context) {
        String reqId = getRequestId(context);
        log.info("Fetching offer '{}' generated at {} (req_id {})",
                travelToken.getOfferId(), travelToken.getGeneratedAt().toString(), reqId);

        return storageSupplier.get()
                .thenCompose(storage -> storage.get(travelToken.getTokenId(), TOfferData.class))
                .whenComplete((r, t) -> {
                    if (t != null) {
                        log.error(String.format("Error while getting offer from storage (req_id {})", reqId), t);
                        meters.getOfferMeters(stage, getPartnerId()).getTokenStorageErrors().increment();
                    }
                })
                .thenApply(offer -> {
                    if (offer == null) {
                        LocalDateTime oldestTokenTime =
                                LocalDateTime.now(ZoneOffset.UTC).minus(offerConfig.getTokenTtl());
                        String reason;
                        if (travelToken.getGeneratedAt().isBefore(oldestTokenTime)) {
                            meters.getOfferMeters(stage, getPartnerId()).getExpiredToken().increment();
                            reason = "likely expired";
                        } else {
                            meters.getOfferMeters(stage, getPartnerId()).getMissingToken().increment();
                            reason = "this is unexpected: token is young enough";
                        }
                        log.error("Offer data is not found in storage, {} (req_id {})", reason, reqId);

                        throw new MissingOfferDataException();
                    } else {
                        long tokenTtl = Duration.between(travelToken.getGeneratedAt(),
                                LocalDateTime.now(ZoneOffset.UTC)).get(ChronoUnit.SECONDS);
                        meters.getOfferMeters(stage, getPartnerId()).getTtl().record(tokenTtl,
                                TimeUnit.SECONDS);
                        log.info("Offer data fetched successfully");
                        return offer;
                    }
                });
    }

    @Override
    public boolean isPostPayAllowed(TOfferData offerData, BookingFlowContext context, HotelInfo hotelInfo) {
        return false;
    }

    private <F> CompletableFuture<F> wrapFailureToNull(CompletableFuture<F> future, BookingFlowContext context,
                                                       BookingProviderMethod method) {
        final Counter bookingProviderErrorCounter = meters
                .getOfferMeters(context.getStage(), context.getProvider().getPartnerId())
                .getBookingProviderErrorCounter(method);
        return future.handle((r, t) -> {
            if (t != null) {
                if (t.getCause() != null && t.getCause() instanceof UnhealthyServiceException) {
                    throw (UnhealthyServiceException) t.getCause();
                }
                if (t.getCause() != null && t.getCause() instanceof MissingOfferDataException) {
                    log.error("Offer data is missing, will use 'null' instead of {} value", method.getMethodName());
                } else {
                    bookingProviderErrorCounter.increment();
                    log.error("Future '{}' completed with an error, will use null as its return value",
                            method.getMethodName(), t);
                }
                return null;
            } else {
                if (r == null) {
                    bookingProviderErrorCounter.increment();
                }
                return r;
            }
        });
    }

    private void throwGeosearchHotelNotFoundError(BookingFlowContext context) {
        // https://st.yandex-team.ru/TRAVELBACK-3769
        meters.getOfferMeters(
                context.getStage(), getPartnerId()).getGeosearchHotelNotFoundErrors().increment();
        throw new HotelInfoNotFoundException(
                context.getToken(), "Can't get hotel info from geosearch");
    }

    public EOrderType getOrderType() {
        return EOrderType.OT_HOTEL_EXPEDIA;
    }

    public abstract EServiceType getServiceType();

    abstract T getPartnerOffer(TOfferData offerData);

    abstract PartnerFutures getPartnerFutures(BookingFlowContext context, CompletableFuture<T> offerDataFuture);

    CompletableFuture<THotelTestContext> getTestContextFuture(CompletableFuture<TOfferData> offerDataFuture) {
        return offerDataFuture.thenApply(od -> od.hasTestContext() ? od.getTestContext() : null)
                .exceptionally(throwable -> null);
    }

    public enum BookingProviderMethod {
        HOTEL_INFO("HotelInfo"),
        PARTNER_HOTEL_INFO("PartnerHotelInfo"),
        RATE_INFO("RateInfo"),
        STAY_INFO("StayInfo"),
        ROOM_INFO("RoomInfo"),
        REFUND_RULES("RefundRules"),
        HOTEL_LEGAL("HotelLegal"),
        PARTNER_LEGAL("PartnerLegal"),
        DISCOUNT_INFO("DiscountInfo");

        private final String methodName;

        BookingProviderMethod(String methodName) {
            this.methodName = methodName;
        }

        public String getMethodName() {
            return methodName;
        }
    }
}
