package ru.yandex.travel.api.services.hotels_booking_flow;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.api.config.hotels.BNovoConfigurationProperties;
import ru.yandex.travel.api.services.hotels.legal_info.HotelLegalInfoDictionary;
import ru.yandex.travel.api.services.localization.LocalizationService;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.administrator.export.proto.HotelLegalInfo;
import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.bnovo.BNovoClient;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Hotel;
import ru.yandex.travel.hotels.common.partners.bnovo.model.HotelStayMap;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Offer;
import ru.yandex.travel.hotels.common.partners.bnovo.model.PriceLosRequest;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RatePlan;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RoomPhoto;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RoomType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Service;
import ru.yandex.travel.hotels.common.partners.bnovo.model.ServiceType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Stay;
import ru.yandex.travel.hotels.common.partners.bnovo.utils.BNovoPansionHelper;
import ru.yandex.travel.hotels.common.partners.bnovo.utils.BNovoRefundRulesBuilder;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.models.booking_flow.BreakdownType;
import ru.yandex.travel.hotels.models.booking_flow.Coordinates;
import ru.yandex.travel.hotels.models.booking_flow.HotelInfo;
import ru.yandex.travel.hotels.models.booking_flow.Image;
import ru.yandex.travel.hotels.models.booking_flow.LegalInfo;
import ru.yandex.travel.hotels.models.booking_flow.LocalizedPansionInfo;
import ru.yandex.travel.hotels.models.booking_flow.PartnerHotelInfo;
import ru.yandex.travel.hotels.models.booking_flow.Rate;
import ru.yandex.travel.hotels.models.booking_flow.RateInfo;
import ru.yandex.travel.hotels.models.booking_flow.RateStatus;
import ru.yandex.travel.hotels.models.booking_flow.RoomInfo;
import ru.yandex.travel.hotels.models.booking_flow.StayInfo;
import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TBNovoOffer;
import ru.yandex.travel.hotels.proto.TOfferData;
import ru.yandex.travel.orders.commons.proto.EServiceType;

@PartnerBean(EPartnerId.PI_BNOVO)
@RequiredArgsConstructor
@Slf4j
public class BNovoPartnerBookingProvider extends AbstractPartnerBookingProvider<TBNovoOffer> {
    private final BNovoClient bNovoClient;
    private final LocalizationService localizationService;
    private final BedInflector bedInflector;
    private final OfferServiceConfigurationProperties offerConfig;
    private final BNovoConfigurationProperties properties;
    private final HotelLegalInfoDictionary hotelLegalInfoDictionary;

    @Override
    public EServiceType getServiceType() {
        return EServiceType.PT_BNOVO_HOTEL;
    }

    @Override
    public EPartnerId getPartnerId() {
        return EPartnerId.PI_BNOVO;
    }

    @Override
    TBNovoOffer getPartnerOffer(TOfferData offerData) {
        return offerData.getBNovoOffer();
    }

    private PartnerHotelInfo mapHotelInfo(Hotel hotel) {
        return PartnerHotelInfo.builder()
                .name(hotel.getName())
                .address(hotel.getAddress())
                .coordinates(hotel.getGeoData() == null ? null :
                        new Coordinates(hotel.getGeoData().getLongitude(), hotel.getGeoData().getLatitude()))
                .phone(hotel.getPhone())
                .build();
    }

    private Image mapImage(RoomPhoto photo) {
        return Image.builder()
                .l(photo.getUrl())
                .m(photo.getUrl())
                .s(photo.getThumb())
                .build();
    }

    private CompletableFuture<CallContext> getCallContextFuture(BookingFlowContext flowContext,
                                                                CompletableFuture<TBNovoOffer> offerFuture) {
        return CompletableFuture.allOf(flowContext.getTestContextFuture(), offerFuture)
                .thenApply(ignored ->
                        new CallContext(flowContext.getStage() == BookingFlowContext.Stage.GET_OFFER ?
                                CallContext.CallPhase.OFFER_VALIDATION : CallContext.CallPhase.ORDER_CREATION,
                                flowContext.getTestContextFuture().join(),
                                offerFuture.join(), flowContext.getDecodedToken(), null, null));
    }

    private RoomInfo mapRoomInfo(RoomType roomType, RatePlan ratePlan, Map<Long, Service> serviceMap) {
        EPansionType pansionType = BNovoPansionHelper.getPansionType(ratePlan, serviceMap);
        String roomName = roomType.getDefaultName();
        if (properties.isDebugShowRateInfo() && ratePlan.getId() != 0) {
            // 0 is id of test-context ratePlan, this is used in FE tests
            roomName += " [" + ratePlan.getId() + "] '" + ratePlan.getName() + "'";
        }
        return RoomInfo.builder()
                .name(roomName)
                .description(roomType.getDefaultDescription())
                .roomAmenities(Collections.emptyList())
                .images(roomType.getPhotos() == null ? Collections.emptyList() :
                        roomType.getPhotos().stream().map(this::mapImage).collect(Collectors.toList()))
                .pansionInfo(new LocalizedPansionInfo(pansionType,
                        localizationService.localizePansion(pansionType, "ru")))
                .build();
    }

    private Stay getBNovoStay(TBNovoOffer offerData) {
        return ProtoUtils.fromTJson(offerData.getBNovoStay(), Stay.class);
    }

    private RateInfo mapRateInfo(long accountId, Stay searchedStay, HotelStayMap response) {
        Preconditions.checkArgument(searchedStay.getRates().size() == 1,
                "Unexpected amount of offers in searched stay");
        long roomTypeId = searchedStay.getRates().get(0).getRoomtypeId();
        long ratePlanId = searchedStay.getRates().get(0).getPlanId();
        BigDecimal expectedPrice = searchedStay.getRates().get(0).getPrice();
        Stay confirmedStay = response.get(String.valueOf(accountId)).get(searchedStay.getCheckin());
        Offer confirmedStayOffer = confirmedStay.getRates().stream()
                .filter(offer -> offer.getRoomtypeId() == roomTypeId && offer.getPlanId() == ratePlanId)
                .collect(Collectors.collectingAndThen(Collectors.toList(), list -> {
                    if (list.size() != 1) {
                        throw new IllegalStateException("Unexpected amount of offers in verified stay");
                    }
                    return list.get(0);
                }));
        BigDecimal actualPrice = confirmedStayOffer.getPrice();
        RateStatus status = RateStatus.fromComparison(expectedPrice, actualPrice);
        Preconditions.checkState(status != null);
        return RateInfo.builder()
                .baseRate(new Rate(actualPrice.setScale(2, RoundingMode.HALF_UP).toString(), "RUB"))
                .breakdownType(BreakdownType.NIGHT)
                .baseRateBreakdown(confirmedStayOffer.getPricesByDates().entrySet().stream()
                        .sorted(Map.Entry.comparingByKey())
                        .map(e -> new Rate(e.getValue().setScale(2, RoundingMode.HALF_UP).toString(), "RUB"))
                        .collect(Collectors.toList()))
                .totalRate(new Rate(actualPrice.setScale(2, RoundingMode.HALF_UP).toString(), "RUB"))
                .status(status)
                .build();
    }

    private StayInfo mapStayInfo(Hotel hotel, RatePlan ratePlan, Map<Long, Service> serviceMap) {
        var includedNonMealServices = ratePlan.getAdditionalServicesIds().stream()
                .map(serviceMap::get)
                .filter(s -> s != null && s.getType() != ServiceType.BOARD)
                .collect(Collectors.toList());
        String stayInstructions = null;
        if (!includedNonMealServices.isEmpty()) {
            StringBuilder builder = new StringBuilder();
            builder.append("<p>В стоимость включены следующие услуги:")
                    .append("<ul>");
            includedNonMealServices.forEach(s -> {
                builder.append("<li><p>");
                builder.append("<b>").append(s.getDefaultName()).append("</b><br/>");
                if (StringUtils.isNotBlank(s.getDefaultDescription())) {
                    builder.append(s.getDefaultDescription().replaceAll("\\n", "<br/>"));
                }
                builder.append("</p></li>");
            });
            builder.append("</ul></p>");
            stayInstructions = builder.toString();
        }
        return StayInfo.builder()
                .checkInStartTime(hotel.getCheckin())
                .checkOutEndTime(hotel.getCheckout())
                .stayInstruction(stayInstructions)
                .build();
    }

    private RefundRules mapRefundRules(RatePlan ratePlan, Stay stay, Instant checkinMoment, Instant creationMoment) {
        Preconditions.checkState(stay.getRates().size() == 1,
                "Unexpected amount of offers in verified stay");
        return BNovoRefundRulesBuilder.build(ratePlan, stay.getRates().get(0), checkinMoment,
                creationMoment,
                "RUB");
    }

    private RoomType getParentRoomType(Map<Long, RoomType> map, long id) {
        RoomType rt = map.get(id);
        while (rt.getParentId() != 0) {
            rt = map.get(rt.getParentId());
        }
        return rt;
    }

    @Override
    PartnerFutures getPartnerFutures(BookingFlowContext context, CompletableFuture<TBNovoOffer> offerDataFuture) {
        long accountId = Long.parseLong(context.getDecodedToken().getOriginalId());
        String httpRequestId = ProtoUtils.randomId();
        CompletableFuture<Hotel> hotelInfoFuture = offerDataFuture.thenCompose(od ->
                getCallContextFuture(context, offerDataFuture).thenCompose(callContext -> {
                    if (od.hasUIDMapping()) {
                        return bNovoClient.withCallContext(callContext).getHotelByUID(od.getUIDMapping().getUID());
                    } else {
                        return bNovoClient.withCallContext(callContext).getHotelByAccountId(accountId, httpRequestId);
                    }
                }));
        CompletableFuture<Map<Long, RoomType>> roomTypeFuture =
                getCallContextFuture(context, offerDataFuture)
                        .thenCompose(callContext ->
                                bNovoClient.withCallContext(callContext).getRoomTypes(accountId, httpRequestId));

        CompletableFuture<Map<Long, RatePlan>> ratePlanFuture =
                getCallContextFuture(context, offerDataFuture)
                        .thenCompose(callContext ->
                                bNovoClient.withCallContext(callContext).getRatePlans(accountId, httpRequestId));

        CompletableFuture<Map<Long, Service>> serviceFuture =
                getCallContextFuture(context, offerDataFuture)
                        .thenCompose(callContext ->
                                bNovoClient.withCallContext(callContext).getServices(accountId, httpRequestId));


        CompletableFuture<Stay> stayFuture = offerDataFuture.thenApply(this::getBNovoStay);
        CompletableFuture<HotelStayMap> priceFuture = getCallContextFuture(context, offerDataFuture)
                .thenCompose(callContext -> bNovoClient.withCallContext(callContext).getPrices(
                        PriceLosRequest.builder()
                                .account(accountId)
                                .adults(context.getDecodedToken().getOccupancy().getAdults())
                                .children(context.getDecodedToken().getOccupancy().getChildren().size())
                                .checkinFrom(context.getDecodedToken().getCheckInDate())
                                .nights((int) ChronoUnit.DAYS.between(
                                        context.getDecodedToken().getCheckInDate(),
                                        context.getDecodedToken().getCheckOutDate()))
                                .build()
                ));

        return new PartnerFutures() {
            @Override
            public CompletableFuture<PartnerHotelInfo> getPartnerHotelInfo() {
                return hotelInfoFuture.thenApply(h -> mapHotelInfo(h));
            }

            @Override
            public CompletableFuture<RateInfo> getRateInfo() {
                return stayFuture.thenCombine(priceFuture, (stay, price) -> mapRateInfo(accountId, stay, price));
            }

            @Override
            public CompletableFuture<RoomInfo> getRoomInfo() {
                return CompletableFuture.allOf(roomTypeFuture, ratePlanFuture, stayFuture, serviceFuture)
                        .thenApply(ignored -> {
                            Stay stay = stayFuture.join();
                            Preconditions.checkState(stay.getRates().size() == 1);
                            long roomTypeId = stay.getRates().get(0).getRoomtypeId();
                            long ratePlanId = stay.getRates().get(0).getPlanId();
                            return mapRoomInfo(
                                    getParentRoomType(roomTypeFuture.join(), roomTypeId),
                                    ratePlanFuture.join().get(ratePlanId), serviceFuture.join());
                        });
            }

            @Override
            public CompletableFuture<StayInfo> getStayInfo() {
                return CompletableFuture.allOf(hotelInfoFuture, ratePlanFuture, stayFuture).thenApply(ignored -> {
                    Stay stay = stayFuture.join();
                    return mapStayInfo(hotelInfoFuture.join(),
                            ratePlanFuture.join().get(stay.getRates().get(0).getPlanId()), serviceFuture.join());
                });
            }

            @Override
            public CompletableFuture<RefundRules> getRefundRules() {
                return CompletableFuture.allOf(ratePlanFuture, stayFuture, hotelInfoFuture).thenApply(ignored -> {
                            Stay stay = stayFuture.join();
                            Preconditions.checkState(stay.getRates().size() == 1);
                            long ratePlanId = stay.getRates().get(0).getPlanId();
                            return mapRefundRules(ratePlanFuture.join().get(ratePlanId),
                                    stay, hotelInfoFuture.join().getCheckinInstantForDate(
                                            context.getDecodedToken().getCheckInDate()), context.getCreatedAt());
                        }
                );
            }

            @Override
            public CompletableFuture<LegalInfo.LegalInfoItem> getPartnerLegalInfoItem() {
                return CompletableFuture.completedFuture(null);
            }

            @Override
            public CompletableFuture<LegalInfo.LegalInfoItem> getHotelLegalInfoItem() {
                HotelLegalInfo hotelLegalInfo = hotelLegalInfoDictionary.getHotelLegalInfo(
                        context.getDecodedToken().getPartnerId(),
                        context.getDecodedToken().getOriginalId());
                if (hotelLegalInfo == null) {
                    meters.incrementHotelLegalDataMissing(getPartnerId());
                    return CompletableFuture.failedFuture(new RuntimeException(
                            String.format("HotelLegalInfo is not found for Partner: %s, HotelCode: %s", getPartnerId(),
                                    context.getDecodedToken().getOriginalId())));
                }
                return CompletableFuture.completedFuture(LegalInfo.LegalInfoItem.builder()
                        .name(hotelLegalInfo.getLegalName())
                        .workingHours(hotelLegalInfo.getWorkingHours())
                        .legalAddress(hotelLegalInfo.getLegalAddress())
                        .ogrn(hotelLegalInfo.getOgrn())
                        .build());
            }

            @Override
            public CompletableFuture<HotelItinerary> createHotelItinerary() {
                return offerDataFuture.thenApply(od -> {
                    Stay stay = getBNovoStay(od);
                    BNovoHotelItinerary itinerary = new BNovoHotelItinerary();
                    itinerary.setAccountId(accountId);
                    itinerary.setBNovoStay(stay);
                    itinerary.setOccupancy(context.getDecodedToken().getOccupancy());
                    Preconditions.checkState(stay.getRates().size() == 1, "Unexpected number of items in stay");
                    Preconditions.checkState(stay.getRates().get(0).getPrice().compareTo(BigDecimal.ZERO) > 0,
                            "Unexpected offer price");
                    itinerary.setFiscalPrice(Money.of(stay.getRates().get(0).getPrice().setScale(2,
                            RoundingMode.HALF_UP), "RUB"));
                    itinerary.setYandexNumberPrefix(Strings.nullToEmpty(properties.getYandexNumberPrefix()));
                    return itinerary;
                });
            }
        };
    }

    @Override
    public boolean isPostPayAllowed(TOfferData offerData, BookingFlowContext context,
                                    HotelInfo hotelInfo) {
        var testContext = FutureUtils.joinCompleted(context.getTestContextFuture());
        if (testContext != null && testContext.getIsPostPay()) {
            return true;
        }
        return offerData != null && offerData.hasBNovoOffer() && offerData.getBNovoOffer().getPostPayAllowed();
    }
}
