package ru.yandex.travel.api.services.hotels_booking_flow

import com.google.common.base.Preconditions
import org.javamoney.moneta.Money
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.travel.api.config.hotels.BronevikConfigurationProperties
import ru.yandex.travel.api.services.hotels_booking_flow.BookingFlowContext.Stage.GET_OFFER
import ru.yandex.travel.api.services.localization.LocalizationService
import ru.yandex.travel.commons.proto.ECurrency
import ru.yandex.travel.hotels.common.orders.BronevikHotelItinerary
import ru.yandex.travel.hotels.common.orders.HotelItinerary
import ru.yandex.travel.hotels.common.orders.MealData
import ru.yandex.travel.hotels.common.partners.base.CallContext
import ru.yandex.travel.hotels.common.partners.base.CallContext.CallPhase.OFFER_VALIDATION
import ru.yandex.travel.hotels.common.partners.base.CallContext.CallPhase.ORDER_CREATION
import ru.yandex.travel.hotels.common.partners.bronevik.AvailableAmenities
import ru.yandex.travel.hotels.common.partners.bronevik.BedSet
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikClient
import ru.yandex.travel.hotels.common.partners.bronevik.HotelOffer
import ru.yandex.travel.hotels.common.partners.bronevik.HotelWithInfo
import ru.yandex.travel.hotels.common.partners.bronevik.HotelWithOffers
import ru.yandex.travel.hotels.common.partners.bronevik.SearchHotelOffersResponse
import ru.yandex.travel.hotels.common.partners.bronevik.Tax
import ru.yandex.travel.hotels.common.partners.bronevik.model.BedType
import ru.yandex.travel.hotels.common.partners.bronevik.model.TaxType
import ru.yandex.travel.hotels.common.partners.bronevik.utils.BronevikUtils
import ru.yandex.travel.hotels.common.refunds.RefundRules
import ru.yandex.travel.hotels.models.booking_flow.Amenity
import ru.yandex.travel.hotels.models.booking_flow.BedGroupInfo
import ru.yandex.travel.hotels.models.booking_flow.BreakdownType
import ru.yandex.travel.hotels.models.booking_flow.Coordinates
import ru.yandex.travel.hotels.models.booking_flow.ExtraFee
import ru.yandex.travel.hotels.models.booking_flow.ExtraFeeType
import ru.yandex.travel.hotels.models.booking_flow.Image
import ru.yandex.travel.hotels.models.booking_flow.LegalInfo.LegalInfoItem
import ru.yandex.travel.hotels.models.booking_flow.LocalizedPansionInfo
import ru.yandex.travel.hotels.models.booking_flow.PartnerHotelInfo
import ru.yandex.travel.hotels.models.booking_flow.Rate
import ru.yandex.travel.hotels.models.booking_flow.RateInfo
import ru.yandex.travel.hotels.models.booking_flow.RateStatus
import ru.yandex.travel.hotels.models.booking_flow.RateStatus.CONFIRMED
import ru.yandex.travel.hotels.models.booking_flow.RateStatus.SOLD_OUT
import ru.yandex.travel.hotels.models.booking_flow.RoomInfo
import ru.yandex.travel.hotels.models.booking_flow.StayInfo
import ru.yandex.travel.hotels.proto.EPartnerId.PI_BRONEVIK
import ru.yandex.travel.hotels.proto.TBronevikOffer
import ru.yandex.travel.hotels.proto.TOfferData
import ru.yandex.travel.orders.commons.proto.EServiceType.PT_BRONEVIK_HOTEL
import ru.yandex.travel.orders.commons.proto.EVat
import java.math.BigDecimal
import java.util.UUID
import java.util.concurrent.CompletableFuture
import javax.xml.datatype.XMLGregorianCalendar
import kotlin.math.roundToInt

@PartnerBean(PI_BRONEVIK)
open class BronevikPartnerBookingProvider(
    @Autowired private val properties: BronevikConfigurationProperties,
    @Autowired private val bronevikClient: BronevikClient,
    @Autowired private val localizationService: LocalizationService,
) : AbstractPartnerBookingProvider<TBronevikOffer>() {
    private val log = LoggerFactory.getLogger(this.javaClass)
    private val currencyMap = mapOf(
        ECurrency.C_RUB to "RUB"
    )

    override fun getServiceType() = PT_BRONEVIK_HOTEL

    override fun getPartnerId() = PI_BRONEVIK

    override fun getPartnerOffer(offerData: TOfferData): TBronevikOffer {
        Preconditions.checkArgument(offerData.hasBronevikOffer())

        return offerData.bronevikOffer
    }

    override fun getPartnerFutures(
        context: BookingFlowContext,
        offerDataFuture: CompletableFuture<TBronevikOffer>
    ): PartnerFutures {
        val requestId = generateReqId()
        val clientFuture = getBronevikClient(context, offerDataFuture)
        val hotelInfoFuture = clientFuture.thenCompose { client ->
            offerDataFuture.thenCompose {
                client.getHotelInfo(it.hotelId, generateReqId(requestId))
                    .thenApply { response -> checkNotNull(response) { "Hotel with id ${it.hotelId} not found" } }
            }
        }
        val hotelOffersFuture = clientFuture.thenCompose { client ->
            offerDataFuture.thenCompose {
                val currency = currencyMap[it.currency]
                client.searchHotelOffers(
                    it.occupancy, it.childrenAgeList, listOf(it.hotelId), it.checkin, it.checkout, currency,
                    generateReqId(requestId)
                )
            }
        }
        val amenitiesFuture = clientFuture.thenCompose { client -> client.getAmenities(generateReqId(requestId)) }
            .thenApply { response -> response.amenities.amenity.associate { it.id to it.name } }

        return object : PartnerFutures {
            override fun getPartnerHotelInfo(): CompletableFuture<PartnerHotelInfo> {
                if (!properties.enabled) {
                    CompletableFuture.completedFuture(null)
                }

                return CompletableFuture.allOf(hotelInfoFuture, clientFuture, amenitiesFuture).thenApply {
                    mapPartnerHotelInfo(hotelInfoFuture.join(), amenitiesFuture.join())
                }
            }

            override fun getRateInfo(): CompletableFuture<RateInfo> {
                return offerDataFuture.thenCompose { offerData ->
                    hotelOffersFuture.thenApply { offers -> mapRateInfo(offerData, offers) }
                }
            }

            override fun getRoomInfo(): CompletableFuture<RoomInfo> {
                return CompletableFuture.allOf(offerDataFuture, hotelInfoFuture, hotelOffersFuture, amenitiesFuture)
                    .thenApply {
                        mapRoomInfo(
                            offerDataFuture.join(),
                            hotelInfoFuture.join(),
                            hotelOffersFuture.join(),
                            amenitiesFuture.join()
                        )
                    }
            }

            override fun getStayInfo(): CompletableFuture<StayInfo> {
                return hotelOffersFuture.thenApply { offers -> mapStayInfo(offers) }
            }

            override fun getRefundRules(): CompletableFuture<RefundRules> {
                return offerDataFuture.thenCompose { offerData ->
                    hotelOffersFuture.thenApply { offers -> mapRefundRules(offerData, offers) }
                }
            }

            override fun getPartnerLegalInfoItem(): CompletableFuture<LegalInfoItem> {
                return CompletableFuture.completedFuture(mapPartnerLegalInfoItem())
            }

            override fun getHotelLegalInfoItem(): CompletableFuture<LegalInfoItem> {
                return CompletableFuture.completedFuture(null)
            }

            override fun createHotelItinerary(): CompletableFuture<HotelItinerary> {
                return CompletableFuture.allOf(offerDataFuture).thenApply {
                    val offerData = offerDataFuture.join()
                    val nights = BronevikUtils.getNights(offerData.checkin, offerData.checkout)
                    val meals = offerData.mealsList.filter { !it.included }
                    val mealData = MealData.builder().items(
                        meals.map {
                            return@map MealData.MealItem.builder()
                                .mealPrice(
                                    Money.of(
                                        it.price * offerData.occupancy * nights,
                                        currencyMap[offerData.currency]
                                    )
                                )
                                .mealVat(getEVat(it.vatPercent))
                                .mealName(BronevikUtils.getMealName(it.id, properties.client.soapType))
                                .build()
                        }
                    ).build()
                    BronevikHotelItinerary.builder()
                        .hotelId(offerData.hotelId)
                        .offerCode(offerData.offerCode)
                        .meals(meals.map { it.id })
                        .mealData(mealData)
                        .currency(currencyMap[offerData.currency])
                        .fiscalPrice(Money.of(offerData.price, currencyMap[offerData.currency]))
                        .yandexNumberPrefix(properties.yandexNumberPrefix ?: "")
                        .build()
                }
            }
        }
    }

    private fun mapRateInfo(offerData: TBronevikOffer, hotelOffers: SearchHotelOffersResponse): RateInfo {
        val builder = RateInfo.builder()
        val actualOfferInfo = hotelOffers.getOffer(offerData.offerCode)
        val hotel = hotelOffers.getHotel()
        if (actualOfferInfo.freeRooms <= 0) {
            return builder.status(SOLD_OUT).build()
        }

        val actualMealMap = actualOfferInfo.meals.meal.associateBy { it.id }
        offerData.mealsList.forEach {
            val actualMeal = actualMealMap[it.id] ?: return soldOut(builder, "Meal id ${it.id} not found in actual partner offer ${offerData.offerCode}")
            if (actualMeal.isIncluded != it.included || BronevikUtils.getPriceLong(actualMeal.priceDetails) != it.price) {
                return soldOut(builder, "Meal id ${it.id} has been changed on partner side, offer ${offerData.offerCode}")
            }
        }
        val mealIds = offerData.mealsList.map { it.id }.toSet()
        val actualMeals = actualOfferInfo.meals
        val nights = BronevikUtils.getNights(offerData.checkin, offerData.checkout)
        actualMeals.meal = actualMeals.meal.filter { mealIds.contains(it.id) }
        val totalPrice = BronevikUtils.getTotalPrice(actualOfferInfo.priceDetails, actualMeals, offerData.occupancy, nights)
        val status = RateStatus.fromComparison(
            BigDecimal.valueOf(offerData.price, 2),
            BigDecimal.valueOf(totalPrice.toLong(), 2)
        )

        if (status != CONFIRMED) {
            log.info("Price mismatch on offer validation: searcher price is ${offerData.price}, verified price is $totalPrice")
        }

        val taxes = hotel.taxes?.tax ?: listOf()

        builder
            .baseRate(Rate(totalPrice.toString(), "RUB"))
            .totalRate(Rate(totalPrice.toString(), "RUB"))
            .breakdownType(BreakdownType.NIGHT)
            .baseRateBreakdown(actualOfferInfo.dailyPrices.dailyPrice.map { dailyPrice ->
                Rate(dailyPrice.rate.clientCurrency.gross.price.roundToInt().toString(), "RUB")
            })
            .extraFees(taxes.filter { it.isIncluded }
                .map { tax -> ExtraFee(tax.amount.roundToInt().toString(), tax.currency, tax.getExtraFeeType()) }
            )
            .status(status)

        return builder.build()
    }

    private fun mapPartnerHotelInfo(hotelInfo: HotelWithInfo, amenityNames: Map<Int, String>): PartnerHotelInfo {
        val builder = PartnerHotelInfo.builder()
            .name(hotelInfo.name)
            .address(hotelInfo.address)
            .amenities(hotelInfo.descriptionDetails.availableAmenities.toAmenities(amenityNames))
            .images(hotelInfo.descriptionDetails.photos.photo.map { photo ->
                Image.builder()
                    .s(photo.url)
                    .m(photo.url)
                    .l(photo.url)
                    .build()
            })
            .description(PartnerHotelInfo.TextBlock("Описание", hotelInfo.descriptionDetails.description))

        if (hotelInfo.descriptionDetails.latitude != null && hotelInfo.descriptionDetails.longitude != null) {
            builder.coordinates(
                Coordinates(
                    hotelInfo.descriptionDetails.longitude.toDouble(),
                    hotelInfo.descriptionDetails.latitude.toDouble()
                )
            )
        }
        hotelInfo.category?.let { builder.stars(hotelInfo.category) }

        return builder.build()
    }

    private fun mapRoomInfo(
        offerData: TBronevikOffer, hotelInfo: HotelWithInfo,
        hotelOffers: SearchHotelOffersResponse, amenityNames: Map<Int, String>
    ): RoomInfo {
        val roomInfo = hotelInfo.rooms.room.find { it.id == offerData.roomId }

        checkNotNull(roomInfo) { "Unknown room with id ${offerData.roomId}" }

        val actualOfferInfo = hotelOffers.getOffer(offerData.offerCode)
        val pansionType = BronevikUtils.parseMeals(actualOfferInfo.meals, properties.client.soapType)
        val pansionName = localizationService.localizePansion(pansionType, "ru")

        return RoomInfo.builder()
            .name(roomInfo.name)
            .description(roomInfo.description)
            .images(roomInfo.photos.photo.map {
                Image.builder()
                    .s(it.url)
                    .m(it.url)
                    .l(it.url)
                    .build()
            })
            .bedGroups(mapBedGroups(roomInfo.availableBedSets.bedSet))
            .pansionInfo(LocalizedPansionInfo(pansionType, pansionName))
            .roomAmenities(roomInfo.availableAmenities.toAmenities(amenityNames))
            .build()
    }

    private fun mapPartnerLegalInfoItem(): LegalInfoItem {
        return LegalInfoItem.builder()
            .name(properties.partnerLegalData.name)
            .ogrn(properties.partnerLegalData.ogrn)
            .legalAddress(properties.partnerLegalData.address)
            .workingHours(properties.partnerLegalData.workingHours)
            .build()
    }

    private fun mapStayInfo(hotelOffers: SearchHotelOffersResponse): StayInfo {
        if (hotelOffers.hotels.hotel.isEmpty()) throw IllegalStateException("Hotel list is empty")

        val hotel = hotelOffers.hotels.hotel[0]

        val stayInstructions = hotel.informationForGuest.notification
            .map { it.value }

        return StayInfo.builder()
            .checkInStartTime(hotel.checkinTime.formatToTime())
            .checkOutEndTime(hotel.checkoutTime.formatToTime())
            .stayInstructions(stayInstructions)
            .build()
    }

    private fun mapRefundRules(offerData: TBronevikOffer, hotelOffers: SearchHotelOffersResponse): RefundRules {
        val actualOfferInfo = hotelOffers.getOffer(offerData.offerCode)
        val offerPrice = BronevikUtils.getPriceInt(actualOfferInfo.priceDetails.client)

        return BronevikUtils.parseRefundRules(
            actualOfferInfo.cancellationPolicies,
            offerPrice
        )
    }

    private fun generateReqId(parentReqId: String? = null): String {
        if (parentReqId == null) {
            return UUID.randomUUID().toString()
        }

        return "$parentReqId/${UUID.randomUUID()}"
    }

    private fun SearchHotelOffersResponse.getOffer(code: String): HotelOffer {
        if (hotels.hotel.isEmpty()) throw IllegalStateException("Hotel list is empty")

        val actualOffer = hotels.hotel[0].offers?.offer?.find { offer -> code == offer.code }

        checkNotNull(actualOffer) { "Unknown offer with id $code" }

        return actualOffer
    }

    private fun SearchHotelOffersResponse.getHotel(): HotelWithOffers {
        if (hotels.hotel.isEmpty()) throw IllegalStateException("Hotel list is empty")
        return hotels.hotel[0]
    }

    private fun Tax.getExtraFeeType(): ExtraFeeType {
        return when (type) {
            TaxType.DEPOSIT.value -> ExtraFeeType.DEPOSIT
            TaxType.RESORT_FEE.value -> ExtraFeeType.RESORT_FEE
            else -> ExtraFeeType.OTHER_FEE
        }
    }

    private fun XMLGregorianCalendar.formatToTime(): String {
        val hours = if (hour < 10) "0$hour" else "$hour"
        val minutes = if (minute < 10) "0$minute" else "$minute"
        val seconds = if (second < 10) "0$second" else "$second"

        return "$hours:$minutes:$seconds"
    }

    private fun mapBedGroups(bedSets: List<BedSet>): List<BedGroupInfo> {
        val description = bedSets.map { bedSet ->
            bedSet.bed
                .map BedMap@{ bed ->
                    if (bed.amount <= 0) {
                        return@BedMap null
                    }

                    when (bed.type) {
                        BedType.SINGLE.value -> makePluralForm(
                            bed.amount,
                            "односпальная кровать",
                            "односпальные кровати",
                            "односпальных кроватей"
                        )
                        BedType.DOUBLE.value -> makePluralForm(
                            bed.amount,
                            "двуспальная кровать",
                            "двуспальные кровати",
                            "двуспальных кроватей"
                        )
                        BedType.SOFA.value -> makePluralForm(bed.amount, "диван", "дивана", "диванов")
                        BedType.BUNK_BED.value -> makePluralForm(
                            bed.amount,
                            "двухъярусная кровать",
                            "двухъярусные кровати",
                            "двухъярусных кроватей"
                        )
                        else -> null
                    }
                }
                .filterNotNull()
                .joinToString()
        }.joinToString { " или " }

        return listOf(BedGroupInfo(0, description))
    }

    private fun makePluralForm(amount: Int, form1: String, form2: String, form3: String): String {
        val doubleDigits = amount % 100
        val digits = amount % 10

        return when {
            doubleDigits in 11..19 -> "$amount $form3"
            digits in 2..4 -> "$amount $form2"
            digits == 1 -> "$amount $form1"
            else -> "$amount $form3"
        }
    }

    private fun getBronevikClient(
        flowContext: BookingFlowContext,
        offerDataFuture: CompletableFuture<TBronevikOffer>
    ): CompletableFuture<BronevikClient> {
        return CompletableFuture.allOf(BookingFlowContextWrapper.getTestContextFuture(flowContext), offerDataFuture)
            .thenApply {
                val offerData = offerDataFuture.join()
                val testContext = BookingFlowContextWrapper.getTestContextFuture(flowContext).join()

                bronevikClient.withCallContext(
                    CallContext(
                        phase = if (BookingFlowContextWrapper.getStage(flowContext) == GET_OFFER) OFFER_VALIDATION else ORDER_CREATION,
                        testContext = testContext,
                        offerData = offerData,
                        travelToken = BookingFlowContextWrapper.getDecodedToken(flowContext)
                    )
                )
            }
    }

    private fun getEVat(vat: Int): EVat {
        return when (vat) {
            0 -> EVat.VAT_0
            10 -> EVat.VAT_10_110
            18 -> EVat.VAT_18_118
            20 -> EVat.VAT_20_120
            else -> EVat.UNRECOGNIZED
        }
    }

    private fun AvailableAmenities.toAmenities(amenityNames: Map<Int, String>): List<Amenity> {
        return availableAmenity
            .filter { it.isIncluded }
            .mapNotNull {
                if (amenityNames.containsKey(it.id))
                    Amenity(it.id.toString(), amenityNames[it.id])
                else null
            }
    }

    private fun soldOut(builder: RateInfo.RateInfoBuilder, reason: String?): RateInfo {
        if (!reason.isNullOrBlank()) {
            log.info("Bronevik sold out. $reason")
        }

        return builder.status(SOLD_OUT).build()
    }
}
