package ru.yandex.travel.api.services.hotels_booking_flow;

import java.io.ByteArrayOutputStream;
import java.io.ObjectOutputStream;
import java.security.MessageDigest;

import com.google.common.io.BaseEncoding;
import lombok.SneakyThrows;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.models.booking_flow.Offer;
import ru.yandex.travel.hotels.models.booking_flow.promo.Taxi2020PromoCampaign;

@Service
public class ChecksumService {

    /**
     * Builds a checksum out of "significant data". If any piece of the significant data changes, the offer will expire.
     * This is needed to protect from very rare (but yet possible) case when something changes between getOffer and
     * creteOrder calls, either in our GeoSearch database or in the partner's.
     * <p>
     * "Significant data" includes the following offer attributes:
     * - Hotel name and address (both from our and partner's points of view)
     * - Room name and meal options
     * - Total sum payable (it should also be detected by partner's price-checking logic, but it may be not available
     * for some partners)
     * - Cancellation policies
     * <p>
     * Note that offer parameters such as checkin/checkout dates, occupancy etc can't change even in theory since
     * they are coded in the token
     */
    @SneakyThrows
    public String buildCheckSum(Offer offer) {
        if (offer.getHotelInfo() == null
                || offer.getPartnerHotelInfo() == null
                || offer.getRoomInfo() == null
                || offer.getRateInfo() == null
                || offer.getRateInfo().getTotalRate() == null
                || offer.getRefundRules() == null) {
            return null;
        }
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        ObjectOutputStream oos = new ObjectOutputStream(buffer);
        oos.writeUTF(offer.getHotelInfo().getName());
        oos.writeUTF(offer.getHotelInfo().getAddress());
        oos.writeUTF(offer.getPartnerHotelInfo().getName());
        oos.writeUTF(offer.getPartnerHotelInfo().getAddress());
        oos.writeUTF(offer.getRoomInfo().getName());
        oos.write(offer.getRoomInfo().getPansionInfo().getId().getNumber());
        oos.writeUTF(offer.getRateInfo().getTotalRate().getAmount());
        oos.writeUTF(offer.getRateInfo().getTotalRate().getCurrency());
        oos.writeBoolean(offer.getRefundRules().isRefundable());
        if (offer.getRefundRules().isRefundable()) {
            for (var rule : offer.getRefundRules().getRules()) {
                if (rule.getStartsAt() != null) {
                    oos.writeObject(rule.getStartsAt());
                }
                if (rule.getEndsAt() != null) {
                    oos.writeObject(rule.getEndsAt());
                }
                if (rule.getPenalty() != null) {
                    oos.writeDouble(rule.getPenalty().getNumber().doubleValue());
                    oos.writeUTF(rule.getPenalty().getCurrency().getCurrencyCode());
                }
                oos.write(rule.getType().ordinal());
            }
        }
        if (offer.getPromoCampaignsInfo() != null) {
            Taxi2020PromoCampaign taxi2020Promo = offer.getPromoCampaignsInfo().getTaxi2020();
            if (taxi2020Promo != null && taxi2020Promo.isEligible()) {
                // to not trigger 'offer expired' behaviour for not eligible offers when the promo has ended and disabled
                // we write the typified flag only when it's set to true
                oos.writeObject(taxi2020Promo);
            }
        }
        if (offer.getDeferredPaymentSchedule() != null) {
            oos.write(offer.getDeferredPaymentSchedule().toProto().toByteArray());
        }
        oos.flush();
        MessageDigest digest = MessageDigest.getInstance("MD5");
        digest.update(buffer.toByteArray());
        return BaseEncoding.base64Url().encode(digest.digest());
    }

}
