package ru.yandex.travel.api.services.hotels_booking_flow;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;
import org.javamoney.moneta.Money;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.api.config.hotels.DolphinConfigurationProperties;
import ru.yandex.travel.api.services.localization.LocalizationService;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.health.HealthCheckedSupplier;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.UpdatableDataHolder;
import ru.yandex.travel.hotels.common.orders.DolphinHotelItinerary;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.base.exceptions.PartnerException;
import ru.yandex.travel.hotels.common.partners.dolphin.DolphinClient;
import ru.yandex.travel.hotels.common.partners.dolphin.exceptions.SoldOutException;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Area;
import ru.yandex.travel.hotels.common.partners.dolphin.model.AreaType;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CalculateOrderRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CalculateOrderResponse;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CheckinServiceModeEnum;
import ru.yandex.travel.hotels.common.partners.dolphin.model.FixedPercentFeeRefundParams;
import ru.yandex.travel.hotels.common.partners.dolphin.model.HotelContent;
import ru.yandex.travel.hotels.common.partners.dolphin.model.IdNameMap;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Offer;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Pansion;
import ru.yandex.travel.hotels.common.partners.dolphin.model.PriceKey;
import ru.yandex.travel.hotels.common.partners.dolphin.model.TourContent;
import ru.yandex.travel.hotels.common.partners.dolphin.model.TourService;
import ru.yandex.travel.hotels.common.partners.dolphin.utils.DolphinRefundRulesBuilder;
import ru.yandex.travel.hotels.common.partners.dolphin.utils.Formatters;
import ru.yandex.travel.hotels.common.partners.dolphin.utils.RoomNameNormalizer;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.models.booking_flow.BedGroupInfo;
import ru.yandex.travel.hotels.models.booking_flow.BreakdownType;
import ru.yandex.travel.hotels.models.booking_flow.Coordinates;
import ru.yandex.travel.hotels.models.booking_flow.ExtraFee;
import ru.yandex.travel.hotels.models.booking_flow.ExtraFeeType;
import ru.yandex.travel.hotels.models.booking_flow.Image;
import ru.yandex.travel.hotels.models.booking_flow.LegalInfo;
import ru.yandex.travel.hotels.models.booking_flow.LocalizedPansionInfo;
import ru.yandex.travel.hotels.models.booking_flow.PartnerHotelInfo;
import ru.yandex.travel.hotels.models.booking_flow.Rate;
import ru.yandex.travel.hotels.models.booking_flow.RateInfo;
import ru.yandex.travel.hotels.models.booking_flow.RateStatus;
import ru.yandex.travel.hotels.models.booking_flow.RoomInfo;
import ru.yandex.travel.hotels.models.booking_flow.StayInfo;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TDolphinOffer;
import ru.yandex.travel.hotels.proto.TOfferData;
import ru.yandex.travel.orders.commons.proto.EServiceType;

@PartnerBean(EPartnerId.PI_DOLPHIN)
@EnableConfigurationProperties(DolphinConfigurationProperties.class)
@Slf4j
public class DolphinPartnerBookingProvider extends AbstractPartnerBookingProvider<TDolphinOffer> {
    private final DolphinClient dolphinClient;
    private final BedInflector bedInflector;
    private final LocalizationService localizationService;
    private final UpdatableDataHolder<IdNameMap> rooms;
    private final UpdatableDataHolder<IdNameMap> roomCategories;
    private final UpdatableDataHolder<Map<AreaType, Map<Long, Area>>> areas;
    private final UpdatableDataHolder<Map<Long, Pansion>> pansions;
    private final DolphinConfigurationProperties properties;
    private final TimezoneDetector tzDetector;
    private final HealthCheckedSupplier<RoomNameNormalizer> roomNameNormalizerSupplier;


    public DolphinPartnerBookingProvider(DolphinConfigurationProperties properties, DolphinClient dolphinClient,
                                         BedInflector bedInflector,
                                         LocalizationService localizationService, TimezoneDetector tzDetector,
                                         HealthCheckedSupplier<RoomNameNormalizer> roomNameNormalizerSupplier) {
        this.dolphinClient = dolphinClient;
        this.bedInflector = bedInflector;
        this.localizationService = localizationService;
        this.properties = properties;
        this.tzDetector = tzDetector;
        this.roomNameNormalizerSupplier = roomNameNormalizerSupplier;
        rooms = new UpdatableDataHolder<>(properties.getStaticUpdateInterval(), properties.getStaticRetryInterval(),
                dolphinClient::getRooms);
        roomCategories = new UpdatableDataHolder<>(properties.getStaticUpdateInterval(),
                properties.getStaticRetryInterval(), dolphinClient::getRoomCategories);
        areas = new UpdatableDataHolder<>(properties.getStaticUpdateInterval(), properties.getStaticRetryInterval(),
                dolphinClient::getAreaMap);
        pansions = new UpdatableDataHolder<>(properties.getStaticUpdateInterval(),
                properties.getStaticRetryInterval(), dolphinClient::getPansionMap);
    }

    PartnerHotelInfo mapPartnerHotelInfo(HotelContent hotelContent, TourContent tourContent, IdNameMap rooms,
                                         IdNameMap roomCategories, Map<AreaType, Map<Long, Area>> areas) {
        List<String> parts = new ArrayList<>(3);
        if (areas.get(AreaType.Country).containsKey(hotelContent.getCountryId())) {
            parts.add(areas.get(AreaType.Country).get(hotelContent.getCountryId()).getTitle());
        }
        if (areas.get(AreaType.Region).containsKey(hotelContent.getRegionId())) {
            parts.add(areas.get(AreaType.Region).get(hotelContent.getRegionId()).getTitle());
        }
        if (areas.get(AreaType.City).containsKey(hotelContent.getCityId())) {
            parts.add(areas.get(AreaType.City).get(hotelContent.getCityId()).getTitle());
        }
        var addrString = parts.stream().filter(p -> !hotelContent.getAddress().toLowerCase().contains(p.toLowerCase()))
                .collect(Collectors.joining(", "));
        if (addrString.isEmpty()) {
            addrString = hotelContent.getAddress();
        } else {
            addrString = addrString + ", " + hotelContent.getAddress();
        }
        int minImagesSize = Math.min(
                Math.min(hotelContent.getPhotos200x150().size(), hotelContent.getPhotos800x600().size()),
                hotelContent.getPhotos200x150().size());
        return PartnerHotelInfo.builder()
                .name(hotelContent.getName())
                .address(addrString)
                .coordinates(new Coordinates(hotelContent.getLongitude(), hotelContent.getLatitude()))
                .stars(hotelContent.getStars())
                .description(new PartnerHotelInfo.TextBlock("Описание", hotelContent.getInfo()))
                .description(new PartnerHotelInfo.TextBlock("Транспортная доступность",
                        hotelContent.getTransportAccessibility()))
                .images(IntStream.range(0, minImagesSize).mapToObj(i ->
                        new Image(hotelContent.getPhotos200x150().get(i),
                                hotelContent.getPhotos800x600().get(i),
                                hotelContent.getPhotos1020x700().get(i))).collect(Collectors.toList()))
                .build();
    }

    RateInfo mapRateInfo(BookingFlowContext context, CalculateOrderResponse response, Offer offer) {
        BigDecimal totalBrutto = BigDecimal.valueOf(response.getCost().getBrutto());
        BigDecimal totalCommission = BigDecimal.valueOf(response.getCost().getFee());
        BigDecimal grandTotal = totalBrutto.add(totalCommission);
        var servicesByType = response.getServiceList().stream().collect(
                Collectors.groupingBy(TourService::getType, Collectors.toList()));
        var hotelBrutto =
                BigDecimal.valueOf(servicesByType.get("Hotel").stream().mapToDouble(TourService::getBrutto).sum());
        var hotelFee = BigDecimal.valueOf(servicesByType.get("Hotel").stream().mapToDouble(TourService::getFee).sum());
        var otherBrutto =
                BigDecimal.valueOf(servicesByType.entrySet().stream().filter(entry -> !("Hotel".equals(entry.getKey())))
                        .flatMap(e -> e.getValue().stream()).mapToDouble(TourService::getBrutto).sum());
        var otherFee =
                BigDecimal.valueOf(servicesByType.entrySet().stream().filter(entry -> !("Hotel".equals(entry.getKey())))
                        .flatMap(e -> e.getValue().stream()).mapToDouble(TourService::getBrutto).sum());

        var hotelTotal = hotelBrutto.add(hotelFee);
        var otherTotal = otherBrutto.add(otherFee);
        var grandTotalByServices = hotelTotal.add(otherTotal);
        if (grandTotalByServices.setScale(0, RoundingMode.HALF_UP).compareTo(grandTotal.setScale(0,
                RoundingMode.HALF_UP)) != 0) {
            log.warn("Grand totals didn't match: total cost is {}, sum of services is {}", grandTotal,
                    grandTotalByServices);
        }
        hotelTotal = hotelTotal.subtract(grandTotalByServices.subtract(grandTotal));

        var rateInfoBuilder = RateInfo.builder()
                .baseRate(new Rate(hotelTotal.setScale(2, RoundingMode.HALF_UP).toString(), "RUB"))
                .baseRateBreakdown(null)
                .breakdownType(BreakdownType.NONE)
                .taxesAndFees(null)
                .totalRate(new Rate(grandTotal.setScale(2, RoundingMode.HALF_UP).toString(), "RUB"));

        BigDecimal shoppingPrice = BigDecimal.valueOf(offer.getPrice());
        // shoppingPrice > grandTotal means that actualized price is lower, consider it find for now
        int priceRelation = grandTotal.compareTo(shoppingPrice);
        if (priceRelation < 0) {
            priceRelation = 0;
        }
        // TODO(tivelkov): this will be changed some day, but for now simply price mismatch if we have
        //  non-accommodation paid services
        if (otherTotal.compareTo(BigDecimal.ZERO) != 0) {
            if (priceRelation == 0) {
                priceRelation = otherTotal.compareTo(BigDecimal.ZERO);
            }
            rateInfoBuilder.extraFees(Collections.singletonList(new ExtraFee(
                    otherTotal.setScale(2, RoundingMode.HALF_UP).toString(),
                    "RUB", ExtraFeeType.OTHER_FEE)));

        }
        boolean roomsAvailable = response.getAvailableRooms() > 0;
        if (!roomsAvailable) {
            rateInfoBuilder.status(RateStatus.SOLD_OUT);
        } else if (priceRelation > 0) {
            rateInfoBuilder.status(RateStatus.PRICE_MISMATCH_UP);
        } else if (priceRelation < 0) {
            rateInfoBuilder.status(RateStatus.PRICE_MISMATCH_DOWN);
        } else {
            rateInfoBuilder.status(RateStatus.CONFIRMED);
        }
        return rateInfoBuilder.build();
    }

    String getBedInfo(List<Integer> beds) {
        return beds.stream().collect(Collectors.groupingBy(i -> i, Collectors.summingInt(i -> 1))).entrySet().stream()
                .sorted(Comparator.comparing(Map.Entry::getKey))
                .map(entry -> bedInflector.inflectBedPlace(getBedType(entry.getKey()), entry.getValue(), "ru"))
                .collect(Collectors.joining(", "));
    }

    private BedInflector.BedType getBedType(int type) {
        switch (type) {
            case 1:
                return BedInflector.BedType.MAIN_BED;
            case 2:
                return BedInflector.BedType.EXTRA_BED;
            case 3:
                return BedInflector.BedType.CHILD_BED;
            case 4:
                return BedInflector.BedType.EXTRA_CHILD_BED;
            case 5:
                return BedInflector.BedType.CHILD_NO_BED;
            default:
                throw new PartnerException("Unexpected bedding type");
        }
    }

    CompletableFuture<RoomInfo> mapRoomInfo(Offer shoppingOffer, TDolphinOffer outerOffer, IdNameMap roomNames,
                                            IdNameMap roomCategories,
                                            Map<Long, Pansion> pansions) {
        var roomName = roomNames.get(shoppingOffer.getRoomId());
        var roomCatName = roomCategories.get(shoppingOffer.getCategoryId());
        var pansionType = pansions.get(outerOffer.getPansionId()).getPansionType();
        String pansionName = localizationService.localizePansion(pansionType, "ru");
        return roomNameNormalizerSupplier.get().thenApply(roomNameNormalizer -> RoomInfo.builder()
                .name(roomNameNormalizer.normalize(String.format("Номер %s %s", roomCatName, roomName)))
                .description("")
                .images(Collections.emptyList())
                .pansionInfo(new LocalizedPansionInfo(pansionType, pansionName))
                .roomAmenities(Collections.emptyList())
                .bedGroups(Collections.singletonList(new BedGroupInfo(0, getBedInfo(shoppingOffer.getBeds()))))
                .build());
    }

    StayInfo mapStayInfo(HotelContent hotelContent, TourContent tourContent) {
        LocalTime hotelCheckinTime = LocalTime.parse(hotelContent.getCheckinTime());
        LocalTime hotelCheckoutTime = LocalTime.parse(hotelContent.getCheckoutTime());
        Preconditions.checkNotNull(hotelCheckinTime, "Dolphin checkin time should not be NULL");
        Preconditions.checkNotNull(hotelCheckoutTime, "Dolphin checkout time should not be NULL");
        var builder = StayInfo.builder()
                .checkInStartTime(hotelCheckinTime.toString())
                .checkOutEndTime(hotelCheckoutTime.toString());
        if (hotelContent.getCheckinServiceMode() == CheckinServiceModeEnum.CSM_SELECTION) {
            LocalTime hotelCheckinServiceFrom = LocalTime.parse(hotelContent.getCheckinServiceFrom());
            LocalTime hotelCheckinServiceTill = LocalTime.parse(hotelContent.getCheckinServiceTill());
            Preconditions.checkNotNull(hotelCheckinServiceFrom, "Dolphin checkin-service from should not be NULL");
            Preconditions.checkNotNull(hotelCheckinServiceTill, "Dolphin checkin-service till should not be NULL");
            if (hotelCheckinServiceFrom.isAfter(hotelCheckinTime)) {
                builder.checkInStartTime(hotelCheckinServiceFrom.toString());
            }
            builder.checkInEndTime(hotelCheckinServiceTill.toString());
            builder.checkOutStartTime(hotelCheckinServiceFrom.toString());
            if (hotelCheckinServiceTill.isBefore(hotelCheckoutTime)) {
                builder.checkOutEndTime(hotelCheckinServiceTill.toString());
            }
        }
        if (tourContent.getRequiredCharges() != null) {
            String charges = tourContent.getRequiredCharges()
                    .replaceAll("^\"", "")
                    .replaceAll("\"$", "");
            if (Strings.isNotBlank(charges)) {
                builder.stayInstruction(charges);
            }
        }
        if (StringUtils.isNotBlank(tourContent.getDocuments())) {
            String documentsPatched = tourContent.getDocuments()
                    .replace("Туристический ваучер-путевка туроператора", "Ваучер")
                    .replace("Путевка", "Ваучер")
                    .replace("\r\n", "<br>");
            String documents = String.format("<p><b>Документы, требуемые для заселения:</b></p><p>%s</p>",
                    documentsPatched);
            builder.rateDescriptionForSupport("Требуемые документы: " + tourContent.getDocuments());
            builder.stayInstruction(documents);
        }
        return builder.build();
    }

    RefundRules mapRefundRules(BookingFlowContext context, CalculateOrderResponse calculateOrderResponse,
                               HotelContent hotelContent) {
        BigDecimal totalBrutto = BigDecimal.valueOf(calculateOrderResponse.getCost().getBrutto());
        BigDecimal totalCommission = BigDecimal.valueOf(calculateOrderResponse.getCost().getFee());
        BigDecimal grandTotal = totalBrutto.add(totalCommission);
        return DolphinRefundRulesBuilder.build(
                context.getCreatedAt(),
                getCheckInMoment(context, hotelContent),
                grandTotal,
                "RUB",
                new FixedPercentFeeRefundParams(properties.getCancellationPenalties()));
    }

    @Override
    PartnerFutures getPartnerFutures(BookingFlowContext context, CompletableFuture<TDolphinOffer> offerDataFuture) {
        CompletableFuture<CallContext> ccf = getCallContextFuture(context, offerDataFuture);
        var hotelContentFuture =
                ccf.thenCompose(ctx -> dolphinClient.withCallContext(ctx).getHotelContent(context.getDecodedToken().getOriginalId()));
        var tourContentFuture = ccf.thenCompose(ctx ->
                offerDataFuture.thenCompose(od -> dolphinClient.withCallContext(ctx).getTourContent(String.valueOf(od.getTourId()))));

        CompletableFuture<IdNameMap> roomsFuture = ccf.thenCompose(ctx -> {
            if (ctx.getTestContext() != null) {
                return dolphinClient.withCallContext(ctx).getRooms();
            } else {
                return rooms.get();
            }
        });
        CompletableFuture<IdNameMap> roomCategoriesFuture = ccf.thenCompose(ctx -> {
            if (ctx.getTestContext() != null) {
                return dolphinClient.withCallContext(ctx).getRoomCategories();
            } else {
                return roomCategories.get();
            }
        });
        CompletableFuture<Map<Long, Pansion>> pansionFuture = ccf.thenCompose(ctx -> {
            if (ctx.getTestContext() != null) {
                return dolphinClient.withCallContext(ctx).getPansionMap();
            } else {
                return pansions.get();
            }
        });
        CompletableFuture<Map<AreaType, Map<Long, Area>>> areasFuture = ccf.thenCompose(ctx -> {
            if (ctx.getTestContext() != null) {
                return dolphinClient.withCallContext(ctx).getAreaMap();
            } else {
                return areas.get();
            }
        });

        var offerFuture =
                offerDataFuture.thenApply(od -> ProtoUtils.fromTJson(od.getShoppingOffer(),
                        Offer.class));

        var priceFuture = ccf.thenCompose(
                ctx -> offerDataFuture.thenCompose(od -> {
                    var offer = ProtoUtils.fromTJson(od.getShoppingOffer(), Offer.class);
                    var request = CalculateOrderRequest.builder()
                            .priceKey(PriceKey.builder()
                                    .hotelId(od.getHotelId())
                                    .tourId(od.getTourId())
                                    .pansionId(od.getPansionId())
                                    .roomId(offer.getRoomId())
                                    .roomCategoryId(offer.getCategoryId())
                                    .date(LocalDate.parse(od.getDate(), Formatters.DOLPHIN_DATE_FORMATTER))
                                    .nights(od.getNights())
                                    .beds(offer.getBeds())
                                    .build())
                            .adults(context.getDecodedToken().getOccupancy().getAdults())
                            .children(context.getDecodedToken().getOccupancy().getChildren().stream()
                                    .sorted((a, b) -> Integer.compare(b, a)).collect(Collectors.toList()))
                            .build();
                    return dolphinClient.withCallContext(ctx).calculateOrder(request);
                }));

        return new PartnerFutures() {
            @Override
            public CompletableFuture<PartnerHotelInfo> getPartnerHotelInfo() {
                return CompletableFuture.allOf(hotelContentFuture, tourContentFuture, roomsFuture,
                        roomCategoriesFuture, areasFuture)
                        .thenApply(ignored -> mapPartnerHotelInfo(
                                hotelContentFuture.join(),
                                tourContentFuture.join(),
                                roomsFuture.join(),
                                roomCategoriesFuture.join(), areasFuture.join()));
            }

            @Override
            public CompletableFuture<RateInfo> getRateInfo() {
                CompletableFuture<RateInfo> res = offerDataFuture.thenCompose(od -> {
                    var offer = ProtoUtils.fromTJson(od.getShoppingOffer(), Offer.class);
                    return priceFuture.thenApply(resp -> mapRateInfo(context, resp, offer));
                });
                return FutureUtils.handleExceptionOfType(res, SoldOutException.class,
                        h -> RateInfo.builder().status(RateStatus.SOLD_OUT).build());
            }

            @Override
            public CompletableFuture<RoomInfo> getRoomInfo() {
                return CompletableFuture.allOf(offerDataFuture, offerFuture, pansionFuture, roomCategoriesFuture,
                        roomsFuture)
                        .thenCompose(ignored -> mapRoomInfo(
                                offerFuture.join(),
                                offerDataFuture.join(),
                                roomsFuture.join(),
                                roomCategoriesFuture.join(),
                                pansionFuture.join()));
            }

            @Override
            public CompletableFuture<StayInfo> getStayInfo() {
                return hotelContentFuture.thenCompose(hc -> tourContentFuture.thenApply(t -> mapStayInfo(hc, t)));
            }

            @Override
            public CompletableFuture<RefundRules> getRefundRules() {
                return priceFuture.thenCompose(pc ->
                        offerDataFuture.thenCompose(od ->
                                hotelContentFuture.thenApply(hc -> mapRefundRules(context, pc, hc))));
            }

            @Override
            public CompletableFuture<LegalInfo.LegalInfoItem> getPartnerLegalInfoItem() {
                return CompletableFuture.completedFuture(
                        LegalInfo.LegalInfoItem.builder()
                                .name(properties.getPartnerLegalData().getName())
                                .ogrn(properties.getPartnerLegalData().getOgrn())
                                .legalAddress(properties.getPartnerLegalData().getAddress())
                                .workingHours(properties.getPartnerLegalData().getWorkingHours())
                                .build()
                );
            }

            @Override
            public CompletableFuture<LegalInfo.LegalInfoItem> getHotelLegalInfoItem() {
                return CompletableFuture.completedFuture(null);
            }

            @Override
            public CompletableFuture<HotelItinerary> createHotelItinerary() {
                return priceFuture.thenCompose(price ->
                        hotelContentFuture.thenCompose(hotelContent ->
                                offerDataFuture.thenApply(od -> {
                                    Offer innerOffer = ProtoUtils.fromTJson(od.getShoppingOffer(), Offer.class);
                                    DolphinHotelItinerary itinerary = new DolphinHotelItinerary();
                                    itinerary.setHotelId(od.getHotelId());
                                    itinerary.setTourId(od.getTourId());
                                    itinerary.setPansionId(od.getPansionId());
                                    itinerary.setRoomId(innerOffer.getRoomId());
                                    itinerary.setRoomCatId(innerOffer.getCategoryId());
                                    itinerary.setCheckinDate(context.getDecodedToken().getCheckInDate());
                                    itinerary.setNights(od.getNights());
                                    itinerary.setBeds(innerOffer.getBeds());
                                    itinerary.setOccupancy(context.getDecodedToken().getOccupancy());
                                    itinerary.setHotelZoneId(getZoneId(hotelContent));
                                    itinerary.setCheckInMoment(getCheckInMoment(context, hotelContent));
                                    itinerary.setRefundParams(new FixedPercentFeeRefundParams(properties.getCancellationPenalties()));
                                    // TODO(tivelkov) move this to base class: partner-adapters should fill-in only
                                    //  partner-specific details
                                    BigDecimal commission = BigDecimal.valueOf(price.getCost().getFee());
                                    var totalPayable =
                                            BigDecimal.valueOf(price.getCost().getBrutto())
                                                    .add(commission);
                                    itinerary.setFiscalPrice(Money.of(totalPayable, ProtoCurrencyUnit.RUB));
                                    return itinerary;
                                })));
            }
        };
    }

    private CompletableFuture<CallContext> getCallContextFuture(BookingFlowContext flowContext,
                                                                CompletableFuture<TDolphinOffer> offerDataFuture) {
        return CompletableFuture.allOf(flowContext.getTestContextFuture(), offerDataFuture).thenApply(ignored ->
                new CallContext(flowContext.getStage() == BookingFlowContext.Stage.GET_OFFER ?
                        CallContext.CallPhase.OFFER_VALIDATION : CallContext.CallPhase.ORDER_CREATION,
                        flowContext.getTestContextFuture().join(), offerDataFuture.join(),
                        flowContext.getDecodedToken(), null, null));
    }

    private ZoneId getZoneId(HotelContent hotelContent) {
        return tzDetector.getZoneOffset(hotelContent.getLatitude(), hotelContent.getLongitude());
    }

    private Instant getCheckInMoment(BookingFlowContext context, HotelContent hotelContent) {
        ZoneOffset hotelZoneOffset = getZoneId(hotelContent).getRules().getOffset(Instant.now());
        return context.getDecodedToken().getCheckInDate().atStartOfDay().toInstant(hotelZoneOffset);
    }

    @Override
    public EServiceType getServiceType() {
        return EServiceType.PT_DOLPHIN_HOTEL;
    }

    @Override
    TDolphinOffer getPartnerOffer(TOfferData offerData) {
        Preconditions.checkArgument(offerData.hasDolphinOffer());
        return offerData.getDolphinOffer();
    }

    @Override
    public EPartnerId getPartnerId() {
        return EPartnerId.PI_DOLPHIN;
    }
}
