package ru.yandex.travel.api.services.hotels_booking_flow;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;
import org.javamoney.moneta.Money;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.data.util.StreamUtils;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.api.exceptions.InputError;
import ru.yandex.travel.api.exceptions.InvalidInputException;
import ru.yandex.travel.api.services.common.CountryNameService;
import ru.yandex.travel.api.services.hotels_booking_flow.models.CheckinCheckoutTime;
import ru.yandex.travel.api.services.localization.LocalizationService;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.streams.CustomCollectors;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.partners.expedia.ApiVersion;
import ru.yandex.travel.hotels.common.partners.expedia.DefaultExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaRefundRulesBuilder;
import ru.yandex.travel.hotels.common.partners.expedia.Helpers;
import ru.yandex.travel.hotels.common.partners.expedia.KnownAmenity;
import ru.yandex.travel.hotels.common.partners.expedia.KnownPropertyAmenity;
import ru.yandex.travel.hotels.common.partners.expedia.Pansions;
import ru.yandex.travel.hotels.common.partners.expedia.exceptions.PriceSumMismatchException;
import ru.yandex.travel.hotels.common.partners.expedia.model.common.PriceItem;
import ru.yandex.travel.hotels.common.partners.expedia.model.common.PricingInformation;
import ru.yandex.travel.hotels.common.partners.expedia.model.content.ExpediaImage;
import ru.yandex.travel.hotels.common.partners.expedia.model.content.PropertyContent;
import ru.yandex.travel.hotels.common.partners.expedia.model.content.PropertyRatingType;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.PriceItemType;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.RoomPriceCheck;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.ShoppingBedGroup;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.ShoppingRate;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.ShoppingRateStatus;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.models.booking_flow.Amenity;
import ru.yandex.travel.hotels.models.booking_flow.BedGroupInfo;
import ru.yandex.travel.hotels.models.booking_flow.BreakdownType;
import ru.yandex.travel.hotels.models.booking_flow.Coordinates;
import ru.yandex.travel.hotels.models.booking_flow.ExtraFee;
import ru.yandex.travel.hotels.models.booking_flow.ExtraFeeType;
import ru.yandex.travel.hotels.models.booking_flow.Image;
import ru.yandex.travel.hotels.models.booking_flow.LegalInfo;
import ru.yandex.travel.hotels.models.booking_flow.LocalizedPansionInfo;
import ru.yandex.travel.hotels.models.booking_flow.PartnerHotelInfo;
import ru.yandex.travel.hotels.models.booking_flow.Rate;
import ru.yandex.travel.hotels.models.booking_flow.RateInfo;
import ru.yandex.travel.hotels.models.booking_flow.RateStatus;
import ru.yandex.travel.hotels.models.booking_flow.RoomInfo;
import ru.yandex.travel.hotels.models.booking_flow.StayInfo;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TExpediaOffer;
import ru.yandex.travel.hotels.proto.TOfferData;
import ru.yandex.travel.orders.commons.proto.EServiceType;

import static ru.yandex.travel.hotels.models.booking_flow.RateStatus.CONFIRMED;
import static ru.yandex.travel.hotels.models.booking_flow.RateStatus.SOLD_OUT;


@Slf4j
@PartnerBean(EPartnerId.PI_EXPEDIA)
@EnableConfigurationProperties(ExpediaPartnerBookingProviderProperties.class)
@RequiredArgsConstructor
public class ExpediaPartnerBookingProvider extends AbstractPartnerBookingProvider<TExpediaOffer> implements InitializingBean {
    static final String S = "70px";
    static final String M = "350px";
    static final String L = "1000px";


    // TODO: may be we need some common place for it?
    private static final Map<ECurrency, String> PROTO_TO_EXPEDIA_CURRENCY_MAP = ImmutableMap.of(
            ECurrency.C_RUB, "RUB",
            ECurrency.C_USD, "USD"
    );
    private final ExpediaClient expediaClient;
    private final CountryNameService contryNameService;
    private final LocalizationService localizationService;
    private final ExpediaPartnerBookingProviderProperties config;

    private ObjectMapper expediaMapper;

    @Override
    public EServiceType getServiceType() {
        return EServiceType.PT_EXPEDIA_HOTEL;
    }

    @Override
    public EPartnerId getPartnerId() {
        return EPartnerId.PI_EXPEDIA;
    }

    @Override
    TExpediaOffer getPartnerOffer(TOfferData offerData) {
        Preconditions.checkArgument(offerData.hasExpediaOffer());
        return offerData.getExpediaOffer();
    }

    private String getExpediaSessionId(BookingFlowContext context, TExpediaOffer offerData) {
        var searchSessionId = offerData.getSearchRequestId();
        if (searchSessionId == null) {
            searchSessionId = "unknown";
        }
        return searchSessionId + ":" + context.getDeduplicationKey();
    }

    private CompletionStage<PropertyContent> getPropertyContent(BookingFlowContext context, TExpediaOffer offerData) {
        return expediaClient.getPropertyContent(offerData.getHotelId(), context.getUserIp(),
                context.getUserAgent(),
                getExpediaSessionId(context, offerData)).whenComplete((r, t) -> {
            if (t == null) {
                log.debug("Expedia content query completed successfully");
            } else {
                log.error("Expedia content query failed with an exception", t);
            }
        });
    }

    private CompletionStage<RoomPriceCheck> getPrice(BookingFlowContext context, TExpediaOffer offerData) {
        var fxRateExpires = ProtoUtils.toInstant(offerData.getExchangeRateValidUntil());
        if (fxRateExpires.isBefore(Instant.now())) {
            return CompletableFuture.failedFuture(new RuntimeException("FX rate expired"));
        }
        ShoppingRate shoppingOfferRate = getPropertyAvailabilityRates(offerData);
        if (context.getOrderCreationData() != null) {
            if (context.getOrderCreationData().getSelectedBedGroupIndex() != 0) {
                return CompletableFuture.failedFuture(new InvalidInputException(Collections.singletonList(
                        new InputError("selected_bed_group_index",
                                context.getOrderCreationData().getSelectedBedGroupIndex(),
                                InputError.ErrorType.INVALID_BED_GROUP_INDEX))));
            }
        }

        var bedGroupsSortedById = shoppingOfferRate.getBedGroups().entrySet()
                .stream()
                .sorted(Map.Entry.comparingByKey())
                .map(Map.Entry::getValue).collect(Collectors.toList());
        ShoppingBedGroup defaultBedGroup = bedGroupsSortedById.stream()
                .filter(bg -> bg.getConfiguration().size() == 1 && bg.getConfiguration().get(0).getQuantity() == 1)
                .findFirst()
                .orElse(bedGroupsSortedById.get(0));
        String checkPriceToken = Helpers.retrievePriceCheckToken(defaultBedGroup.getLinks().getPriceCheck().getHref());
        ApiVersion apiVersion = StringUtils.isNotBlank(offerData.getApiVersion())
                ? ApiVersion.fromString(offerData.getApiVersion())
                : ApiVersion.V2_4;
        return expediaClient.usingApi(apiVersion).checkPrice(offerData.getHotelId(), offerData.getRoomId(),
                offerData.getRateId(),
                checkPriceToken, context.getUserIp(), context.getUserAgent(),
                getExpediaSessionId(context, offerData));
    }

    PartnerHotelInfo mapPartnerHotelInfo(PropertyContent propertyContent) {
        List<String> addressParts = new ArrayList<>(3);
        addressParts.add(contryNameService.resolveCountryCode(propertyContent.getAddress().getCountryCode()));
        addressParts.add(propertyContent.getAddress().getCity());
        addressParts.add(propertyContent.getAddress().getLine1());
        addressParts.add(propertyContent.getAddress().getLine2());
        var addString = addressParts.stream().filter(Strings::isNotBlank).collect(Collectors.joining(", "));
        var builder = PartnerHotelInfo.builder()
                .name(propertyContent.getName())
                .address(addString)
                .amenities(propertyContent.getAmenities().values().stream()
                        .map(a -> new Amenity(String.valueOf(a.getId()), a.getName())).collect(Collectors.toList()))
                .images(propertyContent.getImages().stream().map(this::mapImage).collect(Collectors.toList()))
                .phone(propertyContent.getPhone())
                .description(new PartnerHotelInfo.TextBlock("Расположение",
                        propertyContent.getDescriptions().getLocation()))
                .description(new PartnerHotelInfo.TextBlock("Размещение", propertyContent.getDescriptions().getRooms()))
                .description(new PartnerHotelInfo.TextBlock("Питание", propertyContent.getDescriptions().getDining()))
                .description(new PartnerHotelInfo.TextBlock("Услуги отеля",
                        propertyContent.getDescriptions().getAmenities()))
                .description(new PartnerHotelInfo.TextBlock("Услуги для бизнеса",
                        propertyContent.getDescriptions().getBusinessAmenities()));

        if (propertyContent.getLocation() != null && propertyContent.getLocation().getCoordinates() != null) {
            builder.coordinates(new Coordinates(
                    propertyContent.getLocation().getCoordinates().getLongitude(),
                    propertyContent.getLocation().getCoordinates().getLatitude()));
        }
        if (propertyContent.getRatings() != null && propertyContent.getRatings().getProperty() != null &&
                PropertyRatingType.STAR.equals(propertyContent.getRatings().getProperty().getType())) {
            var expRating = propertyContent.getRatings().getProperty().getRating();
            if (expRating != null) {
                builder.stars(expRating.intValue());
            }
        }
        return builder.build();
    }

    Image mapImage(ExpediaImage img) {
        var imageBuilder = Image.builder();
        if (img.getLinks().containsKey(S)) {
            imageBuilder.s(img.getLinks().get(S).getHref());
        }
        if (img.getLinks().containsKey(M)) {
            imageBuilder.m(img.getLinks().get(M).getHref());
        }
        if (img.getLinks().containsKey(L)) {
            imageBuilder.l(img.getLinks().get(L).getHref());
        }
        return imageBuilder.build();
    }

    private RateInfo mapRateInfo(TExpediaOffer offer, RoomPriceCheck priceCheck) {
        var builder = RateInfo.builder();

        if (priceCheck.getStatus() != ShoppingRateStatus.SOLD_OUT) {
            PricingInformation pricing = priceCheck.getOccupancyPricing().get(offer.getOccupancy());
            PricingInformation roundedPricing = Helpers.round(pricing);

            List<BigDecimal> nightBaseRates = roundedPricing.getNightly().stream().map(night ->
                    night.stream()
                            .filter(item -> item.getType() == PriceItemType.BASE_RATE)
                            .map(PriceItem::getValue)
                            .reduce(BigDecimal.ZERO, BigDecimal::add))
                    .collect(Collectors.toList());
            BigDecimal nightsBaseTotal = nightBaseRates.stream().reduce(BigDecimal.ZERO, BigDecimal::add);

            Map<PriceItemType, BigDecimal> feesByType = roundedPricing.getNightly().stream()
                    .flatMap(List::stream)
                    .filter(item -> item.getType() != PriceItemType.BASE_RATE)
                    .collect(
                            Collectors.groupingBy(PriceItem::getType,
                                    CustomCollectors.summingBigDecimal(PriceItem::getValue)));
            Map<PriceItemType, BigDecimal> stayByType = roundedPricing.getStay().stream().collect(
                    Collectors.groupingBy(PriceItem::getType,
                            CustomCollectors.summingBigDecimal(PriceItem::getValue)));
            // stay by types added to fees by type
            stayByType.forEach((type, value) ->
                    feesByType.put(type, feesByType.getOrDefault(type, BigDecimal.ZERO).add(value)));

            BigDecimal feesTotal = feesByType.values().stream().reduce(BigDecimal.ZERO, BigDecimal::add);
            BigDecimal grandTotal = nightsBaseTotal.add(feesTotal);
            BigDecimal expectedTotal = roundedPricing.getTotals().getInclusive().getBillableCurrency().getValue();
            if (grandTotal.compareTo(expectedTotal) != 0) {
                throw new PriceSumMismatchException(grandTotal, expectedTotal);
            }
            if (priceCheck.getStatus() == ShoppingRateStatus.AVAILABLE) {
                builder.status(CONFIRMED);
            } else {
                var searchedPrice = BigDecimal.valueOf(offer.getPayablePrice(), 2);
                log.info("Price mismatch on offer validation: searcher price is {}, verified price is {}",
                        searchedPrice,
                        grandTotal);
                builder.status(RateStatus.fromComparison(searchedPrice, grandTotal));
            }
            String verifiedCurrency = pricing.getTotals().getInclusive().getBillableCurrency().getCurrency();
            Preconditions.checkState(verifiedCurrency.equals(PROTO_TO_EXPEDIA_CURRENCY_MAP.get(offer.getPayableCurrency())),
                    "Verified currency does not match offer payable currency");

            builder
                    .baseRate(new Rate(nightsBaseTotal.setScale(2, RoundingMode.HALF_UP).toString(), verifiedCurrency))
                    .breakdownType(BreakdownType.NIGHT)
                    .baseRateBreakdown(nightBaseRates.stream().map(price ->
                            new Rate(price.setScale(2, RoundingMode.HALF_UP).toString(), verifiedCurrency)).collect(Collectors.toList()))
                    .taxesAndFees(new Rate(feesTotal.setScale(2, RoundingMode.HALF_UP).toString(), verifiedCurrency))
                    .totalRate(new Rate(grandTotal.setScale(2, RoundingMode.HALF_UP).toString(), verifiedCurrency));

            if (roundedPricing.getFees() != null) {
                List<ExtraFee> extraFees = new ArrayList<>();
                if (roundedPricing.getFees().getResortFee() != null) {
                    extraFees.add(
                            new ExtraFee(roundedPricing.getFees().getResortFee().getBillableCurrency().getValue().setScale(2, RoundingMode.HALF_UP).toString(),
                                    roundedPricing.getFees().getResortFee().getBillableCurrency().getCurrency(),
                                    ExtraFeeType.RESORT_FEE));
                }
                if (roundedPricing.getFees().getMandatoryTax() != null) {
                    extraFees.add(
                            new ExtraFee(roundedPricing.getFees().getMandatoryTax().getBillableCurrency().getValue().setScale(2, RoundingMode.HALF_UP).toString(),
                                    roundedPricing.getFees().getMandatoryTax().getBillableCurrency().getCurrency(),
                                    ExtraFeeType.OTHER_FEE));
                }
                if (roundedPricing.getFees().getMandatoryFee() != null) {
                    extraFees.add(
                            new ExtraFee(roundedPricing.getFees().getMandatoryFee().getBillableCurrency().getValue().setScale(2, RoundingMode.HALF_UP).toString(),
                                    roundedPricing.getFees().getMandatoryFee().getBillableCurrency().getCurrency(),
                                    ExtraFeeType.OTHER_FEE));
                }

                if (!extraFees.isEmpty()) {
                    builder.extraFees(extraFees);
                }
            }
        } else {
            builder.status(SOLD_OUT);
        }
        return builder.build();
    }


    private RoomInfo mapRoomInfo(PropertyContent propertyContent, TExpediaOffer offerData) {
        ShoppingRate shoppingOfferRate = getPropertyAvailabilityRates(offerData);
        var expediaRoom = propertyContent.getRooms().get(offerData.getRoomId());
        Preconditions.checkNotNull(expediaRoom, "Unknown expedia room");
        var bedGroups = shoppingOfferRate.getBedGroups().entrySet()
                .stream()
                .sorted(Map.Entry.comparingByKey())
                .map(Map.Entry::getValue).collect(Collectors.toList());
        if (!"2.4".equals(offerData.getApiVersion())) {
            bedGroups = Helpers.mapBedGroupDetails(bedGroups,
                    expediaRoom.getBedGroups().values());
        }
        var propertyPansionType =
                Pansions.getPropertyPansionType(KnownPropertyAmenity.fromPropertyContent(propertyContent));
        var ratePansionType = Pansions.getPansionType(KnownAmenity.fromPropertyContent(propertyContent,
                offerData.getRateId()));
        var pansionType = Pansions.combinePansionType(propertyPansionType, ratePansionType);
        String pansionName = localizationService.localizePansion(pansionType, "ru");
        List<BedGroupInfo> bedGroupsInfo = StreamUtils.zip(
                        bedGroups.stream(),
                        IntStream.range(0, bedGroups.size()).boxed(),
                        (bedGroup, iter) -> new BedGroupInfo(iter, bedGroup.getDescription()))
                .collect(Collectors.toList());
        BedGroupInfo defaultBedGroupInfo = createOneBedGroupInfo(bedGroupsInfo);
        var roomInfoBuilder = RoomInfo.builder()
                .name(expediaRoom.getName())
                .description(expediaRoom.getDescriptions() == null ? "" : expediaRoom.getDescriptions().getOverview())
                .bedGroups(List.of(defaultBedGroupInfo))
                .pansionInfo(new LocalizedPansionInfo(pansionType, pansionName))
                .roomAmenities(expediaRoom.getAmenities().values().stream()
                        .map(a -> new Amenity(a.getId(), a.getName())).collect(Collectors.toList()));

        if (expediaRoom.getImages() != null && expediaRoom.getImages().size() > 0) {
            roomInfoBuilder.images(expediaRoom.getImages().stream().map(this::mapImage).collect(Collectors.toList()));
        }
        return roomInfoBuilder.build();
    }

    private ShoppingRate getPropertyAvailabilityRates(TExpediaOffer offerData) {
        ShoppingRate shoppingOfferRate;
        try {
            shoppingOfferRate = expediaMapper.readerFor(ShoppingRate.class)
                    .readValue(offerData.getShoppingOffer().getValue());
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
        return shoppingOfferRate;
    }

    StayInfo mapStayInfo(PropertyContent propertyContent, LocalDate checkInDate, LocalDate checkOutDate) {
        var builder = StayInfo.builder();
        String checkOutEndTime = null;
        if (propertyContent.getCheckout() != null) {
            var checkoutTime = CheckinCheckoutTime.fromString(propertyContent.getCheckout().getTime());
            if (checkoutTime != null) {
                checkOutEndTime = checkoutTime.getBegins().toString();
            } else {
                checkOutEndTime = propertyContent.getCheckout().getTime();
            }
        }
        builder.checkOutEndTime(checkOutEndTime);
        if (propertyContent.getCheckin() != null) {
            var checkInTime = CheckinCheckoutTime.fromStrings(propertyContent.getCheckin().getBeginTime(),
                    propertyContent.getCheckin().getEndTime());
            if (checkInTime != null) {
                builder.checkInStartTime(checkInTime.getBegins().toString());
                var checkinEnd = checkInTime.getEnds().toString();
                if (checkInTime.isOvernight()) {
                    //don't set checkIn end time if it is equal to checkOut time
                    int stayDuration = (int) ChronoUnit.DAYS.between(checkInDate, checkOutDate);
                    if (!(checkOutEndTime != null && checkOutEndTime.equals(checkinEnd) && stayDuration == 1)) {
                        builder.checkInEndTime(checkinEnd + " следующего дня");
                    }
                }
                builder.checkInEndTime(checkinEnd);
            } else {
                builder.checkInStartTime(propertyContent.getCheckin().getBeginTime());
                builder.checkInEndTime(propertyContent.getCheckin().getEndTime());
            }
        }

        String mandatoryFees = "";
        String optionalFees = "";
        String specialInstructions = "";
        String regularInstructions = "";
        String knowBeforeYouGo = "";
        if (propertyContent.getFees() != null) {
            if (Strings.isNotBlank(propertyContent.getFees().getMandatory())) {
                mandatoryFees = propertyContent.getFees().getMandatory();
            }
            if (Strings.isNotBlank(propertyContent.getFees().getOptional())) {
                optionalFees = propertyContent.getFees().getOptional();
            }
        }
        if (propertyContent.getCheckin() != null) {
            if (Strings.isNotBlank(propertyContent.getCheckin().getSpecialInstructions())) {
                specialInstructions = propertyContent.getCheckin().getSpecialInstructions();
            }
            if (Strings.isNotBlank(propertyContent.getCheckin().getInstructions())) {
                regularInstructions = propertyContent.getCheckin().getInstructions();
            }
        }
        if (propertyContent.getPolicies() != null && Strings.isNotBlank(propertyContent.getPolicies().getKnowBeforeYouGo())) {
            knowBeforeYouGo = propertyContent.getPolicies().getKnowBeforeYouGo();
        }

        List<String> stayInstructions = Stream.of(
                mandatoryFees,
                specialInstructions,
                regularInstructions,
                optionalFees,
                knowBeforeYouGo)
                .filter(Objects::nonNull).collect(Collectors.toList());
        builder.stayInstructions(stayInstructions);
        return builder.build();
    }

    private RefundRules mapRefundRules(RoomPriceCheck priceCheckResult, TExpediaOffer offerData) {
        if (priceCheckResult.getStatus() == ShoppingRateStatus.SOLD_OUT) {
            log.info("Unable to map cancellation info as the offer has sold out");
            return null;
        }
        var offerRate =
                getPropertyAvailabilityRates(offerData).toBuilder().occupancyPricing(priceCheckResult.getOccupancyPricing()).build();
        return ExpediaRefundRulesBuilder.build(
                offerRate, offerData.getOccupancy(), config.getCancellationSafetyInterval());
    }

    @Override
    PartnerFutures getPartnerFutures(BookingFlowContext context, CompletableFuture<TExpediaOffer> offerDataFuture) {


        CompletableFuture<PropertyContent> contentFuture =
                offerDataFuture.thenCompose(od -> getPropertyContent(context, od));
        CompletableFuture<RoomPriceCheck> priceCheckFuture = offerDataFuture.thenCompose(od -> getPrice(context, od));

        return new PartnerFutures() {
            @Override
            public CompletableFuture<PartnerHotelInfo> getPartnerHotelInfo() {
                return contentFuture.thenApply(ExpediaPartnerBookingProvider.this::mapPartnerHotelInfo);
            }

            @Override
            public CompletableFuture<RateInfo> getRateInfo() {
                return offerDataFuture.thenCompose(o -> priceCheckFuture.thenApply(p -> mapRateInfo(o, p)));
            }


            @Override
            public CompletableFuture<RoomInfo> getRoomInfo() {
                return offerDataFuture.thenCompose(o -> contentFuture.thenApply(c -> mapRoomInfo(c, o)));
            }

            @Override
            public CompletableFuture<StayInfo> getStayInfo() {
                return offerDataFuture.thenCompose(o -> contentFuture.thenApply(content ->
                        mapStayInfo(content, LocalDate.parse(o.getCheckInDate()),
                                LocalDate.parse(o.getCheckOutDate()))));
            }

            @Override
            public CompletableFuture<RefundRules> getRefundRules() {
                return offerDataFuture.thenCompose(o -> priceCheckFuture.thenApply(p -> mapRefundRules(p, o)));
            }

            @Override
            public CompletableFuture<LegalInfo.LegalInfoItem> getPartnerLegalInfoItem() {
                return CompletableFuture.completedFuture(
                        LegalInfo.LegalInfoItem.builder()
                                .name(config.getPartnerLegalData().getName())
                                .registryNumber(config.getPartnerLegalData().getRegistryNumber())
                                .legalAddress(config.getPartnerLegalData().getAddress())
                                .workingHours(config.getPartnerLegalData().getWorkingHours())
                                .build()
                );
            }

            @Override
            public CompletableFuture<LegalInfo.LegalInfoItem> getHotelLegalInfoItem() {
                return CompletableFuture.completedFuture(null);
            }

            @Override
            public CompletableFuture<HotelItinerary> createHotelItinerary() {
                return offerDataFuture.thenCompose(od -> priceCheckFuture.thenApply(pc -> {
                    Preconditions.checkArgument(pc.getStatus() == ShoppingRateStatus.AVAILABLE,
                            "PriceCheck is not matched");
                    ApiVersion apiVersion = StringUtils.isNotBlank(od.getApiVersion())
                            ? ApiVersion.fromString(od.getApiVersion())
                            : ApiVersion.V2_4;
                    ExpediaHotelItinerary itinerary = new ExpediaHotelItinerary();
                    itinerary.setCustomerSessionId(getExpediaSessionId(context, od));
                    itinerary.setHotelId(od.getHotelId());
                    itinerary.setRoomId(od.getRoomId());
                    itinerary.setRateId(od.getRateId());
                    itinerary.setOccupancy(od.getOccupancy());
                    itinerary.setApiVersion(apiVersion);
                    itinerary.setExpediaReservationToken(
                            Helpers.retrieveReservationToken(pc.getLinks().getBook().getHref()));
                    var priceCheckPricing = pc.getOccupancyPricing().get(od.getOccupancy());
                    var roundedPricing = Helpers.round(priceCheckPricing);
                    var totalAmount = roundedPricing.getTotals().getInclusive().getBillableCurrency().getValue();
                    var price = Money.of(totalAmount,
                            ProtoCurrencyUnit.fromProtoCurrencyUnit(od.getPayableCurrency()));
                    itinerary.setFiscalPrice(price);
                    itinerary.setExpiresAtInstant(ProtoUtils.toInstant(od.getExchangeRateValidUntil()));
                    return itinerary;
                }));
            }
        };
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        expediaMapper = DefaultExpediaClient.createObjectMapper();
    }
}
