package ru.yandex.travel.api.services.hotels_booking_flow;

import java.util.Collections;
import java.util.concurrent.CompletableFuture;

import lombok.RequiredArgsConstructor;
import org.apache.logging.log4j.util.Strings;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;
import yandex.maps.proto.common2.geometry.GeometryOuterClass;
import yandex.maps.proto.search.address.AddressOuterClass;
import yandex.maps.proto.search.kind.KindOuterClass;

import ru.yandex.travel.hotels.common.LocationType;
import ru.yandex.travel.hotels.common.PartnerConfigService;
import ru.yandex.travel.hotels.common.token.TravelToken;
import ru.yandex.travel.hotels.geosearch.GeoSearchService;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.geosearch.model.GeoHotelOriginalId;
import ru.yandex.travel.hotels.geosearch.model.GeoOriginEnum;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchReq;
import ru.yandex.travel.hotels.models.booking_flow.Coordinates;
import ru.yandex.travel.hotels.models.booking_flow.HotelInfo;
import ru.yandex.travel.hotels.models.booking_flow.LegalInfo;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TPartner;


@Service
@RequiredArgsConstructor
@EnableConfigurationProperties(GeoSearchHotelContentServiceProperties.class)
public class GeoSearchHotelContentService {
    private final GeoSearchService geoSearchService;
    private final GeoSearchHotelContentServiceProperties properties;
    private final PartnerConfigService partnerConfigService;

    public GeoSearchFutures getFutures(BookingFlowContext context) {
        var geoSearchResponseFuture = getHotelFromGeo(context.getDecodedToken());

        return new GeoSearchFutures() {
            @Override
            public CompletableFuture<HotelInfo> getHotelInfoFuture() {
                if (isTestHotel(context)) {
                    return CompletableFuture.completedFuture(getTestHotel());
                } else {
                    return geoSearchResponseFuture.thenApply(GeoSearchHotelContentService.this::mapHotelInfo);
                }
            }

            @Override
            public CompletableFuture<LegalInfo.LegalInfoItem> getHotelLegalItemFuture() {
                if (isTestHotel(context)) {
                    return CompletableFuture.completedFuture(getTestLegalData());
                }
                return geoSearchResponseFuture.thenApply(GeoSearchHotelContentService.this::mapHotelLegalInfo);
            }
        };
    }

    private CompletableFuture<GeoHotel> getHotelFromGeo(TravelToken travelToken) {
        EPartnerId partnerId = travelToken.getPartnerId();
        if (travelToken.getPartnerId() == EPartnerId.PI_UNUSED) {
            // fix for very old tokens
            partnerId = EPartnerId.PI_EXPEDIA;
        }
        TPartner partner = partnerConfigService.getByKey(partnerId);
        if (partner == null) {
            return CompletableFuture.failedFuture(new RuntimeException("Partner not found by id " + partnerId.toString()));
        }
        GeoHotelOriginalId geoHotelOriginalId = new GeoHotelOriginalId();
        geoHotelOriginalId.setPartnerCode(partner.getCode());
        geoHotelOriginalId.setOriginalId(travelToken.getOriginalId());
        GeoSearchReq geoSearchReq = GeoSearchReq.byOriginalId(GeoOriginEnum.ORDER_PAGE, geoHotelOriginalId, null);
        geoSearchReq.setLimit(1);
        geoSearchReq.setIncludeOfferCache(true);
        geoSearchReq.setIncludeRating(true);
        geoSearchReq.setIncludePhotos(true);
        geoSearchReq.setIncludeLegalInfo(true);
        geoSearchReq.setIncludeClosedHotels(true);
        return geoSearchService.query(geoSearchReq).thenApply(
                response -> {
                    if (response.getHotels().isEmpty()) {
                        return null;
                    } else {
                        return response.getHotels().get(0);
                    }
                }
        );
    }

    private boolean isTestHotel(BookingFlowContext context) {
        return properties.getTestHotels()
                .getOrDefault(context.getDecodedToken().getPartnerId().toString(), Collections.emptySet())
                .contains(context.getDecodedToken().getOriginalId());
    }

    private HotelInfo getTestHotel() {
        return HotelInfo.builder()
                .name("Тестовый отель")
                .address("Россия, г. Москва, ул Льва Толстого 16, подъезд 2, комната 2105")
                .coordinates(new Coordinates(37.587585, 55.734554))
                .imageUrlTemplate("https://avatars.mds.yandex" +
                        ".net/get-altay/223006/2a0000015b19222d131711047d866611eaf2/%s")
                .locationType(LocationType.MOSCOW)
                .stars(5)
                .permalink(1124715036)
                .rating(10.0f)
                .workingHours("Круглосуточно")
                .build();
    }

    private LegalInfo.LegalInfoItem getTestLegalData() {
        return LegalInfo.LegalInfoItem.builder()
                .name("Тестовый отель Яндекс Путешествий")
                .ogrn("1027700229193")
                .actualAddress("Россия, г. Москва, ул Льва Толстого 16, подъезд 2, комната 2105")
                .legalAddress("119021, Россия, г. Москва, ул. Льва Толстого, д. 16")
                .workingHours("Круглосуточно")
                .build();
    }

    @SuppressWarnings("Duplicates")
    private HotelInfo mapHotelInfo(GeoHotel geoHotelInfo) {
        if (geoHotelInfo == null || geoHotelInfo.getGeoObject() == null || geoHotelInfo.getGeoObjectMetadata() == null) {
            return null;
        }
        var builder = HotelInfo.builder()
                .name(geoHotelInfo.getGeoObjectMetadata().getName())
                .address(geoHotelInfo.getGeoObjectMetadata().getAddress().getFormattedAddress())
                .permalink(Long.parseUnsignedLong(geoHotelInfo.getGeoObjectMetadata().getId()))
                .stars(geoHotelInfo.getNumStars())
                .rating(geoHotelInfo.getRating() == null ? 0.0f : (geoHotelInfo.getRating().getScore() / 2.0f))
                .workingHours(geoHotelInfo.getGeoObjectMetadata().getOpenHours().getText());
        if (geoHotelInfo.getGeoObjectMetadata().getPhoneCount() > 0) {
            builder.phone(geoHotelInfo.getGeoObjectMetadata().getPhone(0).getFormatted());
        }
        if (geoHotelInfo.getGeoObject().getGeometryCount() > 0) {
            GeometryOuterClass.Geometry geometry = geoHotelInfo.getGeoObject().getGeometry(0);
            if (geometry.hasPoint()) {
                Coordinates c = new Coordinates(geometry.getPoint().getLon(), geometry.getPoint().getLat());
                builder.coordinates(c);
            }
        }
        if (geoHotelInfo.getPhotos() != null && geoHotelInfo.getPhotos().getPhotoCount() > 0) {
            builder.imageUrlTemplate(geoHotelInfo.getPhotos().getPhoto(0).getUrlTemplate());
        }
        builder.locationType(detectLocationType(geoHotelInfo));

        return builder.build();
    }

    private LocationType detectLocationType(GeoHotel hotel) {
        boolean isRussia = false;
        // TODO(alexcrush) replace with GeoBase lookup
        for (AddressOuterClass.Component comp : hotel.getGeoObjectMetadata().getAddress().getComponentList()) {
            if ("Россия".equals(comp.getName()) && comp.getKindList().stream().anyMatch(k -> k.equals(KindOuterClass.Kind.COUNTRY))) {
                isRussia = true;
            }
            if ("Москва".equals(comp.getName()) && comp.getKindList().stream().anyMatch(k -> k.equals(KindOuterClass.Kind.PROVINCE))) {
                return LocationType.MOSCOW;
            }
        }
        return isRussia ? LocationType.RUSSIA : LocationType.GLOBAL;
    }

    private LegalInfo.LegalInfoItem mapHotelLegalInfo(GeoHotel geoHotelInfo) {
        if (geoHotelInfo == null || geoHotelInfo.getGeoObjectMetadata() == null) {
            return null;
        }
        var builder = LegalInfo.LegalInfoItem.builder()
                .name(geoHotelInfo.getGeoObjectMetadata().getName())
                .actualAddress(geoHotelInfo.getGeoObjectMetadata().getAddress().getFormattedAddress())
                .workingHours(geoHotelInfo.getGeoObjectMetadata().getOpenHours().getText());
        if (geoHotelInfo.getLegalInfo() != null) {
            builder.legalAddress(geoHotelInfo.getLegalInfo().getAddress())
                    .ogrn(geoHotelInfo.getLegalInfo().getOgrn());
            if (Strings.isNotBlank(geoHotelInfo.getLegalInfo().getName())) {
                builder.name(geoHotelInfo.getLegalInfo().getName());
            }
        }
        return builder.build();
    }

    public interface GeoSearchFutures {
        CompletableFuture<HotelInfo> getHotelInfoFuture();

        CompletableFuture<LegalInfo.LegalInfoItem> getHotelLegalItemFuture();
    }
}
