package ru.yandex.travel.api.services.hotels_booking_flow;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.StreamSupport;

import javax.money.CurrencyUnit;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import io.grpc.Context;
import io.opentracing.Span;
import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.api.endpoints.booking_flow.model.OrderGuestInfoDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.OrderPriceInfo;
import ru.yandex.travel.api.endpoints.booking_flow.model.OrderStatus;
import ru.yandex.travel.api.endpoints.booking_flow.model.PaymentErrorCode;
import ru.yandex.travel.api.endpoints.booking_flow.model.PromoCodeApplicationResult;
import ru.yandex.travel.api.endpoints.booking_flow.model.RefundCalculation;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.AppliedPromoCampaignsDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.Mir2020PromoCampaignDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.PromoCampaignsDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.Taxi2020PromoCampaignDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.WhiteLabelCampaignDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.YandexPlusApplicationDto;
import ru.yandex.travel.api.endpoints.booking_flow.req_rsp.CancelOrderReqV1;
import ru.yandex.travel.api.endpoints.booking_flow.req_rsp.OrderStatusRspV1;
import ru.yandex.travel.api.exceptions.InputError;
import ru.yandex.travel.api.exceptions.InvalidInputException;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.infrastucture.ApiTokenEncrypter;
import ru.yandex.travel.api.models.common.MoneyMarkup;
import ru.yandex.travel.api.models.common.PromoCodeApplicationResultType;
import ru.yandex.travel.api.services.hotels.geobase.GeoBase;
import ru.yandex.travel.api.services.hotels.geobase.GeoBaseHelpers;
import ru.yandex.travel.api.services.hotels_booking_flow.models.CurrentPaymentInfo;
import ru.yandex.travel.api.services.hotels_booking_flow.models.EstimateDiscountResult;
import ru.yandex.travel.api.services.hotels_booking_flow.models.HotelOrder;
import ru.yandex.travel.api.services.hotels_booking_flow.models.NextPaymentInfo;
import ru.yandex.travel.api.services.hotels_booking_flow.models.OrderCreationData;
import ru.yandex.travel.api.services.hotels_booking_flow.models.PaymentErrorInfo;
import ru.yandex.travel.api.services.hotels_booking_flow.models.PaymentInfo;
import ru.yandex.travel.api.services.hotels_booking_flow.models.PaymentType;
import ru.yandex.travel.api.services.hotels_booking_flow.models.ReceiptItem;
import ru.yandex.travel.api.services.hotels_booking_flow.models.RefundInfo;
import ru.yandex.travel.api.services.hotels_booking_flow.promo.HotelPromoCampaignsService;
import ru.yandex.travel.api.services.hotels_booking_flow.promo.HotelPromoCampaignsServiceProperties;
import ru.yandex.travel.api.services.orders.Meters;
import ru.yandex.travel.api.services.orders.OrchestratorClientFactory;
import ru.yandex.travel.api.services.orders.OrderStatusMappingService;
import ru.yandex.travel.api.services.promo.YandexPlusService;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.KVExperiments;
import ru.yandex.travel.commons.experiments.OrderExperiments;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.grpc.AppCallIdGenerator;
import ru.yandex.travel.hotels.common.LocationType;
import ru.yandex.travel.hotels.common.orders.BaseRate;
import ru.yandex.travel.hotels.common.orders.GeoRegion;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.HotelUserInfo;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.orders.promo.AppliedPromoCampaigns;
import ru.yandex.travel.hotels.common.orders.promo.YandexPlusApplication;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.models.booking_flow.Coordinates;
import ru.yandex.travel.hotels.models.booking_flow.Offer;
import ru.yandex.travel.hotels.models.booking_flow.Rate;
import ru.yandex.travel.hotels.models.booking_flow.RateInfo;
import ru.yandex.travel.hotels.models.booking_flow.SearchInfo;
import ru.yandex.travel.hotels.models.booking_flow.promo.Mir2020PromoCampaign;
import ru.yandex.travel.hotels.models.booking_flow.promo.PromoCampaignsInfo;
import ru.yandex.travel.hotels.models.booking_flow.promo.YandexPlusPromoCampaign;
import ru.yandex.travel.hotels.proto.EMirEligibility;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TDeterminePromosForOfferRsp;
import ru.yandex.travel.hotels.proto.THotelTestContext;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.EPaymentType;
import ru.yandex.travel.orders.proto.ETaxi2020PromoStatusEnum;
import ru.yandex.travel.orders.proto.TCalculateRefundReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TEstimateDiscountReq;
import ru.yandex.travel.orders.proto.TEstimateDiscountRsp;
import ru.yandex.travel.orders.proto.TGenerateBusinessTripPdfReq;
import ru.yandex.travel.orders.proto.TGetOrderAggregateStateReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TMir2020PromoCampaignInfo;
import ru.yandex.travel.orders.proto.TOrderAggregateState;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TOrderPriceInfo;
import ru.yandex.travel.orders.proto.TPaymentInfo;
import ru.yandex.travel.orders.proto.TPromoCampaignsInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartCancellationReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.proto.TTaxi2020PromoCampaignInfo;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.orders.proto.TWhiteLabelPromoCampaignInfo;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.tracing.TracerHelpers;

import static java.util.stream.Collectors.toList;
import static ru.yandex.travel.api.services.common.PromoCampaignUtils.createPromoCampaignsDto;
import static ru.yandex.travel.commons.concurrent.FutureUtils.buildCompletableFuture;

@Service
@RequiredArgsConstructor
@Slf4j
@EnableConfigurationProperties({AggregateOrderStatusProperties.class, HotelOrdersServiceConfigurationProperties.class})
public class HotelOrdersService {
    // Will be replaced by global config in https://st.yandex-team.ru/HOTELS-4315
    private final static ImmutableMap<EPartnerId, String> partnerIdToProviderIdMap = ImmutableMap.of(
            EPartnerId.PI_EXPEDIA, "3501497177",
            EPartnerId.PI_DOLPHIN, "3501630980"
    );

    private final static Set<EPaymentState> PENDING_PAYMENT_STATES = Set.of(
            EPaymentState.PS_INVOICE_PENDING,
            EPaymentState.PS_PAYMENT_IN_PROGRESS,
            EPaymentState.PS_PARTIALLY_PAID);

    private final static Set<EPaymentState> NEXT_PENDING_PAYMENT_STATES = Set.of(
            EPaymentState.PS_DRAFT,
            EPaymentState.PS_INVOICE_PENDING,
            EPaymentState.PS_PAYMENT_IN_PROGRESS,
            EPaymentState.PS_PARTIALLY_PAID);


    private final OfferService service;
    private final OrchestratorClientFactory orchestratorClientFactory;
    private final OrderStatusMappingService statusMappingService;
    private final PartnerDispatcher partnerDispatcher;
    private final Meters meters;
    private final Tracer tracer;
    private final TimezoneDetector timezoneDetector;
    private final HotelPromoCampaignsService hotelPromoCampaignsService;
    private final AggregateOrderStatusProperties aggregateOrderStatusProperties;
    private final HotelOrdersServiceConfigurationProperties orderServiceProperties;
    private final HotelPromoCampaignsServiceProperties hotelPromoCampaignsProperties;
    private final PaymentScheduleService paymentScheduleService;
    private final GeoBase geoBase;
    private final ApiTokenEncrypter apiTokenEncrypter;
    private final YandexPlusService yandexPlusService;
    private final ExperimentDataProvider experimentDataProvider;


    public CompletableFuture<HotelOrder> createOrder(BookingFlowContext flowContext) {
        try {
            Preconditions.checkNotNull(flowContext.getOrderCreationData(), "CreateOrderRequest must be passed in " +
                    "context");
            Preconditions.checkArgument(!flowContext.getOrderCreationData().isUseDeferredPayments() || !flowContext.getOrderCreationData().isUsePostPay(),
                    "Cannot use deferred payment and post payment together");

            AppCallIdGenerator callIdGenerator = AppCallIdGenerator.KEY.get();
            Span span = tracer.activeSpan();
            CompletableFuture<Integer> yandexPlusBalanceFuture = yandexPlusService.getYandexPlusBalanceWithoutException(
                    flowContext.getUserCredentials().getPassportId(),
                    flowContext.getUserCredentials().getUserIp(),
                    YandexPlusService.DEFAULT_PLUS_CURRENCY
            );

            return yandexPlusBalanceFuture
                    .thenCompose(yandexPlusBalance -> service.getOffer(flowContext, yandexPlusBalance)
                            .thenApply(offer -> {
                                validateAppliedPromoCampaigns(offer, flowContext.getAppliedPromoCampaigns());
                                return offer;
                            })
                            .thenCompose(offer -> flowContext.getTestContextFuture()
                                    .thenCompose(TracerHelpers.applyWithActiveSpanScope(
                                            tracer,
                                            span,
                                            testContext -> createOrderImpl(
                                                    offer,
                                                    flowContext,
                                                    testContext,
                                                    callIdGenerator,
                                                    yandexPlusBalance)))
                            ));
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private YandexPlusApplicationDto getYandexPlusApplication(AppliedPromoCampaignsDto appliedPromoCampaigns) {
        return appliedPromoCampaigns != null ? appliedPromoCampaigns.getYandexPlus() : null;
    }

    // TODO TRAVELBACK-2701 checking logic had migrated to orders and that method can be simplified or completely removed
    private void validateAppliedPromoCampaigns(Offer offer, AppliedPromoCampaignsDto appliedPromoCampaigns) {
        YandexPlusApplicationDto yandexPlusApplication = getYandexPlusApplication(appliedPromoCampaigns);
        if (yandexPlusApplication != null) {
            Preconditions.checkArgument(offer.getPromoCampaignsInfo() != null,
                    "No offer promo campaigns at all, can't validate Yandex Plus application");
            YandexPlusPromoCampaign yandexPlus = offer.getPromoCampaignsInfo().getYandexPlus();
            Preconditions.checkArgument(yandexPlus != null, "Yandex Plus is not available for this offer");
            Preconditions.checkArgument(yandexPlus.getEligible() == Boolean.TRUE, "Yandex Plus is not eligible");
            Preconditions.checkArgument(yandexPlusApplication.getPoints() != null,
                    "Yandex Plus application without points");
            switch (yandexPlusApplication.getMode()) {
                case TOPUP:
                    // todo(tlg-13): temporary w/a: we have to estimate discount once again
                    // and then get an updated plus offer
                    //Preconditions.checkArgument(Objects.equals(
                    //                yandexPlusApplication.getPoints(), yandexPlus.getPoints()),
                    Preconditions.checkArgument(yandexPlusApplication.getPoints() <= yandexPlus.getPoints(),
                            "Unexpected Plus points promise to be added to user account: " +
                                    "promised %s, allowed by terms %s",
                            yandexPlusApplication.getPoints(), yandexPlus.getPoints());
                    break;
                case WITHDRAW:
                    Preconditions.checkArgument(hotelPromoCampaignsProperties.getYandexPlusWithdrawEnabled(),
                            "Yandex Plus withdraw operations aren't enabled yet");
                    Preconditions.checkArgument(yandexPlusApplication.getPoints() < offer.calculateActualPrice().getNumber().intValue(),
                            "The offer can be fully paid with Yandex Plus; {} plus points -vs- offer for {}",
                            yandexPlusApplication.getPoints(), offer.calculateActualPrice().getNumber().intValue());
                    break;
                default:
                    throw new RuntimeException("Unsupported Yandex Plus application mode: " +
                            yandexPlusApplication.getMode());
            }
        }
    }

    public CompletableFuture<OrderStatusRspV1> getOrderStatus(String orderId) {
        if (aggregateOrderStatusProperties.isEnabled()) {
            return FutureUtils.buildCompletableFuture(orchestratorClientFactory.createFutureStubForHotels().getOrderAggregateState(
                    TGetOrderAggregateStateReq.newBuilder()
                            .setOrderId(orderId)
                            .setCalculateAggregateStateOnTheFly(aggregateOrderStatusProperties.isCalculateOnTheFly())
                            .build()
            )).thenApply(aggStatusRsp -> getOrderStatusFromAggregate(aggStatusRsp.getOrderAggregateState()));
        } else {
            return FutureUtils.buildCompletableFuture(
                    orchestratorClientFactory.createFutureStubForHotels()
                            .getOrderInfo(
                                    TGetOrderInfoReq.newBuilder()
                                            .setCalculateAggregateStateOnTheFly(aggregateOrderStatusProperties.isCalculateOnTheFly())
                                            .setOrderId(orderId).build()
                            )
            ).thenApply(orderRsp -> getOrderStatusFromProto(orderRsp.getResult()));
        }
    }

    public CompletableFuture<EstimateDiscountResult> estimateDiscount(BookingFlowContext flowContext) {
        try {
            AppCallIdGenerator callIdGenerator = AppCallIdGenerator.KEY.get();
            Span span = tracer.activeSpan();
            CompletableFuture<Integer> yandexPlusBalanceFuture = yandexPlusService.getYandexPlusBalanceWithoutException(
                    flowContext.getUserCredentials().getPassportId(),
                    flowContext.getUserCredentials().getUserIp(),
                    YandexPlusService.DEFAULT_PLUS_CURRENCY
            );

            return yandexPlusBalanceFuture
                    .thenCompose(yandexPlusBalance -> service.getOffer(flowContext, yandexPlusBalance)
                            .thenApply(offer -> {
                                validateAppliedPromoCampaigns(offer, flowContext.getAppliedPromoCampaigns());
                                return offer;
                            })
                            .thenCompose(offer -> flowContext.getTestContextFuture()
                                    .thenCompose(TracerHelpers.applyWithActiveSpanScope(
                                            tracer,
                                            span,
                                            testContext -> estimateDiscountImpl(
                                                    offer,
                                                    flowContext,
                                                    callIdGenerator,
                                                    yandexPlusBalance)))
                            ));
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public CompletableFuture<HotelOrder> getOrderById(String id) {
        return buildCompletableFuture(orchestratorClientFactory.createFutureStubForHotels()
                .getOrderInfo(TGetOrderInfoReq.newBuilder()
                        .setCalculateAggregateStateOnTheFly(aggregateOrderStatusProperties.isCalculateOnTheFly())
                        .setOrderId(id)
                        .build()))
                .thenApply(res -> this.getOrderFromProto(res.getResult()));
    }

    public CompletableFuture<RefundCalculation> calculateRefund(String id) {
        return buildCompletableFuture(orchestratorClientFactory.createFutureStubForHotels()
                .calculateRefund(
                        TCalculateRefundReq.newBuilder().setOrderId(id)
                                .setCallId(callIdForMethod("calculateRefund"))
                                .build()
                )
        ).thenApply(response -> {
            RefundCalculation res = new RefundCalculation();
            res.setRefund(BaseRate.fromTPrice(response.getRefundAmount()));
            res.setPenalty(BaseRate.fromTPrice(response.getPenaltyAmount()));
            res.setValidUntil(ProtoUtils.toLocalDateTime(response.getExpiresAt()));
            res.setRefundToken(response.getRefundToken());
            return res;
        });
    }

    public CompletableFuture<Void> startRefund(String id, String refundToken) {
        return buildCompletableFuture(orchestratorClientFactory.createFutureStubForHotels()
                .startRefund(TStartRefundReq.newBuilder()
                        .setCallId(callIdForMethod("startRefund"))
                        .setOrderId(id)
                        .setRefundToken(refundToken).build()))
                .thenApply(ignoredResponse -> {
                    meters.getHotelOrdersRefunded().increment();
                    return null;
                });
    }

    public CompletableFuture<Void> cancelOrder(CancelOrderReqV1 req) {
        return buildCompletableFuture(orchestratorClientFactory.createFutureStubForHotels()
                .startCancellation(
                        TStartCancellationReq.newBuilder()
                                .setCallId(callIdForMethod("cancelOrder"))
                                .setOrderId(req.getId()).build()
                )
        ).thenApply(ignored -> null);
    }

    public CompletableFuture<Void> payOrder(String orderId, String returnUrl, String customerSource,
                                            String paymentTestContextToken, OrderExperiments experiments) {
        if (StringUtils.isBlank(customerSource)) {
            customerSource = "desktop";
        }
        TStartPaymentReq.Builder reqBuilder = TStartPaymentReq.newBuilder()
                .setInvoiceType(EInvoiceType.IT_TRUST)
                .setCallId(callIdForMethod("startPayment"))
                .setOrderId(orderId)
                .setReturnUrl(returnUrl)
                .setSource(customerSource)
                .setUseNewInvoiceModel(orderServiceProperties.isUseNewPayments())
                .setNewCommonPaymentWebForm(experiments.isNewCommonPaymentWebForm());
        if (!Strings.isNullOrEmpty(paymentTestContextToken)) {
            reqBuilder.setPaymentTestContext(apiTokenEncrypter.fromPaymentTestContextToken(paymentTestContextToken));
        }
        return buildCompletableFuture(orchestratorClientFactory.createFutureStubForHotels()
                .startPayment(reqBuilder
                        .build()))
                .thenApply(ignored -> null);
    }

    public CompletableFuture<Void> payInvoice(String orderId, String invoiceId, String returnUrl,
                                              String customerSource, String paymentTestContextToken,
                                              OrderExperiments experiments) {
        Preconditions.checkState(orderServiceProperties.isUseNewPayments(), "New payments model is not enabled");
        if (StringUtils.isBlank(customerSource)) {
            customerSource = "desktop";
        }
        TStartPaymentReq.Builder reqBuilder = TStartPaymentReq.newBuilder()
                .setInvoiceType(EInvoiceType.IT_TRUST)
                .setCallId(callIdForMethod("startInvoicePayment"))
                .setOrderId(orderId)
                .setInvoiceId(invoiceId)
                .setReturnUrl(returnUrl)
                .setSource(customerSource)
                .setUseNewInvoiceModel(true)
                .setNewCommonPaymentWebForm(experiments.isNewCommonPaymentWebForm());
        if (!Strings.isNullOrEmpty(paymentTestContextToken)) {
            reqBuilder.setPaymentTestContext(apiTokenEncrypter.fromPaymentTestContextToken(paymentTestContextToken));
        }
        return buildCompletableFuture(orchestratorClientFactory.createFutureStubForHotels()
                .startPayment(reqBuilder.build()))
                .thenApply(ignored -> null);
    }

    public CompletableFuture<Void> generateBusinessTripPdf(String orderId) {
        return buildCompletableFuture(orchestratorClientFactory.createFutureStubForHotels()
                .generateBusinessTripPdf(TGenerateBusinessTripPdfReq.newBuilder()
                        .setOrderId(orderId)
                        .setCallId(callIdForMethod("generateBusinessTripPdf"))
                        .build()))
                .thenApply(ignored -> null);
    }

    private CompletableFuture<HotelOrder> createOrderImpl(Offer offer, BookingFlowContext flowContext,
                                                          THotelTestContext testContext,
                                                          AppCallIdGenerator callIdGenerator,
                                                          Integer yandexPlusBalance) {
        Span span = tracer.activeSpan();
        switch (offer.checkOfferState()) {
            case READY:
                if (offer.getMetaInfo().getCheckSum().equals(flowContext.getOrderCreationData().getChecksum())) {
                    log.debug("Offer is up to date, creating order for it");
                    boolean usePostPay = flowContext.getOrderCreationData().isUsePostPay();
                    return createItinerary(offer, flowContext, true, yandexPlusBalance, usePostPay).thenCompose(itinerary -> {
                        try {
                            return Context.current().withValue(UserCredentials.KEY,
                                    flowContext.getUserCredentials()).call(() -> {
                                boolean useDeferredPayment = offer.getDeferredPaymentSchedule() != null &&
                                        offer.getDeferredPaymentSchedule().getDeferredPayments().size() > 0 &&
                                        flowContext.getOrderCreationData().isUseDeferredPayments();
                                var request = createOrderReq(itinerary, flowContext, testContext,
                                        useDeferredPayment, usePostPay);
                                return buildCompletableFuture(orchestratorClientFactory.createFutureStubForHotels()
                                        .createOrder(request))
                                        .thenApply(rsp -> getOrderFromProto(rsp.getNewOrder()))
                                        .thenCompose(TracerHelpers.applyWithActiveSpanScope(tracer,
                                                span,
                                                o -> startReservation(o, callIdGenerator)
                                        ))
                                        .whenComplete((hotelOrder, t) -> {
                                            if (t == null) {
                                                meters.getHotelOrdersCreated().increment();
                                            }
                                        });
                            });
                        } catch (Exception e) {
                            return CompletableFuture.failedFuture(e);
                        }
                    });
                } else {
                    log.warn("Content checksum mismatch");
                    meters.getOfferMeters(flowContext.getStage(), flowContext.getProvider().getPartnerId())
                            .getContentMismatch().increment();
                    return CompletableFuture.completedFuture(
                            emptyOrderBuilder(offer, flowContext).status(OrderStatus.FAILED).build());
                }
            case PRICE_CONFLICT:
                log.debug("Price conflict in offer, unable to create order");
                return CompletableFuture.completedFuture(
                        emptyOrderBuilder(offer, flowContext).status(OrderStatus.FAILED).build());
            case MISSING_DATA:
                log.debug("Missing content in offer, unable to create order");
                return CompletableFuture.completedFuture(
                        emptyOrderBuilder(offer, flowContext).status(OrderStatus.FAILED).build());
            default:
                throw new AssertionError("We should not be here");
        }
    }

    private CompletableFuture<EstimateDiscountResult> estimateDiscountImpl(Offer offer,
                                                                           BookingFlowContext flowContext,
                                                                           AppCallIdGenerator callIdGenerator,
                                                                           Integer yandexPlusBalance) {
        switch (offer.checkOfferState()) {
            case READY:
                if (offer.getMetaInfo().getCheckSum().equals(flowContext.getOrderCreationData().getChecksum())) {
                    log.debug("Offer is up to date, creating order for it");
                    return createItinerary(offer, flowContext, false, yandexPlusBalance, false).thenCompose(itinerary -> {
                        Context.current().withValue(UserCredentials.KEY, flowContext.getUserCredentials()).attach();

                        TEstimateDiscountReq estimateDiscountReq = TEstimateDiscountReq.newBuilder()
                                .addService(TEstimateDiscountReq.TEstimateDiscountService.newBuilder()
                                        .setServiceType(flowContext.getProvider().getServiceType())
                                        .setSourcePayload(ProtoUtils.toTJson(itinerary))
                                        .build())
                                .addAllPromoCode(flowContext.getPromoCodes())
                                .addAllKVExperiments(KVExperiments.toProto(flowContext.getHeaders().getExperiments()))
                                .build();

                        return buildCompletableFuture(orchestratorClientFactory.createFutureStubForHotels()
                                .estimateDiscount(estimateDiscountReq))
                                .thenApply(r -> getEstimateDiscountFromProto(offer, r, flowContext, yandexPlusBalance));
                    });
                } else {
                    log.warn("Content checksum mismatch");
                    meters.getOfferMeters(flowContext.getStage(), flowContext.getProvider().getPartnerId())
                            .getContentMismatch().increment();
                    return CompletableFuture.failedFuture(new TravelApiBadRequestException("Content checksum mismatch"));
                }
            case PRICE_CONFLICT:
                log.debug("Price conflict in offer, unable to estimate discount");
                return CompletableFuture.failedFuture(new TravelApiBadRequestException("Price conflict in offer, " +
                        "unable to estimate discount"));
            case MISSING_DATA:
                log.debug("Missing content in offer, unable to estimate discount");
                return CompletableFuture.failedFuture(new TravelApiBadRequestException("Missing content in offer, " +
                        "unable to estimate discount"));
            default:
                throw new AssertionError("We should not be here");
        }
    }

    private HotelOrder.HotelOrderBuilder emptyOrderBuilder(Offer offer, BookingFlowContext context) {
        OrderGuestInfoDto orderGuestInfo = new OrderGuestInfoDto();
        List<Guest> guests = getGuests(offer, context.getOrderCreationData());

        orderGuestInfo.setAllowsSubscription(context.getOrderCreationData().isAllowsSubscription());
        orderGuestInfo.setCustomerEmail(context.getOrderCreationData().getCustomerEmail());
        orderGuestInfo.setCustomerPhone(context.getOrderCreationData().getCustomerPhone());
        orderGuestInfo.setGuests(guests);

        return HotelOrder.builder()
                .offerInfo(offer)
                .createdAt(LocalDateTime.now())
                .guestInfo(orderGuestInfo);
    }

    private CompletableFuture<HotelOrder> startReservation(HotelOrder order, AppCallIdGenerator callIdGenerator) {
        return buildCompletableFuture(
                orchestratorClientFactory.createFutureStubForHotels().reserve(
                        TReserveReq.newBuilder().setOrderId(order.getId().toString())
                                .setCallId(callIdGenerator.generate("reserve"))
                                .build()
                )).thenApply(ignored -> order);
    }

    private OrderStatusRspV1 getOrderStatusFromProto(TOrderInfo orderInfo) {
        OrderStatusRspV1 result = new OrderStatusRspV1();
        if (orderInfo.hasCurrentInvoice()) {
            if (!Strings.isNullOrEmpty(orderInfo.getCurrentInvoice().getPaymentUrl())) {
                result.setPaymentUrl(orderInfo.getCurrentInvoice().getPaymentUrl());
            }
            if (!Strings.isNullOrEmpty(orderInfo.getCurrentInvoice().getAuthorizationErrorCode())) {
                result.setPaymentError(PaymentErrorCode.fromErrorCode(orderInfo.getCurrentInvoice().getAuthorizationErrorCode()));
            }
        }
        result.setStatus(
                statusMappingService.getStatus(orderInfo));
        return result;
    }

    private OrderStatusRspV1 getOrderStatusFromAggregate(TOrderAggregateState orderAggregateState) {
        OrderStatusRspV1 result = new OrderStatusRspV1();
        result.setStatus(OrderStatus.fromAggregateState(orderAggregateState.getHotelOrderAggregateState()));
        result.setPaymentUrl(Strings.emptyToNull(orderAggregateState.getPaymentUrl()));
        result.setPaymentError(PaymentErrorCode.BY_PROTO.getByValueOrNull(orderAggregateState.getPaymentError()));
        return result;
    }

    @SuppressWarnings("Duplicates")
    public HotelOrder getOrderFromProto(TOrderInfo protoOrder) {
        var serviceTypes =
                partnerDispatcher.map(PartnerBookingProvider::getServiceType).collect(Collectors.toSet());
        var services = protoOrder.getServiceList().stream()
                .filter(s -> serviceTypes.contains(s.getServiceType()))
                .collect(toList());
        if (services.size() == 0) {
            String message = String.format("No hotel services in order '%s'", protoOrder.getOrderId());
            log.warn(message);
            throw new RuntimeException(message);
        }
        if (services.size() > 1) {
            String message = String.format("Too many hotel services in order '%s'", protoOrder.getOrderId());
            log.warn(message);
            throw new RuntimeException(message);
        }
        HotelItinerary itinerary = ProtoUtils.fromTJson(services.get(0).getServiceInfo().getPayload(),
                HotelItinerary.class);
        Offer offer = Offer.fromJsonNode(itinerary.getUiPayload());
        OrderStatus status;
        if (aggregateOrderStatusProperties.isEnabled()) {
            status = OrderStatus.fromAggregateState(protoOrder.getOrderAggregateState().getHotelOrderAggregateState());
        } else {
            status = statusMappingService.getStatus(protoOrder);
        }

        OrderGuestInfoDto orderGuestInfo = new OrderGuestInfoDto();

        orderGuestInfo.setAllowsSubscription(itinerary.isAllowsSubscription());
        orderGuestInfo.setCustomerEmail(itinerary.getCustomerEmail());
        orderGuestInfo.setCustomerPhone(itinerary.getCustomerPhone());
        orderGuestInfo.setGuests(itinerary.getGuests().stream()
                .map(g -> {
                    Guest guest = new Guest();

                    guest.setFirstName(g.getFirstName());
                    guest.setLastName(g.getLastName());
                    guest.setAge(g.getAge());
                    guest.setChild(g.isChild());

                    return guest;
                })
                .collect(toList()));

        var orderBuilder = HotelOrder.builder()
                .id(UUID.fromString(protoOrder.getOrderId()))
                .prettyId(protoOrder.getPrettyId())
                .offerInfo(offer)
                .status(status)
                .payment(createPaymentInfo(protoOrder, status, offer.getRefundRules()))
                .guestInfo(orderGuestInfo)
                .displayState(protoOrder.getDisplayOrderState());

        orderBuilder.canGenerateBusinessTripDoc(protoOrder.getCanGenerateBusinessTripDoc());
        if (itinerary.getConfirmation() != null) {
            if (StringUtils.isNotBlank(itinerary.getConfirmation().getHotelConfirmationId())) {
                orderBuilder.confirmationId(itinerary.getConfirmation().getHotelConfirmationId());
            } else {
                orderBuilder.confirmationId(itinerary.getConfirmation().getPartnerConfirmationId());
            }
            if (StringUtils.isNotBlank(protoOrder.getDocumentUrl())) {
                orderBuilder.documentUrl(protoOrder.getDocumentUrl());
            }
            if (StringUtils.isNotBlank(protoOrder.getBusinessTripDocUrl())) {
                orderBuilder.businessTripDocUrl(protoOrder.getBusinessTripDocUrl());
            }
        }
        if (itinerary.getRefundInfo() != null) {
            orderBuilder.refundInfo(RefundInfo.builder()
                    .refundDateTime(itinerary.getRefundInfo().getRefundDateTime())
                    .penalty(new Rate(itinerary.getRefundInfo().getPenalty().getAmount(),
                            itinerary.getRefundInfo().getPenalty().getCurrency()))
                    .refund(new Rate(itinerary.getRefundInfo().getRefund().getAmount(),
                            itinerary.getRefundInfo().getRefund().getCurrency()))
                    .penaltyIntervalIndex(itinerary.getRefundInfo().getPenaltyIntervalIndex())
                    .reason(itinerary.getRefundInfo().getReason())
                    .build());
        }

        if (itinerary.getOrderCancellationDetails() != null) {
            orderBuilder.orderCancellationDetails(itinerary.getOrderCancellationDetails());
        }
        orderBuilder.createdAt(ProtoUtils.toLocalDateTime(protoOrder.getCreatedAt()));

        if (protoOrder.hasPriceInfo()) {
            orderBuilder.orderPriceInfo(getOrderPriceInfoFromProto(protoOrder.getPriceInfo()));
        }
        orderBuilder.appliedPromoCampaigns(itinerary.getAppliedPromoCampaigns());
        return orderBuilder.build();
    }

    private OrderPriceInfo getOrderPriceInfoFromProto(TOrderPriceInfo protoOrderPrice) {
        OrderPriceInfo orderPriceInfo = new OrderPriceInfo();
        orderPriceInfo.setOriginalPrice(ProtoUtils.fromTPrice(protoOrderPrice.getOriginalPrice()));
        orderPriceInfo.setDiscountAmount(ProtoUtils.fromTPrice(protoOrderPrice.getDiscountAmount()));
        orderPriceInfo.setPrice(ProtoUtils.fromTPrice(protoOrderPrice.getPrice()));

        if (protoOrderPrice.hasPromoCampaignsInfo()) {
            orderPriceInfo.setPromoCampaigns(getPromoCampaignsFromProto(protoOrderPrice.getPromoCampaignsInfo()));
        }

        if (protoOrderPrice.getPromoCodeApplicationResultsCount() > 0) {
            orderPriceInfo.setPromoCodeApplicationResults(new ArrayList<>());
            protoOrderPrice.getPromoCodeApplicationResultsList().forEach(
                    ar -> orderPriceInfo.getPromoCodeApplicationResults().add(fromProtoCodeApplicationResult(ar))
            );

        }
        return orderPriceInfo;
    }

    private PromoCampaignsDto getPromoCampaignsFromProto(TPromoCampaignsInfo promoCampaignsInfo) {
        var promoCampaignsDto = new PromoCampaignsDto();

        TTaxi2020PromoCampaignInfo taxi2020 = promoCampaignsInfo.getTaxi2020();
        promoCampaignsDto.setTaxi2020(new Taxi2020PromoCampaignDto());
        promoCampaignsDto.getTaxi2020().setEligible(taxi2020.getStatus() == ETaxi2020PromoStatusEnum.OTPS_ELIGIBLE);

        if (promoCampaignsInfo.hasMir2020()) {
            TMir2020PromoCampaignInfo mir2020 = promoCampaignsInfo.getMir2020();
            Mir2020PromoCampaignDto mirDto = new Mir2020PromoCampaignDto();
            promoCampaignsDto.setMir2020(mirDto);
            if (mir2020.getEligible()) {
                mirDto.setEligible(true);
                mirDto.setCashbackAmount(mir2020.getCashbackAmount().getValue());
            } else {
                mirDto.setEligible(false);
            }
        }

        if (promoCampaignsInfo.hasWhiteLabel()) {
            TWhiteLabelPromoCampaignInfo whiteLabel = promoCampaignsInfo.getWhiteLabel();
            WhiteLabelCampaignDto.WhiteLabelCampaignDtoBuilder dtoBuilder = WhiteLabelCampaignDto.builder()
                    .eligible(whiteLabel.getEligible());
            if (whiteLabel.getEligible()) {
                dtoBuilder.points(WhiteLabelCampaignDto.WhiteLabelPoints.builder()
                        .pointsType(whiteLabel.getPoints().getPointsType())
                        .amount(whiteLabel.getPoints().getAmount())
                        .pointsName(whiteLabel.getPointsLinguistics().getNameForNumeralNominative())
                        .build());
            }
            promoCampaignsDto.setWhiteLabel(dtoBuilder.build());
        }
        return promoCampaignsDto;
    }


    private EstimateDiscountResult getEstimateDiscountFromProto(
            Offer offer, TEstimateDiscountRsp rsp, BookingFlowContext context, Integer yandexPlusBalance
    ) {
        Money originalAmount = rateInfoOrNull(offer.getRateInfo());
        Money prePromoDiscountedAmount = ProtoUtils.fromTPrice(rsp.getTotalCost());
        Money afterPromoDiscountedAmount = ProtoUtils.fromTPrice(rsp.getDiscountedCost());

        Money discountAmount = ProtoUtils.fromTPrice(rsp.getDiscountAmount());
        if (offer.getDiscountInfo() != null) {
            discountAmount = discountAmount.add(offer.getDiscountInfo().getDiscountAmount());
        }

        EstimateDiscountResult result = new EstimateDiscountResult();
        result.setOriginalAmount(originalAmount);
        result.setDiscountedAmount(afterPromoDiscountedAmount);
        result.setDiscountAmount(discountAmount);

        CheckParamsProvider params = CheckParamsRequest.buildFromContextAndSearchInfo(originalAmount,
                prePromoDiscountedAmount,
                afterPromoDiscountedAmount,
                context, offer.getMetaInfo().getSearch());

        TDeterminePromosForOfferRsp commonCampaigns = rsp.getPromosForOffer(0).getPromoInfo();

        // todo(tlg-13,alexcrush): this business logic should be moved to the orchestrator
        // (plus application as well as offer.getDiscountInfo() application the above code)
        // we should only copy the pre-calculated values to the result DTOs here

        YandexPlusPromoCampaign yandexPlus = hotelPromoCampaignsService.getYandexPlusCampaign(commonCampaigns, params, yandexPlusBalance);
        YandexPlusApplicationDto yandexPlusDto = getYandexPlusApplication(context.getAppliedPromoCampaigns());
        applyPlusPointsWithdraw(result, yandexPlusDto, yandexPlus);

        PromoCampaignsInfo promoCampaignsInfo = hotelPromoCampaignsService.calculatePromoCampaignsInfo(
                commonCampaigns,
                params,
                yandexPlus,
                result.getDiscountedAmount());
        result.setPromoCampaigns(createPromoCampaignsDto(promoCampaignsInfo));

        result.setCodeApplicationResults(new ArrayList<>());
        Mir2020PromoCampaign mir2020 = promoCampaignsInfo.getMir2020();
        if ((mir2020 == null || mir2020.getEligibility() != EMirEligibility.ME_ELIGIBLE)
                && (yandexPlusDto == null || yandexPlusDto.getMode() != YandexPlusApplicationDto.DtoMode.WITHDRAW)) {
            // deferred payments are disabled in case of MIR and Yandex Plus (withdraw) eligibility
            result.setDeferredPaymentSchedule(paymentScheduleService.getDeferredSchedule(
                    afterPromoDiscountedAmount, originalAmount, offer.getRefundRules(),
                    offer.getHotelInfo(),
                    context, FutureUtils.joinCompleted(context.getTestContextFuture())));
        }

        rsp.getPromoCodeApplicationResultsList().forEach(
                v -> result.getCodeApplicationResults().add(fromProtoCodeApplicationResult(v))
        );

        return result;
    }

    private void applyPlusPointsWithdraw(EstimateDiscountResult result, YandexPlusApplicationDto plusApplication,
                                         YandexPlusPromoCampaign plusTerms) {
        if (plusApplication == null ||
                plusApplication.getMode() != YandexPlusApplicationDto.DtoMode.WITHDRAW ||
                plusTerms.getWithdrawPoints() == null
        ) {
            return;
        }
        // this amount can be different from the previously chosen application as promo codes change the price base
        Integer effectiveWithdrawPoints = plusTerms.getWithdrawPoints();
        Money currentPrice = result.getDiscountedAmount();
        Money plusMoney = Money.of(effectiveWithdrawPoints, currentPrice.getCurrency());
        Money realMoney = currentPrice.subtract(plusMoney);
        result.setDiscountPlusPoints(effectiveWithdrawPoints);
        result.setDiscountedAmount(realMoney);
    }

    private PromoCodeApplicationResult fromProtoCodeApplicationResult
            (ru.yandex.travel.orders.proto.TPromoCodeApplicationResult v) {
        PromoCodeApplicationResult p = new PromoCodeApplicationResult();
        p.setCode(v.getCode());
        if (v.hasDiscountAmount()) {
            p.setDiscountAmount(ProtoUtils.fromTPrice(v.getDiscountAmount()));
        }
        p.setType(PromoCodeApplicationResultType.fromProto(v.getType()));
        return p;
    }

    private PaymentInfo createPaymentInfo(TOrderInfo protoOrder, OrderStatus status, RefundRules refundRules) {
        if (protoOrder.getPaymentsCount() == 0) {  // legacy case {
            return createLegacyPaymentInfo(protoOrder, status);
        }

        CurrencyUnit currency = ProtoUtils.fromTPrice(protoOrder.getPriceInfo().getPrice()).getCurrency();
        PaymentType paymentType = null;
        boolean mayBeStarted;
        CurrentPaymentInfo currentPaymentInfo = null;
        NextPaymentInfo nextPaymentInfo = null;

        PaymentErrorInfo errorInfo = null;
        TPaymentInfo pendingPayment = protoOrder.getPaymentsList().stream()
                .filter(p -> PENDING_PAYMENT_STATES.contains(p.getState()))
                .findFirst()
                .orElse(null);
        TPaymentInfo nextPendingPayment = null;
        if (pendingPayment != null) {
            switch (pendingPayment.getType()) {
                case PT_PENDING_INVOICE:
                    if (pendingPayment.equals(protoOrder.getPayments(0))) {
                        paymentType = PaymentType.FULL;
                    } else {
                        paymentType = PaymentType.EXTRA;
                    }
                    break;
                case PT_PAYMENT_SCHEDULE:
                    nextPendingPayment =
                            pendingPayment.getNextPaymentsList().stream()
                                    .filter(p -> NEXT_PENDING_PAYMENT_STATES.contains(p.getState()))
                                    .findFirst()
                                    .orElse(null);
                    if (pendingPayment.getState() != EPaymentState.PS_PARTIALLY_PAID) {
                        paymentType = PaymentType.INITIAL;
                    } else {
                        paymentType = PaymentType.DEFERRED;
                        pendingPayment = nextPendingPayment;
                        nextPendingPayment = null;
                    }
                    break;
            }
        }

        switch (status) {
            case RESERVED:
            case RESERVED_WITH_RESTRICTIONS:
                mayBeStarted = true;
                nextPendingPayment = null;
                break;
            case AWAITS_PAYMENT:
                Preconditions.checkState(pendingPayment != null, "No pending payment");
                mayBeStarted = false;
                currentPaymentInfo = CurrentPaymentInfo.builder()
                        .amount(ProtoUtils.fromTPrice(pendingPayment.getTotalAmount()))
                        .paymentType(paymentType)
                        .paymentUrl(pendingPayment.getActivePaymentUrl())
                        .build();
                break;
            case PAYMENT_FAILED:
                mayBeStarted = (protoOrder.getHotelOrderState() != EHotelOrderState.OS_CANCELLED);
                TPaymentInfo lastErrorPayment = protoOrder.getPaymentsList().stream()
                        .filter(p -> !Strings.isNullOrEmpty(p.getActivePaymentErrorCode()))
                        .reduce(null, (p, n) -> n);
                if (lastErrorPayment != null) {
                    if (lastErrorPayment.getType() == EPaymentType.PT_PAYMENT_SCHEDULE && lastErrorPayment.getState() == EPaymentState.PS_PARTIALLY_PAID) {
                        lastErrorPayment = lastErrorPayment.getNextPaymentsList().stream()
                                .filter(p -> !Strings.isNullOrEmpty(p.getActivePaymentErrorCode()))
                                .reduce(null, (p, n) -> n);
                    }
                    errorInfo = PaymentErrorInfo.builder()
                            .amount(ProtoUtils.fromTPrice(lastErrorPayment.getTotalAmount()))
                            .amountMarkup(MoneyMarkup.fromProto(lastErrorPayment.getTotalAmountMarkup()))
                            .code(PaymentErrorCode.fromErrorCode(lastErrorPayment.getActivePaymentErrorCode()))
                            .build();
                }
                nextPendingPayment = pendingPayment;
                break;
            case CONFIRMED:
                if (pendingPayment != null) {
                    mayBeStarted = true;
                    nextPendingPayment = pendingPayment;
                } else {
                    mayBeStarted = false;
                }
                break;
            case REFUNDED:
            case CANCELLED:
            case CANCELLED_WITH_REFUND:
                nextPendingPayment = null;
                mayBeStarted = false;
                break;
            default:
                return null;
        }
        if (nextPendingPayment != null) {
            Instant nextEndsAt = null;
            Money penalty = null;
            if (nextPendingPayment.hasPaymentEndsAt()) {
                nextEndsAt = ProtoUtils.toInstant(nextPendingPayment.getPaymentEndsAt());
                if (paymentType == PaymentType.DEFERRED || status == OrderStatus.AWAITS_PAYMENT) {
                    if (!refundRules.isFullyRefundableAt(nextEndsAt)) {
                        penalty = refundRules.getRuleAtInstant(nextEndsAt).getPenalty();
                    } else {
                        penalty = Money.zero(currency);
                    }
                }
            }
            nextPaymentInfo = NextPaymentInfo.builder()
                    .penaltyIfUnpaid(penalty)
                    .paymentEndsAt(nextEndsAt)
                    .amount(ProtoUtils.fromTPrice(nextPendingPayment.getTotalAmount()))
                    .build();
        }
        return PaymentInfo.builder()
                .usesDeferredPayment(protoOrder.getUsesDeferredPayment())
                .usesZeroFirstPayment(protoOrder.getZeroFirstPayment())
                .error(errorInfo)
                .mayBeStarted(mayBeStarted)
                .mayBeCancelled(protoOrder.getHotelOrderState() == EHotelOrderState.OS_WAITING_PAYMENT &&
                        (status == OrderStatus.AWAITS_PAYMENT || status == OrderStatus.PAYMENT_FAILED))
                .current(currentPaymentInfo)
                .next(nextPaymentInfo)
                .amountPaid(protoOrder.getPaymentsList().stream()
                        .filter(TPaymentInfo::hasPaidAmount)
                        .map(p -> ProtoUtils.fromTPrice(p.getPaidAmount()))
                        .reduce(Money::add).orElse(Money.zero(currency))
                )
                .receipts(protoOrder.getPaymentsList().stream()
                        .flatMap(p -> p.getFiscalReceiptsList().stream())
                        .map(fri -> new ReceiptItem(fri.getUrl(), fri.getType())).collect(toList()))
                .build();
    }

    private PaymentInfo createLegacyPaymentInfo(TOrderInfo protoOrder, OrderStatus status) {
        CurrencyUnit currency = ProtoUtils.fromTPrice(protoOrder.getPriceInfo().getPrice()).getCurrency();
        switch (status) {
            case RESERVED:
            case RESERVED_WITH_RESTRICTIONS:
                return PaymentInfo.builder()
                        .amountPaid(Money.zero(currency))
                        .mayBeStarted(true)
                        .mayBeCancelled(false)
                        .build();
            case AWAITS_PAYMENT:
                return PaymentInfo.builder()
                        .amountPaid(Money.zero(currency))
                        .current(CurrentPaymentInfo.builder()
                                .paymentUrl(protoOrder.getInvoiceList().stream()
                                        .filter(i -> i.getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT)
                                        .findAny()
                                        .get()
                                        .getPaymentUrl())
                                .paymentType(PaymentType.FULL)
                                .amount(ProtoUtils.fromTPrice(protoOrder.getPriceInfo().getPrice()))
                                .build())
                        .mayBeStarted(false)
                        .mayBeCancelled(true)
                        .build();
            case PAYMENT_FAILED:
                NextPaymentInfo next = null;
                if (protoOrder.getHotelOrderState() != EHotelOrderState.OS_CANCELLED) {
                    next = NextPaymentInfo.builder()
                            .amount(ProtoUtils.fromTPrice(protoOrder.getPriceInfo().getPrice()))
                            .paymentEndsAt(ProtoUtils.toInstant(protoOrder.getExpiresAt()))
                            .build();
                }

                return PaymentInfo.builder()
                        .amountPaid(Money.zero(currency))
                        .mayBeStarted(protoOrder.getHotelOrderState() != EHotelOrderState.OS_CANCELLED)
                        .mayBeCancelled(protoOrder.getHotelOrderState() != EHotelOrderState.OS_CANCELLED)
                        .next(next)
                        .error(PaymentErrorInfo.builder()
                                .code(protoOrder.getInvoiceList().stream()
                                        .map(i -> PaymentErrorCode.fromErrorCode(i.getAuthorizationErrorCode()))
                                        .reduce((first, second) -> second)  // taking last invoice's
                                        // error code
                                        .orElse(null))
                                .amount(ProtoUtils.fromTPrice(protoOrder.getPriceInfo().getPrice()))
                                .build())
                        .build();
            case CONFIRMED:
            case REFUNDED:
            case CANCELLED:
            case CANCELLED_WITH_REFUND:
                return PaymentInfo.builder()
                        .mayBeStarted(false)
                        .mayBeCancelled(false)
                        .amountPaid(ProtoUtils.fromTPrice(protoOrder.getPriceInfo().getPrice()))
                        .receipts(protoOrder.getInvoiceList().stream()
                                .filter(i -> i.getTrustInvoiceState() == ETrustInvoiceState.IS_HOLD ||
                                        i.getTrustInvoiceState() == ETrustInvoiceState.IS_CLEARED ||
                                        i.getTrustInvoiceState() == ETrustInvoiceState.IS_CANCELLED ||
                                        i.getTrustInvoiceState() == ETrustInvoiceState.IS_REFUNDED)
                                .flatMap(inv -> inv.getFiscalReceiptList().stream())
                                .map(fri -> new ReceiptItem(fri.getUrl(), fri.getType()))
                                .collect(toList()))
                        .build();
            default:
                return null;

        }
    }

    private TCreateOrderReq createOrderReq(HotelItinerary itinerary, BookingFlowContext context,
                                           THotelTestContext testContext, boolean useDeferred, boolean usePostPay) {
        OrderExperiments orderExperiments = experimentDataProvider.getInstance(OrderExperiments.class, context.getHeaders());
        TJson payload = ProtoUtils.toTJson(itinerary);
        TJson experiments = ProtoUtils.toTJson(orderExperiments);
        log.info("Creating order with payload:" + payload.getValue());
        UserCredentials userCredentials = context.getUserCredentials();

        TUserInfo userInfo = TUserInfo.newBuilder()
                .setLogin(Strings.nullToEmpty(userCredentials.getLogin()))
                .setPassportId(Strings.nullToEmpty(userCredentials.getPassportId()))
                .setYandexUid(Strings.nullToEmpty(userCredentials.getYandexUid()))
                .setIp(context.getUserIp())
                .setEmail(context.getOrderCreationData().getCustomerEmail())
                .setPhone(context.getOrderCreationData().getCustomerPhone())
                .setAllowsSubscription(context.getOrderCreationData().isAllowsSubscription())
                .build();

        var serviceReqBuilder = TCreateServiceReq.newBuilder()
                .setServiceType(context.getProvider().getServiceType())
                .setSourcePayload(payload);
        if (testContext != null) {
            serviceReqBuilder.setHotelTestContext(testContext);
        }

        TCreateOrderReq.Builder reqBuilder = TCreateOrderReq.newBuilder()
                .setOwner(userInfo)
                .setOrderType(context.getProvider().getOrderType())
                .addCreateServices(serviceReqBuilder.build())
                .setDeduplicationKey(context.getDeduplicationKey())
                .setCurrency(((ProtoCurrencyUnit) itinerary.getFiscalPrice().getCurrency()).getProtoCurrency())
                .setExperiments(experiments)
                .addAllKVExperiments(KVExperiments.toProto(context.getHeaders().getExperiments()));

        if (!Strings.isNullOrEmpty(context.getPaymentTestContextToken())) {
            reqBuilder.setPaymentTestContext(apiTokenEncrypter.fromPaymentTestContextToken(context.getPaymentTestContextToken()));
            reqBuilder.setMockPayment(true);
        }

        if (context.getOfferLabel() != null) {
            reqBuilder.setLabel(context.getOfferLabel());
        }

        if (context.getPromoCodes() != null) {
            context.getPromoCodes().forEach(reqBuilder::addPromoCodeStrings);
        }

        reqBuilder.setUseDeferredPayment(useDeferred);
        reqBuilder.setUsePostPay(usePostPay);

        return reqBuilder.build();
    }

    private CompletableFuture<HotelItinerary> createItinerary(
            Offer offer, BookingFlowContext context, boolean mapGuests, Integer yandexPlusBalance,
            boolean usePostPay) {
        return context.getPartnerFutures().createHotelItinerary()
                .thenCompose(itinerary -> context.getPartnerFutures().getRefundRules()
                        .thenApply(refundRules -> {
                            if (!offer.getMetaInfo().getCheckSum().equals(context.getOrderCreationData().getChecksum())) {
                                throw new RuntimeException("Checksum mismatch");
                            }
                            var request = context.getOrderCreationData();

                            itinerary.setCustomerEmail(request.getCustomerEmail());
                            itinerary.setCustomerPhone(request.getCustomerPhone());
                            itinerary.setCustomerIp(context.getUserIp());
                            itinerary.setCustomerUserAgent(context.getUserAgent());
                            itinerary.setAllowsSubscription(request.isAllowsSubscription());
                            itinerary.getPostPay().setEligible(offer.isAllowPostPay());
                            itinerary.getPostPay().setUsed(usePostPay);

                            if (mapGuests) {
                                itinerary.setGuests(getGuests(offer, request));
                            }

                            itinerary.setGeneratedAtInstant(context.getCreatedAt());
                            itinerary.setRefundRules(refundRules);
                            itinerary.setUiPayload(offer.toJsonNode());
                            itinerary.setUiPayloadType(offer.getClass().getName());
                            var detailsBuilder = OrderDetails.builder()
                                    .checkinDate(offer.getMetaInfo().getSearch().getCheckIn())
                                    .checkoutDate(offer.getMetaInfo().getSearch().getCheckOut())
                                    .checkinBegin(offer.getStayInfo().getCheckInStartTime())
                                    .checkinEnd(offer.getStayInfo().getCheckInEndTime())
                                    .checkoutBegin(offer.getStayInfo().getCheckOutStartTime())
                                    .checkoutEnd(offer.getStayInfo().getCheckOutEndTime())
                                    .hotelName(offer.getPartnerHotelInfo().getName())
                                    .roomName(offer.getRoomInfo().getName())
                                    .permalink(offer.getHotelInfo().getPermalink())
                                    .providerId(partnerIdToProviderIdMap.getOrDefault(offer.getMetaInfo().getSearch().getPartnerId(), "NOT_FOUND"))
                                    .originalId(offer.getMetaInfo().getSearch().getOriginalId())
                                    .hotelGeoRegions(getHotelGeoRegions(offer.getHotelInfo().getCoordinates()))
                                    .hotelTimeZoneId(getHotelTimeZoneId(offer.getHotelInfo().getCoordinates()))
                                    .addressByYandex(offer.getHotelInfo().getAddress())
                                    .addressByPartner(offer.getPartnerHotelInfo().getAddress())
                                    .hotelPhone(offer.getPartnerHotelInfo().getPhone())
                                    .ratePlanDetails(offer.getStayInfo().getRateDescriptionForSupport());
                            switch (offer.getHotelInfo().getLocationType()) {
                                case GLOBAL:
                                    detailsBuilder.locationType(LocationType.GLOBAL);
                                    break;
                                case RUSSIA:
                                    detailsBuilder.locationType(LocationType.RUSSIA);
                                    break;
                                case MOSCOW:
                                    detailsBuilder.locationType(LocationType.MOSCOW);
                                    break;
                            }
                            if (offer.getDiscountInfo() != null && offer.getDiscountInfo().isApplied()) {
                                itinerary.setDiscount(offer.getDiscountInfo().getDiscountAmount());
                            }
                            itinerary.setOrderDetails(detailsBuilder.build());

                            itinerary.setAppliedPromoCampaigns(
                                    AppliedPromoCampaignsDto.convertToModel(context.getAppliedPromoCampaigns()));
                            itinerary.setActivePromoCampaigns(offer.getPromoCampaignsInfo());

                            if (yandexPlusBalance != null) {
                                itinerary.setUserInfo(HotelUserInfo.builder()
                                        .plusBalance(yandexPlusBalance)
                                        .build()
                                );
                            }

                            applyTemporaryTestScenario(itinerary, request);
                            return itinerary;
                        }));
    }

    private List<Guest> getGuests(Offer offer, OrderCreationData request) {
        SearchInfo search = offer.getMetaInfo().getSearch();
        int numOfChildren = Optional.ofNullable(search.getChildren())
                .map(List::size)
                .orElse(0);
        int numGuests = search.getAdults() + numOfChildren;

        if (request.getGuests().size() != numGuests) {
            // if user has chosen not to fill info, we expect to receive empty fields, not less elements in the list.
            // Otherwise, we wouldn't be able to map children
            throw new InvalidInputException(Collections.singletonList(
                    new InputError("guests.size",
                            request.getGuests().size(),
                            InputError.ErrorType.INVALID_NUMBER_OF_GUESTS)));
        }
        // guests list can contain empty names for guests without filled names
        return IntStream.range(0, numGuests).mapToObj(i -> {
                    var reqGuest = request.getGuests().get(i);
                    var guest = new Guest();
                    if (!reqGuest.isEmpty()) {
                        guest.setFirstName(reqGuest.getFirstName());
                        guest.setLastName(reqGuest.getLastName());
                    }
                    if (i >= search.getAdults()) {
                        guest.setChild(true);
                        guest.setAge(search.getChildren().get(i - search.getAdults()));
                    }
                    return guest;
                })
                .collect(toList());
    }

    // todo(tlg-13): remove this temporary hack when the UI form lets choose the mode
    private void applyTemporaryTestScenario(HotelItinerary itinerary, OrderCreationData request) {
        AppliedPromoCampaigns campaigns = itinerary.getAppliedPromoCampaigns();
        if (campaigns == null || campaigns.getYandexPlus() == null) {
            return;
        }
        if (request.getCustomerEmail() == null || !request.getCustomerEmail().contains("+tmp_plus_withdraw@")) {
            return;
        }
        campaigns.getYandexPlus().setMode(YandexPlusApplication.Mode.WITHDRAW);
    }

    private List<GeoRegion> getHotelGeoRegions(Coordinates coordinates) {
        Integer geoId = GeoBaseHelpers.getRegionIdByLocationOrNull(
                geoBase, coordinates.getLatitude(), coordinates.getLongitude());
        if (geoId == null) {
            log.warn("Null geoID for coordinates: {}", coordinates);
            return Collections.emptyList();
        }
        try {
            return StreamSupport.stream(GeoBaseHelpers.getRegionChain(geoBase, geoId).spliterator(), false)
                    .map(region -> GeoRegion.builder()
                            .geoId(region.getGeoId())
                            .type(region.getType())
                            .name(region.getName())
                            .build())
                    .collect(Collectors.toUnmodifiableList());
        } catch (Exception e) {
            log.error("Cannot get region chain for geoid {}", geoId, e);
            return Collections.emptyList();
        }
    }


    private ZoneId getHotelTimeZoneId(Coordinates coordinates) {
        if (coordinates == null) {
            return null;
        }
        return timezoneDetector.getZoneOffset(coordinates.getLatitude(), coordinates.getLongitude(), null);
    }

    private String callIdForMethod(String methodName) {
        return AppCallIdGenerator.KEY.get().generate(methodName);
    }

    private Money rateInfoOrNull(RateInfo rateInfo) {
        if (rateInfo != null && rateInfo.getTotalRate() != null) {
            return rateInfo.getTotalRate().asMoney();
        }
        return null;
    }
}
