package ru.yandex.travel.api.services.hotels_booking_flow;

import java.util.concurrent.CompletableFuture;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.exceptions.InvalidTravelTokenException;
import ru.yandex.travel.api.services.orders.Meters;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.hotels.common.token.TokenCodec;
import ru.yandex.travel.hotels.common.token.TokenException;
import ru.yandex.travel.hotels.common.token.TravelToken;
import ru.yandex.travel.hotels.models.booking_flow.Offer;
import ru.yandex.travel.hotels.models.booking_flow.OfferState;
import ru.yandex.travel.hotels.proto.EPartnerId;

import static ru.yandex.travel.hotels.proto.EPartnerId.PI_EXPEDIA;
import static ru.yandex.travel.hotels.proto.EPartnerId.PI_UNUSED;

@Service
@RequiredArgsConstructor
@Slf4j
public class OfferService {
    private final PartnerDispatcher partnerDispatcher;
    private final TokenCodec tokenCodec;
    private final Meters meters;


    public CompletableFuture<Offer> getOffer(BookingFlowContext context, Integer yandexPlusBalance) {
        try {
            MDC.put(AsyncHttpClientWrapper.MDC_OFFER_TOKEN, context.getToken());
            var travelToken = decodeToken(context.getToken(), context.getStage());
            context.setDecodedToken(travelToken);
            final EPartnerId partnerId;
            if (travelToken.getPartnerId() != PI_UNUSED) {
                partnerId = travelToken.getPartnerId();
            } else {
                partnerId = PI_EXPEDIA; // fallback for very old tokens
            }
            PartnerBookingProvider bookingProvider = partnerDispatcher.get(partnerId);
            if (bookingProvider == null) {
                throw new RuntimeException("Unknown partner: " + travelToken.getPartnerId());
            }
            context.setProvider(bookingProvider);
            return bookingProvider.getOffer(context, yandexPlusBalance).whenComplete((offer, t) -> {
                Meters.OfferMeters offerMeters = meters.getOfferMeters(context.getStage(), partnerId);
                if (t != null) {
                    offerMeters.getFailedOffers().increment();
                } else {
                    if (offer.getRateInfo() != null) {
                        switch (offer.getRateInfo().getStatus()) {
                            case PRICE_MISMATCH_UP:
                                offerMeters.getPriceMismatch(Meters.PriceMismatchType.UP).increment();
                                break;
                            case PRICE_MISMATCH_DOWN:
                                offerMeters.getPriceMismatch(Meters.PriceMismatchType.DOWN).increment();
                                break;
                            case SOLD_OUT:
                                offerMeters.getSoldOut().increment();
                                break;
                        }
                    }
                    if (offer.checkOfferState() == OfferState.READY) {
                        offerMeters.getSuccessfulOffers().increment();
                    }
                }
            });
        } catch (TokenException tokenException) {
            return CompletableFuture.failedFuture(new InvalidTravelTokenException(tokenException.getMessage(),
                    tokenException));
        } catch (Exception ex) {
            return CompletableFuture.failedFuture(ex);
        } finally {
            MDC.remove(AsyncHttpClientWrapper.MDC_OFFER_TOKEN);
        }
    }

    private TravelToken decodeToken(String tokenString, BookingFlowContext.Stage stage) {
        try {
            return tokenCodec.decode(tokenString);
        } catch (Exception ex) {
            log.error("Unable to decode travel token", ex);
            meters.getOfferMeters(stage).getMalformedToken().increment();
            throw ex;
        }
    }
}
