package ru.yandex.travel.api.services.hotels_booking_flow;

import java.time.Duration;
import java.time.ZoneId;
import java.time.ZoneOffset;

import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.common.schedule.OfferPaymentSchedule;
import ru.yandex.travel.hotels.common.schedule.PaymentScheduleBuilder;
import ru.yandex.travel.hotels.common.schedule.PaymentScheduleHotelList;
import ru.yandex.travel.hotels.common.schedule.PaymentScheduleHotelListImpl;
import ru.yandex.travel.hotels.common.schedule.PaymentScheduleProperties;
import ru.yandex.travel.hotels.models.booking_flow.HotelInfo;
import ru.yandex.travel.hotels.proto.THotelTestContext;

@Service
@Slf4j
public class PaymentScheduleService {
    private final PaymentScheduleProperties properties;
    private final TimezoneDetector timezoneDetector;
    private final PaymentScheduleHotelList paymentScheduleHotelList;

    @Autowired
    public PaymentScheduleService(OfferServiceConfigurationProperties offerServiceConfigurationProperties,
                                  TimezoneDetector timezoneDetector) {
        this.properties = offerServiceConfigurationProperties.getDeferredSchedule();
        this.timezoneDetector = timezoneDetector;
        if (offerServiceConfigurationProperties.getDeferredSchedule().getHotelLists() != null &&
                offerServiceConfigurationProperties.getDeferredSchedule().getHotelLists().isEnabled()) {
            PaymentScheduleHotelListImpl paymentScheduleHotelListImpl =
                    new PaymentScheduleHotelListImpl(offerServiceConfigurationProperties.getDeferredSchedule().getHotelLists());
            paymentScheduleHotelListImpl.start();
            this.paymentScheduleHotelList = paymentScheduleHotelListImpl;

        } else {
            log.warn("PaymentScheduleHotelList is disabled. " +
                    "All enabled partners will return all their hotels as eligible for deferred payments");
            this.paymentScheduleHotelList = (a, b) -> true;
        }
    }

    public OfferPaymentSchedule getDeferredSchedule(Money totalAmount, Money initialAmount, RefundRules refundRules,
                                                    HotelInfo hotelInfo,
                                                    BookingFlowContext context, THotelTestContext testContext) {
        if (!properties.isEnabled()) {
            return null;
        }

        if (properties.checkEnabledForPartner(context.getDecodedToken().getPartnerId()) &&
                paymentScheduleHotelList.checkPaymentScheduleIsAllowedForHotel(context.getDecodedToken().getPartnerId(),
                        context.getDecodedToken().getOriginalId())) {
            if (hotelInfo == null || refundRules == null) {
                return null;
            }
            Duration paymentInterval = properties.getMinPaymentInterval();
            Duration refundSafetyWindow = properties.getRefundSafetyWindow();
            if (testContext != null) {
                if (testContext.getIgnorePaymentScheduleRestrictions()) {
                    paymentInterval = PaymentScheduleProperties.MIN_PAYMENT_INTERVAL;
                    refundSafetyWindow = PaymentScheduleProperties.MIN_SAFETY_WINDOW;
                }
            }
            Money maxFirstPayment = initialAmount.multiply(properties.getMaxFirstPaymentRate());
            Money minFirstPayment;
            if (properties.getMinFirstPayment() != null ){
                minFirstPayment = Money.of(properties.getMinFirstPayment(), ProtoCurrencyUnit.RUB);
            } else if (properties.getMinFirstPaymentRate() != null) {
                minFirstPayment = totalAmount.multiply(properties.getMinFirstPaymentRate());
            } else {
                throw new IllegalStateException("Incorrect configuration of minFirstPayment");
            }
            var hotelZoneId = timezoneDetector.getZoneOffset(hotelInfo.getCoordinates().getLatitude(),
                    hotelInfo.getCoordinates().getLongitude(), ZoneId.systemDefault());
            var offset =
                    hotelZoneId.getRules().getOffset(context.getDecodedToken().getCheckInDate().atStartOfDay().toInstant(ZoneOffset.UTC));
            try {
                return PaymentScheduleBuilder.twoPaymentSchedule(context.getCreatedAt(),
                        context.getDecodedToken().getCheckInDate().atStartOfDay().toInstant(offset), totalAmount,
                        minFirstPayment, maxFirstPayment, refundRules, paymentInterval, refundSafetyWindow);
            } catch (Exception ex) {
                return null;
            }
        } else {
            return null;
        }
    }
}
