package ru.yandex.travel.api.services.hotels_booking_flow;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.util.CollectionUtils;

import ru.yandex.travel.api.config.hotels.TravellineConfigurationProperties;
import ru.yandex.travel.api.services.hotels.legal_info.HotelLegalInfoDictionary;
import ru.yandex.travel.api.services.localization.LocalizationService;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.administrator.export.proto.HotelLegalInfo;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClient;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineRefundRulesBuilder;
import ru.yandex.travel.hotels.common.partners.travelline.exceptions.ReturnedErrorException;
import ru.yandex.travel.hotels.common.partners.travelline.model.Address;
import ru.yandex.travel.hotels.common.partners.travelline.model.Currency;
import ru.yandex.travel.hotels.common.partners.travelline.model.ErrorType;
import ru.yandex.travel.hotels.common.partners.travelline.model.PlacementRate;
import ru.yandex.travel.hotels.common.partners.travelline.model.RespHotelReservation;
import ru.yandex.travel.hotels.common.partners.travelline.model.ResponseRoomStay;
import ru.yandex.travel.hotels.common.partners.travelline.model.RoomType;
import ru.yandex.travel.hotels.common.partners.travelline.model.ServiceKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.StayUnitKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.VerifyReservationRequest;
import ru.yandex.travel.hotels.common.partners.travelline.model.VerifyReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.dto.OfferDto;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.models.booking_flow.Amenity;
import ru.yandex.travel.hotels.models.booking_flow.BedGroupInfo;
import ru.yandex.travel.hotels.models.booking_flow.BreakdownType;
import ru.yandex.travel.hotels.models.booking_flow.Coordinates;
import ru.yandex.travel.hotels.models.booking_flow.ExtraFee;
import ru.yandex.travel.hotels.models.booking_flow.ExtraFeeType;
import ru.yandex.travel.hotels.models.booking_flow.HotelInfo;
import ru.yandex.travel.hotels.models.booking_flow.Image;
import ru.yandex.travel.hotels.models.booking_flow.LegalInfo;
import ru.yandex.travel.hotels.models.booking_flow.LocalizedPansionInfo;
import ru.yandex.travel.hotels.models.booking_flow.PartnerHotelInfo;
import ru.yandex.travel.hotels.models.booking_flow.Rate;
import ru.yandex.travel.hotels.models.booking_flow.RateInfo;
import ru.yandex.travel.hotels.models.booking_flow.RateStatus;
import ru.yandex.travel.hotels.models.booking_flow.RoomInfo;
import ru.yandex.travel.hotels.models.booking_flow.StayInfo;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.THotelTestContext;
import ru.yandex.travel.hotels.proto.TOfferData;
import ru.yandex.travel.hotels.proto.TTravellineOffer;
import ru.yandex.travel.orders.commons.proto.EServiceType;

@PartnerBean(EPartnerId.PI_TRAVELLINE)
@EnableConfigurationProperties(TravellineConfigurationProperties.class)
@RequiredArgsConstructor
@Slf4j
public class TravellinePartnerBookingProvider extends AbstractPartnerBookingProvider<TTravellineOffer> {

    private final TimezoneDetector timezoneDetector;

    // https://a.yandex-team.ru/arc/trunk/arcadia/travel/hotels/tools/permaroom_builder/builder/resources/travelline_beds_mapping.csv список взят отсюда посредством обсуждения
    private final Set<String> bedKinds = new HashSet<>(List.of(
            "two_single_or_big_double_bed",
            "bunk_beds",
            "bunk_beds_3",
            "child_rollaway",
            "double_bed",
            "eight_single_beds",
            "five_single_beds",
            "four_beds",
            "full_bed",
            "king_bed",
            "queen_bed",
            "single_bed",
            "six_beds",
            "ten_single_beds",
            "three_double_beds",
            "three_full_beds",
            "three_single_beds",
            "two_double_beds",
            "two_full_beds",
            "two_king_size_beds",
            "two_queen_size_beds",
            "two_single_beds"));

    private final TravellineClient travellineClient;
    private final BedInflector bedInflector;
    private final LocalizationService localizationService;
    private final TravellineConfigurationProperties properties;
    private final HotelLegalInfoDictionary hotelLegalInfoDictionary;

    @Override
    PartnerFutures getPartnerFutures(BookingFlowContext context, CompletableFuture<TTravellineOffer> offerDataFuture) {
        var verifyOrErrorFuture =
                offerDataFuture.thenCompose(offerData -> context.getTestContextFuture().thenCompose(testContext ->
                {
                    OfferDto travellineOffer = ProtoUtils.fromTJson(offerData.getTravellineOfferDTO(), OfferDto.class);
                    VerifyReservationRequest requestBody = VerifyReservationRequest.create(travellineOffer, 0);
                    return travellineClient
                            .withCallContext(getCallContext(context, offerData, testContext))
                            .verifyReservation(requestBody);
                }));

        var verifyReservationFuture = FutureUtils.handleExceptionOfType(verifyOrErrorFuture,
                ReturnedErrorException.class, e -> {
                    var soldOutError = e.getErrorOfType(ErrorType.SOLD_OUT, ErrorType.ARRIVAL_DATE_IS_NOT_AVAILABLE);
                    if (soldOutError != null) {
                        VerifyReservationResponse resp = new VerifyReservationResponse();
                        resp.setErrors(e.getErrors());
                        return resp;
                    }
                    throw e;
                });

        return new PartnerFutures() {
            @Override
            public CompletableFuture<PartnerHotelInfo> getPartnerHotelInfo() {
                return offerDataFuture.thenApply(offerData -> mapPartnerHotelInfo(offerData));
            }

            @Override
            public CompletableFuture<RateInfo> getRateInfo() {
                return offerDataFuture.thenCombine(verifyReservationFuture,
                        TravellinePartnerBookingProvider.this::mapRateInfo);
            }

            @Override
            public CompletableFuture<RoomInfo> getRoomInfo() {
                return offerDataFuture.thenCombine(verifyReservationFuture,
                        TravellinePartnerBookingProvider.this::mapRoomInfo);
            }

            @Override
            public CompletableFuture<StayInfo> getStayInfo() {
                return offerDataFuture.thenApply(TravellinePartnerBookingProvider.this::mapStayInfo);
            }

            @Override
            public CompletableFuture<RefundRules> getRefundRules() {
                return offerDataFuture.thenCombine(verifyReservationFuture,
                        TravellinePartnerBookingProvider.this::mapRefundRules);
            }

            @Override
            public CompletableFuture<LegalInfo.LegalInfoItem> getPartnerLegalInfoItem() {
                return CompletableFuture.completedFuture(null);
            }

            @Override
            public CompletableFuture<LegalInfo.LegalInfoItem> getHotelLegalInfoItem() {
                HotelLegalInfo hotelLegalInfo = hotelLegalInfoDictionary.getHotelLegalInfo(
                        context.getDecodedToken().getPartnerId(),
                        context.getDecodedToken().getOriginalId());
                if (hotelLegalInfo == null) {
                    meters.incrementHotelLegalDataMissing(getPartnerId());
                    return CompletableFuture.failedFuture(new RuntimeException(
                            String.format("HotelLegalInfo is not found for Partner: %s, HotelCode: %s", getPartnerId(),
                                    context.getDecodedToken().getOriginalId())));
                }
                return CompletableFuture.completedFuture(LegalInfo.LegalInfoItem.builder()
                        .name(hotelLegalInfo.getLegalName())
                        .workingHours(hotelLegalInfo.getWorkingHours())
                        .legalAddress(hotelLegalInfo.getLegalAddress())
                        .ogrn(hotelLegalInfo.getOgrn())
                        .build());
            }

            @Override
            public CompletableFuture<HotelItinerary> createHotelItinerary() {
                return CompletableFuture.allOf(offerDataFuture, verifyReservationFuture).thenApply(ignored ->
                        mapHotelItinerary(offerDataFuture.join(), verifyReservationFuture.join()));
            }
        };
    }

    @Override
    public boolean isPostPayAllowed(TOfferData offerData, BookingFlowContext context,
                                    HotelInfo hotelInfo) {
        var testContext = FutureUtils.joinCompleted(context.getTestContextFuture());
        if (testContext != null && testContext.getIsPostPay()) {
            return true;
        }
        ZoneId hotelZoneId = getHotelTimeZoneId(hotelInfo.getCoordinates());
        try {
            return isPostPayAllowed(context.getPartnerFutures().getRefundRules().get(),
                    context.getDecodedToken().getCheckInDate(), hotelZoneId);
        } catch (ExecutionException | InterruptedException ignored) {
            return false;
        }
    }

    /**
     * @return true if full refund is possible at the check-in day
      */
    private boolean isPostPayAllowed(RefundRules refundRules, LocalDate checkInDate, ZoneId hotelZoneId) {
        if (hotelZoneId == null) {
            hotelZoneId = ZoneId.systemDefault();
        }
        Instant checkInInstant = checkInDate.atStartOfDay(hotelZoneId).toInstant();
        return refundRules != null && refundRules.isFullyRefundableAt(checkInInstant);
    }

    private ZoneId getHotelTimeZoneId(Coordinates coordinates) {
        if (coordinates == null) {
            return null;
        }
        return timezoneDetector.getZoneOffset(coordinates.getLatitude(), coordinates.getLongitude(), null);
    }

    private TravellineHotelItinerary mapHotelItinerary(TTravellineOffer offerData,
                                                       VerifyReservationResponse verifyReservationResponse) {
        var hotelReservation = verifyReservationResponse.getHotelReservations().get(0);
        Preconditions.checkNotNull(hotelReservation);
        Preconditions.checkState(hotelReservation.getRoomStays().size() > 0, "No room stays in verify response");
        OfferDto travellineOffer = ProtoUtils.fromTJson(offerData.getTravellineOfferDTO(),
                OfferDto.class);
        travellineOffer.toBuilder().stayDates(hotelReservation.getRoomStays().get(0).getStayDates());
        TravellineHotelItinerary itinerary = new TravellineHotelItinerary();
        itinerary.setOffer(travellineOffer);
        itinerary.setSelectedPlacement(travellineOffer.getBestPlacementIndex()); // TODO: select proper index based
        // on context
        itinerary.setYandexNumberPrefix(properties.getYandexNumberPrefix());

        BigDecimal amount = BigDecimal.valueOf(hotelReservation.getTotal().getPriceBeforeTax()).setScale(0,
                RoundingMode.HALF_UP);
        itinerary.setFiscalPrice(Money.of(amount, hotelReservation.getTotal().getCurrency().getValue()));
        itinerary.setCommission(Money.of(BigDecimal.ZERO, hotelReservation.getTotal().getCurrency().getValue()));
        return itinerary;
    }

    String formatRoomDescription(String description) {
        StringBuilder builder = new StringBuilder();
        for (String paragraph : description.split("\\n\\n")) {
            builder.append("<p>");
            builder.append(paragraph
                    .replaceAll("\\n", "<br/>")
                    .replaceAll("\\t", ""));
            builder.append("</p>");
        }
        return builder.toString();
    }

    private RefundRules mapRefundRules(TTravellineOffer offerData,
                                       VerifyReservationResponse verifyReservationResponse) {
        if (verifyReservationResponse.getErrors() != null && verifyReservationResponse.getErrors().size() > 0) {
            return null;
        }
        RespHotelReservation hotelReservation = verifyReservationResponse.getHotelReservations().get(0);
        ResponseRoomStay roomStay = hotelReservation.getRoomStays().get(0);
        OfferDto travellineOffer = ProtoUtils.fromTJson(offerData.getTravellineOfferDTO(),
                OfferDto.class);

        return TravellineRefundRulesBuilder.build(
                roomStay.getCancelPenaltyGroup(),
                travellineOffer.getPossiblePlacements().get(travellineOffer.getBestPlacementIndex()).getPlacements(),
                roomStay.getRatePlans().get(0).getCode(),
                roomStay.getStayDates(),
                ZoneOffset.of(travellineOffer.getHotel().getTimezone().getOffset()),
                (int) Math.round(travellineOffer.getBestPrice())
        ).actualize(); //remove penalties in the past
    }

    private CallContext getCallContext(BookingFlowContext flowContext, TTravellineOffer travellineOffer,
                                       THotelTestContext testContext) {
        return new CallContext(flowContext.getStage() == BookingFlowContext.Stage.GET_OFFER ?
                CallContext.CallPhase.OFFER_VALIDATION : CallContext.CallPhase.ORDER_CREATION,
                testContext, travellineOffer, flowContext.getDecodedToken(), null, null);
    }

    private StayInfo mapStayInfo(TTravellineOffer offerData) {
        OfferDto travellineOffer = ProtoUtils.fromTJson(offerData.getTravellineOfferDTO(),
                OfferDto.class);
        var includedNonMealServices = travellineOffer.getServices().values().stream()
                .filter(serviceDto -> serviceDto.getInfo().getKind() != ServiceKind.MEAL && serviceDto.getConditions().isInclusive())
                .collect(Collectors.toList());
        String stayInstructions = null;
        if (!includedNonMealServices.isEmpty()) {
            StringBuilder builder = new StringBuilder();
            builder.append("<p>В стоимость включены следующие услуги:")
                    .append("<ul>");
            includedNonMealServices.forEach(s -> {
                builder.append("<li><p>");
                builder.append("<b>").append(s.getInfo().getName()).append("</b><br/>");
                builder.append(s.getInfo().getDescription().replaceAll("\\n", "<br/>"));
                builder.append("</p></li>");
            });
            builder.append("</ul></p>");
            stayInstructions = builder.toString();
        }

        return StayInfo.builder()
                .checkInStartTime(travellineOffer.getStayDates().getStartDate().toLocalTime().toString())
                .checkOutEndTime(travellineOffer.getStayDates().getEndDate().toLocalTime().toString())
                .stayInstruction(stayInstructions)
                .rateDescriptionForSupport(travellineOffer.getRatePlan().getDescription())
                .build();
    }

    private RoomInfo mapRoomInfo(TTravellineOffer offerData, VerifyReservationResponse verifyReservationResponse) {
        OfferDto travellineOffer = ProtoUtils.fromTJson(offerData.getTravellineOfferDTO(),
                OfferDto.class);

        LocalizedPansionInfo pansionInfo = new LocalizedPansionInfo(travellineOffer.getPansionType(),
                localizationService.localizePansion(travellineOffer.getPansionType(), "ru"));

        RoomType roomType = travellineOffer.getRoomType();
        var roomTypeAmenities = roomType.getAmenities();
        List<Amenity> amenities = IntStream.range(0, roomTypeAmenities.size())
                .mapToObj(index -> new Amenity(String.valueOf(index), roomTypeAmenities.get(index).getName()))
                .collect(Collectors.toList());

        var bedGroupAmenities = roomTypeAmenities
                .stream()
                .filter(amenity -> bedKinds.contains(amenity.getKind()))
                .collect(Collectors.toList());
        List<BedGroupInfo> bedGroupsInfo = IntStream
                .range(0, bedGroupAmenities.size())
                .mapToObj(index -> new BedGroupInfo(index, bedGroupAmenities.get(index).getName()))
                .collect(Collectors.toList());
        String roomName = roomType.getName();
        if (properties.isDebugShowRateInfo() && !"0".equals(travellineOffer.getRatePlan().getCode())) {
            // "0" is id of test-context ratePlan, this is used in FE tests
            roomName += " [" + travellineOffer.getRatePlan().getCode() + "] '" + travellineOffer.getRatePlan().getName() + "'";
        }
        return RoomInfo.builder()
                .name(roomName)
                .description(formatRoomDescription(roomType.getDescription()))
                .images(roomType
                        .getImages()
                        .stream()
                        .map(image -> new Image(image.getUrl(), image.getUrl(), image.getUrl()))
                        .collect(Collectors.toList()))
                .pansionInfo(pansionInfo)
                .roomAmenities(amenities)
                .bedGroups(List.of(createOneBedGroupInfo(bedGroupsInfo)))
                .build();
    }

    private RateInfo mapRateInfo(TTravellineOffer offerData, VerifyReservationResponse verifyReservationResponse) {
        OfferDto travellineOffer = ProtoUtils.fromTJson(offerData.getTravellineOfferDTO(),
                OfferDto.class);
        BigDecimal initialPrice = new BigDecimal(travellineOffer.getBestPrice()).setScale(0, RoundingMode.HALF_UP);
        RateStatus rateStatus;
        Rate rate;
        List<ExtraFee> extraFees;
        if (!CollectionUtils.isEmpty(verifyReservationResponse.getErrors()) && (
                verifyReservationResponse.getErrors().get(0).getErrorCode() == ErrorType.SOLD_OUT ||
                        verifyReservationResponse.getErrors().get(0).getErrorCode() == ErrorType.ARRIVAL_DATE_IS_NOT_AVAILABLE)) {
            rateStatus = RateStatus.SOLD_OUT;
            rate = new Rate(initialPrice.toString(), Currency.RUB.getValue());
            extraFees = new ArrayList<>();
        } else {
            RespHotelReservation hotelReservation = verifyReservationResponse.getHotelReservations().get(0);
            BigDecimal verifiedPrice = new BigDecimal(hotelReservation.getTotal().getPriceBeforeTax())
                    .setScale(0, RoundingMode.HALF_UP);
            rateStatus = RateStatus.fromComparison(initialPrice, verifiedPrice);
            rate = new Rate(
                    verifiedPrice.toString(),
                    hotelReservation.getTotal().getCurrency().toString());
            extraFees = verifyReservationResponse.getHotelReservations().get(0).getTotal().getTaxes().stream()
                    .filter(tax -> tax.getAmount() != null)
                    .map(tax -> new ExtraFee(
                            BigDecimal.valueOf(tax.getAmount()).setScale(2, RoundingMode.HALF_UP).toString(),
                            verifyReservationResponse.getHotelReservations().get(0).getTotal().getCurrency().getValue(),
                            ExtraFeeType.OTHER_FEE))
                    .collect(Collectors.toList());
        }

        List<List<Double>> ratePlacementList =
                travellineOffer.getPossiblePlacements().get(travellineOffer.getBestPlacementIndex()).getPlacements().stream()
                        .map(placement -> placement.getRates().stream().map(PlacementRate.Rate::getPriceBeforeTax).collect(Collectors.toList()))
                        .collect(Collectors.toList());
        var rateList = IntStream.range(0, ratePlacementList.get(0).size())
                .mapToObj(i -> {
                    double day = ratePlacementList.stream()
                            .mapToDouble(l -> l.get(i))
                            .sum();
                    return new Rate(BigDecimal.valueOf(day).setScale(2, RoundingMode.HALF_UP).toString(),
                            travellineOffer.getPossiblePlacements().get(0).getPlacements().get(0).getCurrency().getValue());
                })
                .collect(Collectors.toList());
        BreakdownType bt = BreakdownType.NIGHT;
        if (travellineOffer.getHotel().getStayUnitKind() == StayUnitKind.DAY) {
            bt = BreakdownType.DAY;
        }

        return RateInfo.builder()
                .baseRate(rate)
                .baseRateBreakdown(rateList)
                .breakdownType(bt)
                .taxesAndFees(null)
                .totalRate(rate)
                .extraFees(extraFees)
                .status(rateStatus)
                .build();
    }

    private PartnerHotelInfo mapPartnerHotelInfo(TTravellineOffer offerData) {
        OfferDto travellineOffer = ProtoUtils.fromTJson(offerData.getTravellineOfferDTO(),
                OfferDto.class);
        Address address = travellineOffer.getHotel().getContactInfo().getAddresses().get(0); //TODO which address to
        // take?
        String defaultPhone = null;
        if (travellineOffer.getHotel().getContactInfo().getPhones() != null &&
                travellineOffer.getHotel().getContactInfo().getPhones().size() > 0) {
            defaultPhone = travellineOffer.getHotel().getContactInfo().getPhones().get(0).getPhoneNumber();
        }

        return PartnerHotelInfo.builder()
                .name(travellineOffer.getHotel().getName())
                .address(getAddressString(address))
                .phone(defaultPhone)
                .coordinates(new Coordinates(address.getLongitude(), address.getLatitude()))
                .stars(travellineOffer.getHotel().getStars())
                .amenities(List.of())  //TODO clarify
                .description(new PartnerHotelInfo.TextBlock("Транспортная доступность", address.getRemark())) //TODO
                // where is hotel description?
                .images(travellineOffer.getHotel().getImages().stream()
                        .skip(1)
                        .map(image -> new Image(image.getUrl(), image.getUrl(), image.getUrl()))
                        .collect(Collectors.toList()))
                .build();
    }

    private String getAddressString(Address address) {
        StringBuilder addressStr = new StringBuilder();
        addressStr.append(address.getPostalCode());
        addressStr.append(", ").append(address.getRegion());
        if (!address.getRegion().equals(address.getCityName())) {
            addressStr.append(", ").append(address.getCityName());
        }
        address.getAddressLine().forEach(line -> addressStr.append(", ").append(line));
        return addressStr.toString();
    }


    @Override
    public EServiceType getServiceType() {
        return EServiceType.PT_TRAVELLINE_HOTEL;
    }

    @Override
    TTravellineOffer getPartnerOffer(TOfferData offerData) {
        Preconditions.checkArgument(offerData.hasTravellineOffer());
        return offerData.getTravellineOffer();
    }

    @Override
    public EPartnerId getPartnerId() {
        return EPartnerId.PI_TRAVELLINE;
    }
}
