package ru.yandex.travel.api.services.hotels_booking_flow.promo;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Clock;
import java.time.Instant;
import java.time.LocalDate;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.services.hotels_booking_flow.CheckParamsProvider;
import ru.yandex.travel.api.services.hotels_booking_flow.promo.HotelPromoCampaignsServiceProperties.Taxi2020PromoCampaignProperties;
import ru.yandex.travel.api.services.promo.YandexPlusService;
import ru.yandex.travel.clients.promo_service_booking_flow.PromoServiceBookingFlowUtils;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.lang.ComparatorUtils;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.models.booking_flow.promo.Mir2020PromoCampaign;
import ru.yandex.travel.hotels.models.booking_flow.promo.PromoCampaignsInfo;
import ru.yandex.travel.hotels.models.booking_flow.promo.Taxi2020PromoCampaign;
import ru.yandex.travel.hotels.models.booking_flow.promo.WhiteLabelPromoCampaign;
import ru.yandex.travel.hotels.models.booking_flow.promo.YandexEdaPromoCampaign;
import ru.yandex.travel.hotels.models.booking_flow.promo.YandexPlusPromoCampaign;
import ru.yandex.travel.hotels.proto.EMirEligibility;
import ru.yandex.travel.hotels.proto.EWhiteLabelEligibility;
import ru.yandex.travel.hotels.proto.EYandexEdaEligibility;
import ru.yandex.travel.hotels.proto.EYandexPlusEligibility;
import ru.yandex.travel.hotels.proto.TDeterminePromosForOfferRsp;
import ru.yandex.travel.hotels.proto.TExperimentInfo;
import ru.yandex.travel.hotels.proto.TMirPromoStatus;
import ru.yandex.travel.hotels.proto.TOfferInfo;
import ru.yandex.travel.hotels.proto.TUserInfo;
import ru.yandex.travel.hotels.proto.TWhiteLabelInfo;
import ru.yandex.travel.hotels.proto.TWhiteLabelStatus;
import ru.yandex.travel.hotels.proto.TYandexEda2022Status;
import ru.yandex.travel.hotels.proto.TYandexEdaPromoInfo;
import ru.yandex.travel.hotels.proto.TYandexPlusStatus;
import ru.yandex.travel.hotels.services.promoservice.PromoServiceClient;
import ru.yandex.travel.order_type.proto.EOrderType;

@Service
@RequiredArgsConstructor
@EnableConfigurationProperties(HotelPromoCampaignsServiceProperties.class)
@Slf4j
public class HotelPromoCampaignsService {
    private final HotelPromoCampaignsServiceProperties properties;
    private final Clock clock;
    private final PromoServiceClient promoServiceClient;
    private final YandexPlusService yandexPlusService;
    private final ExperimentDataProvider experimentDataProvider;

    private final Counter hotelPromoErrorsCounter =
            Counter.builder("hotels.promo.error").register(Metrics.globalRegistry);

    public PromoCampaignsInfo calculatePromoCampaignsInfo(TDeterminePromosForOfferRsp commonCampaigns,
                                                          CheckParamsProvider checkParams,
                                                          YandexPlusPromoCampaign yandexPlusPromoCampaign,
                                                          Money priceAfterPlusWithdraw) {
        try {
            // most new promo campaigns should come from the promo service
            return PromoCampaignsInfo.builder()
                    .taxi2020(getTaxi2020Campaign(checkParams))
                    .mir2020(getMir2020Campaign(commonCampaigns, checkParams, priceAfterPlusWithdraw))
                    .yandexPlus(yandexPlusPromoCampaign)
                    .yandexEda(getYandexEdaCampaign(commonCampaigns))
                    .whiteLabel(getWhiteLabelCampaign(commonCampaigns))
                    .build();
        } catch (Throwable e) {
            log.error("Failed to build promo campaigns info: {}", e.getMessage());
            hotelPromoErrorsCounter.increment();
            // No any fallback here as each promo campaign should decide how to handle its own exceptions;
            // We've decided that plus promo is an important part of the offer so we can't proceed in case of
            // any uncertain errors and the whole booking flow should stop (except for Trust unavailability)
            throw e instanceof RuntimeException ? (RuntimeException) e : new RuntimeException(e);
        }
    }

    @VisibleForTesting
    Mir2020PromoCampaign getMir2020Campaign(TDeterminePromosForOfferRsp commonCampaigns,
                                            CheckParamsProvider checkParams,
                                            Money priceAfterPlusWithdraw
    ) {
        if (checkParams.getPriceAfterPromoCodes() == null) {
            return null;
        }
        TMirPromoStatus status = commonCampaigns.getMir();
        var builder = Mir2020PromoCampaign.builder()
                .eligibility(status.getEligibility());

        if (status.getEligibility() == EMirEligibility.ME_ELIGIBLE) {
            boolean displayAmount = !properties.getMir2020().isAlwaysShowRate() &&
                    checkParams.getPriceAfterPromoCodes()
                            .getNumberStripped()
                            .compareTo(properties.getMir2020().getPriceLimitForAmount()) >= 0;
            // TODO: TRAVELITM-298
            var cashbackRate = BigDecimal.valueOf(status.getCashbackRate().getValue());
            int cashbackAmount = BigDecimal.valueOf(status.getMaxCashbackAmount().getValue())
                    .min(priceAfterPlusWithdraw.getNumberStripped().multiply(cashbackRate))
                    .setScale(0, RoundingMode.DOWN).intValue();
            builder.mirId(status.getMirId())
                    .cashbackAmount(cashbackAmount)
                    .cashbackRate(cashbackRate)
                    .displayAmount(displayAmount)
                    .expiresAt(ProtoUtils.toInstant(status.getExpiresAt()));
        }
        return builder.build();
    }

    @VisibleForTesting
    Taxi2020PromoCampaign getTaxi2020Campaign(CheckParamsProvider checkParams) {
        if (checkParams.getPriceBeforePromoCodes() == null) {
            return new Taxi2020PromoCampaign(false);
        }

        Taxi2020PromoCampaignProperties properties = this.properties.getTaxi2020();
        BigDecimal price = checkParams.getPriceBeforePromoCodes().getNumberStripped();
        String currency = checkParams.getPriceBeforePromoCodes().getCurrency().getCurrencyCode();
        Instant now = Instant.now(clock);
        LocalDate checkIn = checkParams.getCheckIn();
        boolean matchesTerms =
                properties.getStartsAt().isBefore(now) &&
                        properties.getEndsAt().isAfter(now) &&
                        properties.getMinPriceCurrency().equals(currency) &&
                        properties.getMinPriceAmount().compareTo(price) <= 0 &&
                        !properties.getMaxCheckInDate().isBefore(checkIn);
        return new Taxi2020PromoCampaign(matchesTerms);
    }

    public YandexEdaPromoCampaign getYandexEdaCampaign(TDeterminePromosForOfferRsp commonCampaigns) {
        TYandexEda2022Status yandexEda2022Status = commonCampaigns.getYandexEda2022Status();

        var builder = YandexEdaPromoCampaign.builder()
                .eligible(yandexEda2022Status.getEligibility());
        if (yandexEda2022Status.getEligibility() == EYandexEdaEligibility.YEE_ELIGIBLE) {
            TYandexEdaPromoInfo yandexEdaPromoInfo = yandexEda2022Status.getPromoInfo();
            builder.data(YandexEdaPromoCampaign.YandexEdaPromocodePayload.builder()
                    .numberOfPromocodes(yandexEdaPromoInfo.getPromoCodeCount())
                    .promocodeCost(Money.of(yandexEdaPromoInfo.getPromoCodeNominal(), "RUB"))
                    .firstSendDate(yandexEdaPromoInfo.getFirstDate())
                    .lastSendDate(yandexEdaPromoInfo.getLastDate())
                    .build());
        }
        return builder.build();
    }

    public WhiteLabelPromoCampaign getWhiteLabelCampaign(TDeterminePromosForOfferRsp commonCampaigns) {
        TWhiteLabelStatus whiteLabelStatus = commonCampaigns.getWhiteLabelStatus();
        var builder = WhiteLabelPromoCampaign.builder()
                .eligible(whiteLabelStatus.getEligibility());
        if (whiteLabelStatus.getEligibility() == EWhiteLabelEligibility.WLE_ELIGIBLE)
            builder.partnerId(whiteLabelStatus.getPartnerId())
                    .points(WhiteLabelPromoCampaign.WhiteLabelPoints.builder()
                            .amount(whiteLabelStatus.getPoints().getAmount())
                            .pointsType(whiteLabelStatus.getPoints().getPointsType())
                            .build())
                    .pointsLinguistics(WhiteLabelPromoCampaign.WhiteLabelPointsLinguistics.builder()
                            .nameForNumeralNominative(
                                    whiteLabelStatus.getPointsLinguistics().getNameForNumeralNominative())
                            .build());

        return builder.build();
    }

    public CompletableFuture<TDeterminePromosForOfferRsp> getCommonPromoCampaignsInfo(
            CheckParamsProvider params,
            List<EOrderType> userExistingOrderTypes,
            CommonHttpHeaders headers
    ) {
        try {
            TOfferInfo offerInfo = PromoServiceBookingFlowUtils.buildOfferInfo(
                    params.getPartnerId(),
                    params.getHotelId(),
                    params.getCheckIn().toString(),
                    params.getCheckOut().toString(),
                    params.getPriceFromPartnerOffer(),
                    params.getPriceBeforePromoCodes(),
                    params.getPriceAfterPromoCodes()
            );
            TUserInfo userInfo = PromoServiceBookingFlowUtils.buildUserInfo(
                    params.getPassportId(),
                    params.getIsPlusUser(),
                    userExistingOrderTypes,
                    true
            );
            TExperimentInfo experimentInfo = PromoServiceBookingFlowUtils.buildExperimentInfo(
                    experimentDataProvider, headers);
            TWhiteLabelInfo whiteLabelInfo = PromoServiceBookingFlowUtils.buildWhiteLabelInfo(headers);

            return promoServiceClient.determinePromosForOffer(offerInfo, userInfo, experimentInfo, whiteLabelInfo);

        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public YandexPlusPromoCampaign getYandexPlusCampaign(
            TDeterminePromosForOfferRsp commonCampaigns,
            CheckParamsProvider checkParams,
            Integer yandexPlusBalance
    ) {
        if (properties.getYandexPlusEnabled() != Boolean.TRUE) {
            return null;
        }
        Money totalPrice = checkParams.getPriceAfterPromoCodes();
        Preconditions.checkArgument(yandexPlusService.isCurrencySupported(totalPrice.getCurrency()),
                "Unsupported offer currency %s", totalPrice.getCurrency());
        TYandexPlusStatus plusInfo = commonCampaigns.getPlus();
        boolean eligible = plusInfo.getEligibility() == EYandexPlusEligibility.YPE_ELIGIBLE;
        Integer withdrawPoints = getPointsToWithdraw(eligible, yandexPlusBalance, totalPrice);

        return YandexPlusPromoCampaign.builder()
                .eligible(eligible)
                .points(plusInfo.getPoints().getValue())
                .withdrawPoints(withdrawPoints)
                .build();
    }

    private Integer getPointsToWithdraw(boolean eligible, Integer balance, Money totalPrice) {
        if (!eligible || properties.getYandexPlusWithdrawEnabled() != Boolean.TRUE) {
            // not available
            return null;
        }
        if (balance == null || balance <= 0) {
            // no plus points to spend
            return null;
        }
        if (totalPrice.getNumber().intValue() < 2) {
            // test offers? checking just in case as we need to pay at least 1 'rouble' (whole currency unit)
            return null;
        }
        // balance >= 1 && totalPrice >= 2 => pointsToWithdraw >= 1
        return ComparatorUtils.min(balance, totalPrice.getNumberStripped().intValue() - 1);
    }
}
