package ru.yandex.travel.api.services.idm

import org.springframework.beans.factory.InitializingBean
import org.springframework.context.ApplicationContext
import org.springframework.context.ApplicationContextAware
import org.springframework.stereotype.Service
import ru.yandex.travel.api.services.idm.model.CombinedRolesInfo
import ru.yandex.travel.api.services.idm.model.CombinedUserRole
import ru.yandex.travel.api.services.idm.model.IdmResult
import ru.yandex.travel.api.services.idm.model.LocalizedName
import ru.yandex.travel.api.services.idm.model.errors.MissingSlugException
import ru.yandex.travel.api.services.idm.model.errors.UnknownServiceException
import java.util.concurrent.CompletableFuture

private const val serviceSlug = "service"
private const val roleSlug = "role"
private val serviceName = LocalizedName("Сервис", "Service")
private val roleName = LocalizedName("Роль", "Role")

@Service
class IdmServiceGateway : ApplicationContextAware, InitializingBean {
    lateinit var context: ApplicationContext
    private lateinit var serviceBindings: Map<String, IdmServiceBinding>


    override fun setApplicationContext(applicationContext: ApplicationContext) {
        context = applicationContext
    }

    override fun afterPropertiesSet() {
        serviceBindings = context.getBeansOfType(IdmServiceBinding::class.java).values
            .associateBy { it.serviceKey }
    }


    fun roles(): CompletableFuture<CombinedRolesInfo> {
        val futures = serviceBindings.values.map { binding ->
            binding.rolesInfo().thenApply {
                Pair(binding, it)
            }
        }
        return CompletableFuture.allOf(*futures.toTypedArray()).thenApply {
            CombinedRolesInfo(
                slug = serviceSlug,
                name = serviceName,
                values = futures.map { it.join() }.associateBy({ it.first.serviceKey }) {
                    CombinedRolesInfo.ServiceRolesInfo(
                        name = it.first.serviceName,
                        roles = CombinedRolesInfo.RoleInfo(
                            slug = roleSlug,
                            name = roleName,
                            values = it.second,
                        )
                    )
                }
            )
        }
    }

    fun addRole(
        login: String,
        slugs: Map<String, String>,
        passportUid: String?,
        fields: Map<String, Any>?,
    ): CompletableFuture<IdmResult> {

        val serviceName = slugs[serviceSlug] ?: throw MissingSlugException(serviceSlug)
        if (!serviceBindings.containsKey(serviceName)) {
            throw UnknownServiceException(serviceName)
        }
        val roleName = slugs[roleSlug] ?: throw MissingSlugException(roleSlug)
        return serviceBindings.getValue(serviceName).addRole(login, roleName, passportUid, fields)
    }

    fun removeRole(
        login: String,
        slugs: Map<String, String>
    ): CompletableFuture<IdmResult> {

        val serviceName = slugs[serviceSlug] ?: throw MissingSlugException(serviceSlug)
        if (!serviceBindings.containsKey(serviceName)) {
            throw UnknownServiceException(serviceName)
        }
        val roleName = slugs[roleSlug] ?: throw MissingSlugException(roleSlug)

        return serviceBindings.getValue(serviceName).removeRole(login, roleName)
    }

    fun getAllRoles(): CompletableFuture<List<CombinedUserRole>> {
        val futures = serviceBindings.values.map { binding ->
            binding.getAllRoles().thenApply { roles ->
                roles.map {
                    object {
                        var login: String = it.login
                        var slugs: Map<String, String> = binding.buildSlugs(it.role)
                        var fields: Map<String, Any> = it.fields ?: emptyMap()
                    }
                }
            }
        }
        return CompletableFuture.allOf(*futures.toTypedArray()).thenApply {
            futures.flatMap { it.join() }.groupBy { it.login }.entries.map {
                CombinedUserRole(
                    login = it.key,
                    roles = it.value.map { r ->
                        CombinedUserRole.SluggedUserRole(
                            slugs = r.slugs,
                            fields = r.fields,
                        )
                    }
                )
            }
        }
    }

    private fun IdmServiceBinding.buildSlugs(role: String): Map<String, String> {
        return mapOf(serviceSlug to this.serviceKey, roleSlug to role)
    }
}
