package ru.yandex.travel.api.services.idm.bindings

import com.google.common.base.Strings
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.stereotype.Component
import ru.yandex.travel.api.services.extranet.ExtranetClientFactory
import ru.yandex.travel.api.services.idm.IdmServiceBinding
import ru.yandex.travel.api.services.idm.model.IdmField
import ru.yandex.travel.api.services.idm.model.IdmInfoRole
import ru.yandex.travel.api.services.idm.model.IdmResult
import ru.yandex.travel.api.services.idm.model.LocalizedName
import ru.yandex.travel.api.services.idm.model.ServiceUserRole
import ru.yandex.travel.api.services.idm.model.errors.InvalidRoleException
import ru.yandex.travel.api.services.idm.model.errors.MissingDataException
import ru.yandex.travel.commons.concurrent.toCompletable
import ru.yandex.travel.hotels.extranet.AddRoleRequest
import ru.yandex.travel.hotels.extranet.GetAllRolesRequest
import ru.yandex.travel.hotels.extranet.IDMRoleType
import ru.yandex.travel.hotels.extranet.IdmResponse
import ru.yandex.travel.hotels.extranet.RemoveRoleRequest
import ru.yandex.travel.hotels.extranet.RolesInfoRequest
import java.util.concurrent.CompletableFuture

private val rolesToStringMap = mapOf(
    IDMRoleType.IDM_ROLE_TYPE_DEVELOPER to "developer",
    IDMRoleType.IDM_ROLE_TYPE_ACCOUNT_MANAGER to "account_manager",
)
private val stringToRolesMap = rolesToStringMap.entries.associateBy({ it.value }) { it.key }

@Component
@ConditionalOnProperty("idm.extranet.enabled")
class ExtranetIdmBinding(private val clientFactory: ExtranetClientFactory) : IdmServiceBinding {
    override val serviceKey = "hotels_extranet"
    override val serviceName = LocalizedName("Отельный Экстранет", "Hotel Extranet")

    override fun rolesInfo(): CompletableFuture<Map<String, IdmInfoRole>> {
        return clientFactory.createIdmServiceFutureStub().rolesInfo(RolesInfoRequest.newBuilder().build())
            .toCompletable().thenApply { response ->
                response.rolesList
                    .filter { it.type in rolesToStringMap }
                    .associateBy({ rolesToStringMap.getValue(it.type) }) {
                        IdmInfoRole(
                            name = LocalizedName(ru = it.name.nameRu, en = it.name.nameEn),
                            fields = listOf(IdmField.passport())
                        )
                    }
            }
    }

    override fun addRole(
        login: String,
        role: String,
        passportUid: String?,
        fields: Map<String, Any>?
    ): CompletableFuture<IdmResult> {
        val roleVal = stringToRolesMap[role] ?: throw InvalidRoleException(role)
        val passportLogin =
            fields?.get(IdmField.passport().slug) as? String ?: throw MissingDataException(IdmField.passport().slug)

        return clientFactory.createIdmServiceFutureStub()
            .addRole(
                AddRoleRequest.newBuilder()
                    .setRole(roleVal)
                    .setLogin(login)
                    .setPassportLogin(passportLogin)
                    .build()
            ).toCompletable().thenApply { mapIdmResult(it.details) }
    }

    private fun mapIdmResult(details: IdmResponse): IdmResult {
        return IdmResult(
            code = details.code,
            warning = Strings.emptyToNull(details.warning),
            error = Strings.emptyToNull(details.error),
            fatal = Strings.emptyToNull(details.fatal)
        )
    }

    override fun removeRole(login: String, role: String): CompletableFuture<IdmResult> {
        val roleVal = stringToRolesMap[role] ?: throw InvalidRoleException(role)
        return clientFactory.createIdmServiceFutureStub()
            .removeRole(
                RemoveRoleRequest.newBuilder()
                    .setRole(roleVal)
                    .setLogin(login)
                    .build()
            ).toCompletable().thenApply  { mapIdmResult(it.details) }
    }

    override fun getAllRoles(): CompletableFuture<List<ServiceUserRole>> {
        return clientFactory.createIdmServiceFutureStub().getAllRoles(GetAllRolesRequest.getDefaultInstance())
            .toCompletable()
            .thenApply { response ->
                response.usersList
                    .filter { it.type in rolesToStringMap }
                    .map {
                        ServiceUserRole(
                            it.login,
                            role = rolesToStringMap.getValue(it.type),
                            fields = mapOf(IdmField.passport().slug to it.passportLogin)
                        )
                    }
            }
    }
}
