package ru.yandex.travel.api.services.idm.bindings

import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.stereotype.Component
import ru.yandex.travel.api.config.hotels.administrator.AdministratorAdminInterfaceProvider
import ru.yandex.travel.api.services.idm.IdmServiceBinding
import ru.yandex.travel.api.services.idm.model.IdmInfoRole
import ru.yandex.travel.api.services.idm.model.IdmResult
import ru.yandex.travel.api.services.idm.model.LocalizedName
import ru.yandex.travel.api.services.idm.model.ServiceUserRole
import ru.yandex.travel.api.services.idm.model.errors.InvalidRoleException
import ru.yandex.travel.api.services.idm.model.errors.MissingPassportUidException
import ru.yandex.travel.commons.concurrent.toCompletable
import ru.yandex.travel.hotels.administrator.grpc.proto.*
import java.util.concurrent.CompletableFuture


private val rolesToStringMap = mapOf(
    AdminRole.AR_DEVELOPER to "developer",
    AdminRole.AR_BIZDEV to "bizdev",
)
private val stringToRolesMap = rolesToStringMap.entries.associateBy({ it.value }) { it.key }


@Component
@ConditionalOnProperty("idm.hotel-admin.enabled")
class HotelAdminIdmBinding(private val grpcProvider: AdministratorAdminInterfaceProvider) : IdmServiceBinding {
    override val serviceKey = "hotel_admin"
    override val serviceName = LocalizedName("Отельный Администратор", "Hotel Administrator")

    override fun rolesInfo(): CompletableFuture<Map<String, IdmInfoRole>> {
        return grpcProvider.idmInterface().rolesInfo(RolesInfoRequest.newBuilder().build())
            .toCompletable().thenApply { response ->
                response.rolesList
                    .filter { it.type in rolesToStringMap }
                    .associateBy({ rolesToStringMap.getValue(it.type)}) {
                        IdmInfoRole(
                            name = LocalizedName(ru = it.names.nameRu, en = it.names.nameEn),
                        )
                    }
            }
    }

    override fun addRole(
        login: String,
        role: String,
        passportUid: String?,
        fields: Map<String, Any>?
    ): CompletableFuture<IdmResult> {
        passportUid ?: throw MissingPassportUidException()
        val roleVal = stringToRolesMap[role] ?: throw InvalidRoleException(role)
        return grpcProvider.idmInterface()
            .addRole(AddRoleRequest.newBuilder()
                .setRole(roleVal)
                .setLogin(login)
                .setPassportUid(passportUid).build()
            ).toCompletable().thenApply {
                if (!it.added) {
                    IdmResult(code = 1, warning = "role already exists")
                } else {
                    IdmResult()
                }
            }
    }

    override fun removeRole(login: String, role: String): CompletableFuture<IdmResult> {
        val roleVal = stringToRolesMap[role] ?: throw InvalidRoleException(role)
        return grpcProvider.idmInterface()
            .removeRole(RemoveRoleRequest.newBuilder()
                .setRole(roleVal)
                .setLogin(login).build()
            ).toCompletable().thenApply {
                if (!it.removed) {
                    IdmResult(warning = "User does not belong to role")
                } else {
                    IdmResult()
                }
            }
    }

    override fun getAllRoles(): CompletableFuture<List<ServiceUserRole>> {
        return grpcProvider.idmInterface().getAllRoles(GetAllRolesRequest.getDefaultInstance())
            .toCompletable()
            .thenApply { response ->
                response.usersList
                    .filter { it.type in rolesToStringMap }
                    .map {
                        ServiceUserRole(
                            it.login,
                            role = rolesToStringMap.getValue(it.type),
                        )
                    }
            }
    }
}
