package ru.yandex.travel.api.services.idm.bindings

import com.google.common.base.Strings
import io.grpc.Status
import io.grpc.StatusRuntimeException
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.stereotype.Component
import ru.yandex.misc.ExceptionUtils
import ru.yandex.travel.api.services.idm.IdmServiceBinding
import ru.yandex.travel.api.services.idm.model.IdmInfoRole
import ru.yandex.travel.api.services.idm.model.IdmResult
import ru.yandex.travel.api.services.idm.model.LocalizedName
import ru.yandex.travel.api.services.idm.model.ServiceUserRole
import ru.yandex.travel.api.services.idm.model.errors.InvalidRoleException
import ru.yandex.travel.api.services.orders.OrchestratorIdmClientFactory
import ru.yandex.travel.commons.concurrent.toCompletable
import ru.yandex.travel.orders.commons.proto.EAdminRole
import ru.yandex.travel.orders.idm.proto.*
import java.util.concurrent.CompletableFuture

private val protoToString = mapOf(
    EAdminRole.AR_DEVELOPER to "developer",
    EAdminRole.AR_OPERATOR to "operator",
    EAdminRole.AR_ADVANCED_OPERATOR to "advancedOperator",
    EAdminRole.AR_ADV_BUS_OPERATOR to "advancedBusOperator",
    EAdminRole.AR_ADV_TRAINS_OPERATOR to "advancedTrainsOperator",
    EAdminRole.AR_ADV_AVIA_OPERATOR to "advancedAviaOperator",
    EAdminRole.AR_ADV_HOTEL_OPERATOR to "advancedHotelsOperator",
    EAdminRole.AR_ADV_SUBURBAN_OPERATOR to "advancedSuburbanOperator",
    EAdminRole.AR_BIZDEV to "bizdev"
)

private val stringToProto = protoToString.entries.associateBy({ it.value }) { it.key }

@Component
@ConditionalOnProperty("idm.orders-admin.enabled")
class OrdersAdminIdmBinding(private val idmClientFactory: OrchestratorIdmClientFactory) : IdmServiceBinding {
    override val serviceKey = "orders_admin"
    override val serviceName = LocalizedName(
        "Админка заказов Путешествий",
        "Travel Order Admin"
    )


    override fun rolesInfo(): CompletableFuture<Map<String, IdmInfoRole>> {
        return idmClientFactory.createIdmFutureStub().rolesInfo(TIdmRolesInfoReq.getDefaultInstance()).toCompletable()
            .thenApply { resp ->
                resp.getRoleGroups(0).roleList.filter { it.value in protoToString }
                    .associateBy({ protoToString.getValue(it.value) })
                    {
                        IdmInfoRole(
                            name = LocalizedName(ru = it.name.nameRu, en = it.name.nameEn),
                        )
                    }
            }
    }

    override fun addRole(
        login: String,
        role: String,
        passportUid: String?,
        fields: Map<String, Any>?
    ): CompletableFuture<IdmResult> {
        val roleVal = stringToProto[role] ?: throw InvalidRoleException(role)

        return idmClientFactory.createIdmFutureStub().addRole(
            TIdmAddRoleReq.newBuilder()
                .setLogin(login)
                .setRole(TIdmRole.newBuilder().setValue(roleVal).build())
                .build()
        ).toCompletable().handle { r, t ->
            if (r != null) {
                IdmResult(code = if (Strings.isNullOrEmpty(r.warning)) 0 else 1, warning = r.warning)
            } else {
                if (t is StatusRuntimeException && t.status.code == Status.Code.ALREADY_EXISTS) {
                    IdmResult(code = 1, warning = t.message)
                } else {
                    throw ExceptionUtils.throwException(t)
                }
            }
        }
    }

    override fun removeRole(login: String, role: String): CompletableFuture<IdmResult> {
        val roleVal = stringToProto[role] ?: throw InvalidRoleException(role)
        return idmClientFactory.createIdmFutureStub().removeRole(
            TIdmRemoveRoleReq.newBuilder()
                .setLogin(login)
                .setRole(TIdmRole.newBuilder().setValue(roleVal).build())
                .build()
        ).toCompletable().handle { r, t ->
            if (r != null) {
                IdmResult(code = 0, warning = r.warning)
            } else {
                if (t is StatusRuntimeException && t.status.code == Status.Code.NOT_FOUND) {
                    IdmResult(code = 0, warning = t.message)
                } else {
                    throw ExceptionUtils.throwException(t)
                }
            }
        }
    }

    override fun getAllRoles(): CompletableFuture<List<ServiceUserRole>> {
        return idmClientFactory.createIdmFutureStub().getAllRoles(TIdmGetAllRolesReq.getDefaultInstance())
            .toCompletable()
            .thenApply { response ->
                response.usersList.flatMap { user ->
                    user.roleGroup.roleList
                        .filter { it.value in protoToString }
                        .map {
                            ServiceUserRole(
                                login = user.login,
                                role = protoToString.getValue(it.value)
                            )
                        }
                }
            }
    }
}
