package ru.yandex.travel.api.services.localization;

import java.io.IOException;
import java.net.URL;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.services.localization.model.LocalizationData;
import ru.yandex.travel.hotels.proto.EPansionType;

@Component
@Slf4j
public class LocalizationService {
    private final Inflector inflector;

    private LocalizationData localization;

    public LocalizationService(Inflector inflector) {
        ObjectMapper mapper = new ObjectMapper(new YAMLFactory());
        this.inflector = inflector;
        URL resource = getClass().getClassLoader().getResource("translations.yaml");
        try {
            localization = mapper.readValue(resource, LocalizationData.class);
        } catch (IOException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    public String getWhom(String word) {
        try {
            return inflector.inflect(word, "gen");
        } catch (InflectorException e) {
            log.warn("Unable to get a genitive form of '{}': {}", word, e);
            return word;
        }
    }

    private String getGenLocalization(int number, String singleKey, String pluralKey) {
        Preconditions.checkArgument(number > 0 && number <= 10);
        String numberStr = getLocalizedValue(String.valueOf(number) + "Collective", "ru");
        String word;
        if (number == 1) {
            word = getLocalizedValue(singleKey, "ru");
        } else {
            word = getLocalizedValue(pluralKey, "ru");
        }
        String sentence = String.format("%s %s", numberStr, word);
        return getWhom(sentence);
    }

    public String localizePansion(EPansionType pansionType, String locale) {
        return getLocalizedValue("PANSION_" + pansionType, locale);
    }

    public String getLocalizedValue(String key, String locale) {
        String result = getLocalizedValueWithoutDefault(key, locale);
        return result == null ? key : result;
    }

    public String getLocalizedValueWithoutDefault(String key, String locale) {
        var locales = localization.getKeys().get(key);
        if (locales == null) {
            return null;
        }
        return locales.get(locale);
    }
}
