package ru.yandex.travel.api.services.orders;

import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.service.dto.OrderDTO;
import ru.yandex.travel.api.config.avia.AviaBookingConfiguration;
import ru.yandex.travel.api.endpoints.booking_flow.model.DisplayableOrderStatus;
import ru.yandex.travel.api.models.travel_orders.AviaOrderListItem;
import ru.yandex.travel.api.services.avia.orders.AviaOrderService;

@Service
@ConditionalOnBean(AviaBookingConfiguration.class)
@RequiredArgsConstructor
public class AviaOrderListService implements TravelOrderSubtypeListFetcher {

    private final AviaOrderService orderService;

    @Override
    public CompletableFuture<TravelOrderSubtypeList> listOrders(int offset, int limit,
                                                                DisplayableOrderStatus displayableOrderStatus,
                                                                String searchTerm) {
        return orderService.listOrders(offset, limit,
                displayableOrderStatus != null ? displayableOrderStatus.getProtoStates() : null)
                .thenApply(list -> {
                    TravelOrderSubtypeList res = new TravelOrderSubtypeList();
                    res.setOffset(list.getOffset());
                    res.setLimit(list.getLimit());
                    res.setHasMoreOrders(list.isHasMoreOrders());
                    res.setOrders(list.getOrders().stream()
                            .map(this::convertOrderListItem)
                            .collect(Collectors.toList()));
                    return res;
                });
    }

    private AviaOrderListItem convertOrderListItem(OrderDTO order) {
        AviaOrderListItem item = new AviaOrderListItem();
        item.setId(order.getId());
        item.setYandexOrderId(order.getPrettyId());
        item.setOrderType(OrderType.AVIA);
        item.setStatus(DisplayableOrderStatus.fromProto(order.getEDisplayOrderState()));
        item.setAirReservation(order.getAirReservation());
        item.setTravellers(order.getTravellers());
        item.setPrice(order.getPrice());
        item.setState(order.getState());
        item.setTimeLimitAt(order.getTimeLimitAt());
        item.setErrorCode(order.getErrorCode());
        item.setReference(order.getReference());
        item.setServicedAt(order.getServicedAt());
        return item;
    }

}
