package ru.yandex.travel.api.services.orders;

import java.util.Map;
import java.util.NoSuchElementException;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.infrastucture.ApiTokenEncrypter;
import ru.yandex.travel.api.services.dictionaries.country.CountryDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.settlement.TrainSettlementDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.station.TrainStationDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.time_zone.TrainTimeZoneDataProvider;
import ru.yandex.travel.api.services.geo.PointProviderHelper;
import ru.yandex.travel.api.spec.Coordinates;
import ru.yandex.travel.api.spec.DocumentType;
import ru.yandex.travel.api.spec.Gender;
import ru.yandex.travel.api.spec.LegalEntity;
import ru.yandex.travel.api.spec.RegisterType;
import ru.yandex.travel.api.spec.buses.BenefitType;
import ru.yandex.travel.api.spec.buses.Fare;
import ru.yandex.travel.api.spec.buses.Passenger;
import ru.yandex.travel.api.spec.buses.Point;
import ru.yandex.travel.api.spec.buses.PointSource;
import ru.yandex.travel.api.spec.buses.PointType;
import ru.yandex.travel.api.spec.buses.Ride;
import ru.yandex.travel.api.spec.buses.ServiceInfo;
import ru.yandex.travel.api.spec.buses.Ticket;
import ru.yandex.travel.api.spec.buses.TicketTypeType;
import ru.yandex.travel.bus.model.BusBenefitType;
import ru.yandex.travel.bus.model.BusDocumentType;
import ru.yandex.travel.bus.model.BusGenderType;
import ru.yandex.travel.bus.model.BusLegalEntity;
import ru.yandex.travel.bus.model.BusPointInfo;
import ru.yandex.travel.bus.model.BusPointInfoSource;
import ru.yandex.travel.bus.model.BusRegisterType;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.model.BusRide;
import ru.yandex.travel.bus.model.BusTicketType;
import ru.yandex.travel.bus.model.BusesPassenger;
import ru.yandex.travel.bus.model.BusesTicket;
import ru.yandex.travel.buses.backend.proto.EPointKeyType;
import ru.yandex.travel.buses.backend.proto.TPointKey;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.dicts.rasp.proto.TSettlement;
import ru.yandex.travel.dicts.rasp.proto.TStation;
import ru.yandex.travel.orders.proto.TDownloadBlankToken;
import ru.yandex.travel.orders.proto.TOrderServiceInfo;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;

import static ru.yandex.travel.api.services.common.ApiSpecProtoUtils.consumeNonEmpty;
import static ru.yandex.travel.api.services.common.ApiSpecProtoUtils.consumeNotNull;
import static ru.yandex.travel.api.services.common.ApiSpecProtoUtils.instantToProtoWithoutTimezone;
import static ru.yandex.travel.api.services.common.ApiSpecProtoUtils.localDateToProto;
import static ru.yandex.travel.api.services.common.ApiSpecProtoUtils.moneyToProto;

@Service
@Slf4j
@RequiredArgsConstructor
public class BusModelMapService {
    private final ApiTokenEncrypter apiTokenEncrypter;
    private final CountryDataProvider countryDataProvider;

    // TODO move and rename providers
    private final TrainStationDataProvider stationDataProvider;
    private final TrainSettlementDataProvider settlementDataProvider;
    private final TrainTimeZoneDataProvider timeZoneDataProvider;

    private static final Map<BusGenderType, Gender.Enum> GENDER_MAP = Map.of(
            BusGenderType.MALE, Gender.Enum.male,
            BusGenderType.FEMALE, Gender.Enum.female
    );
    private static final Map<BusRegisterType, RegisterType.Enum> REGISTER_TYPE_MAP = Map.of(
            BusRegisterType.COMPANY, RegisterType.Enum.COMPANY,
            BusRegisterType.ENTREPRENEUR, RegisterType.Enum.ENTREPRENEUR
    );
    private static final Map<BusDocumentType, DocumentType.Enum> DOCUMENT_TYPE_MAP = Map.of(
            BusDocumentType.RU_PASSPORT, DocumentType.Enum.ru_national_passport,
            BusDocumentType.RU_BIRTH_CERTIFICATE, DocumentType.Enum.ru_birth_certificate,
            BusDocumentType.RU_INTERNATIONAL_PASSPORT, DocumentType.Enum.ru_foreign_passport,
            BusDocumentType.RU_FOREIGN_PASSPORT, DocumentType.Enum.other
    );
    private static final Map<BusBenefitType, BenefitType.Enum> BENEFIT_TYPE_MAP = Map.ofEntries(
            Map.entry(BusBenefitType.COFFEE, BenefitType.Enum.COFFEE),
            Map.entry(BusBenefitType.CHARGER, BenefitType.Enum.CHARGER),
            Map.entry(BusBenefitType.PRESS, BenefitType.Enum.PRESS),
            Map.entry(BusBenefitType.TV, BenefitType.Enum.TV),
            Map.entry(BusBenefitType.WI_FI, BenefitType.Enum.WI_FI),
            Map.entry(BusBenefitType.NO_TICKET_REQUIRED, BenefitType.Enum.NO_TICKET_REQUIRED),
            Map.entry(BusBenefitType.WC, BenefitType.Enum.WC),
            Map.entry(BusBenefitType.CONDITIONER, BenefitType.Enum.CONDITIONER),
            Map.entry(BusBenefitType.COMMON_AUDIO, BenefitType.Enum.COMMON_AUDIO)
    );
    private static final Map<BusTicketType, TicketTypeType.Enum> BUS_TICKET_TYPE_MAP = Map.of(
            BusTicketType.FULL, TicketTypeType.Enum.FULL,
            BusTicketType.CHILD, TicketTypeType.Enum.CHILD,
            BusTicketType.BAGGAGE, TicketTypeType.Enum.BAGGAGE
    );
    private static final Map<EPointKeyType, PointType.Enum> POINT_KEY_TYPE_MAP = Map.of(
            EPointKeyType.POINT_KEY_TYPE_SETTLEMENT, PointType.Enum.SETTLEMENT,
            EPointKeyType.POINT_KEY_TYPE_STATION, PointType.Enum.STATION
    );
    private static final Map<BusPointInfoSource, PointSource.Enum> POINT_SOURCE_MAP = Map.of(
            BusPointInfoSource.MATCHING, PointSource.Enum.MATCHING,
            BusPointInfoSource.MATCHING_PARENT, PointSource.Enum.MATCHING_PARENT,
            BusPointInfoSource.QUERY, PointSource.Enum.QUERY,
            BusPointInfoSource.QUERY_PARENT, PointSource.Enum.QUERY_PARENT
    );

    private static final Map<Gender.Enum, BusGenderType> GENDER_MAP_BACK = GENDER_MAP
            .entrySet().stream()
            .collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));
    private static final Map<DocumentType.Enum, BusDocumentType> DOCUMENT_TYPE_MAP_BACK = DOCUMENT_TYPE_MAP
            .entrySet().stream()
            .collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));
    private static final Map<TicketTypeType.Enum, BusTicketType> BUS_TICKET_TYPE_MAP_BACK = BUS_TICKET_TYPE_MAP
            .entrySet().stream()
            .collect(Collectors.toMap(Map.Entry::getValue, Map.Entry::getKey));

    public static BusGenderType genderFromSpec(Gender.Enum g) {
        return GENDER_MAP_BACK.get(g);
    }

    public static Gender.Enum genderToSpec(BusGenderType g) {
        return GENDER_MAP.get(g);
    }

    public static BusDocumentType documentTypeFromSpec(DocumentType.Enum dt) {
        return DOCUMENT_TYPE_MAP_BACK.get(dt);
    }

    public static DocumentType.Enum documentTypeToSpec(BusDocumentType dt) {
        return DOCUMENT_TYPE_MAP.get(dt);
    }

    public static BusTicketType ticketTypeFromSpec(TicketTypeType.Enum tt) {
        return BUS_TICKET_TYPE_MAP_BACK.get(tt);
    }

    public static TicketTypeType.Enum ticketTypeToSpec(BusTicketType tt) {
        return BUS_TICKET_TYPE_MAP.getOrDefault(tt, TicketTypeType.Enum.FULL);
    }

    public ServiceInfo buildBusServiceInfo(TOrderServiceInfo source, String documentUrl, String orderId) {
        BusReservation payload = ProtoUtils.fromTJson(source.getServiceInfo().getPayload(), BusReservation.class);
        var result = ServiceInfo.newBuilder()
                .setRide(convertRide(payload.getRide()));
        if (payload.getOrder() != null) {
            result.setPartnerOrderId(payload.getOrder().getId());
            result.addAllTickets(payload.getOrder().getTickets().stream()
                    .map(this::convertTicket).collect(Collectors.toList()));
        }
        if ((source.getServiceInfo().getGenericOrderItemState() == EOrderItemState.IS_CONFIRMED ||
                source.getServiceInfo().getGenericOrderItemState() == EOrderItemState.IS_REFUNDING ||
                source.getServiceInfo().getGenericOrderItemState() == EOrderItemState.IS_REFUNDED) &&
                !Strings.isNullOrEmpty(documentUrl)) {
            TDownloadBlankToken token = TDownloadBlankToken.newBuilder().setOrderId(orderId).build();
            result.setDownloadBlankToken(apiTokenEncrypter.toDownloadBlankToken(token));
        }
        return result.build();
    }

    private Ride convertRide(BusRide ride) {
        var builder = Ride.newBuilder();

        builder.setId(ride.getRideId());

        builder.setDeparture(instantToProtoWithoutTimezone(ride.getDepartureTime()));
        consumeNonEmpty(builder::setArrival, instantToProtoWithoutTimezone(ride.getArrivalTime()));
        builder.setDuration(ride.getDuration());

        consumeNotNull(builder::setPointFrom, convertPoint(ride.getPointFrom()));
        consumeNotNull(builder::setPointTo, convertPoint(ride.getPointTo()));
        consumeNotNull(builder::setTitlePointFrom, convertPoint(ride.getTitlePointFrom()));
        consumeNotNull(builder::setTitlePointTo, convertPoint(ride.getTitlePointTo()));

        consumeNonEmpty(builder::setBusDescription, ride.getBus());
        consumeNonEmpty(builder::setRouteNumber, ride.getRouteNumber());
        consumeNonEmpty(builder::setRouteName, ride.getRouteName());
        builder.addAllBenefits(ride.getBenefits().stream().map(this::convertBenefit)::iterator);
        consumeNotNull(builder::setCarrier, convertLegalEntity(ride.getCarrier()));
        consumeNotNull(builder::setSupplier, convertLegalEntity(ride.getSupplier()));

        var freeSeats = ride.getFreeSeats();
        if (freeSeats != null && freeSeats >= 0) {
            builder.setFreeSeats(freeSeats);
        }
        var ticketLimit = ride.getTicketLimit();
        if (ticketLimit != null && ticketLimit != 0) {
            builder.setTicketLimit(ticketLimit);
        }
        consumeNonEmpty(builder::setRefundConditions, ride.getRefundConditions());
        return builder.build();
    }

    private LegalEntity convertLegalEntity(BusLegalEntity legalEntity) {
        if (legalEntity == null) {
            return null;
        }
        var result = LegalEntity.newBuilder()
                .setRegisterNumber(legalEntity.getRegisterNumber());
        if (legalEntity.getRegisterType() != null) {
            result.setRegisterType(REGISTER_TYPE_MAP.getOrDefault(legalEntity.getRegisterType(),
                    RegisterType.Enum.INVALID));
        }
        consumeNonEmpty(result::setName, legalEntity.getName());
        consumeNonEmpty(result::setLegalName, legalEntity.getLegalName());
        consumeNonEmpty(result::setLegalAddress, legalEntity.getLegalAddress());
        consumeNonEmpty(result::setTaxationNumber, legalEntity.getTaxationNumber());
        consumeNonEmpty(result::setActualAddress, legalEntity.getActualAddress());
        consumeNonEmpty(result::setTimetable, legalEntity.getTimetable());
        return result.build();
    }

    private BenefitType.Enum convertBenefit(BusBenefitType benefitType) {
        return BENEFIT_TYPE_MAP.getOrDefault(benefitType, BenefitType.Enum.INVALID);
    }

    public PointType.Enum convertPointType(EPointKeyType source) {
        return POINT_KEY_TYPE_MAP.getOrDefault(source, PointType.Enum.INVALID);
    }

    private Ticket convertTicket(BusesTicket source) {
        var result = Ticket.newBuilder();
        result.setId(source.getId());
        var fareBuilder = Fare.newBuilder()
                .setTicket(moneyToProto(source.getTicketPrice()))
                .setSupplierFee(moneyToProto(source.getPartnerFee()))
                .setTotal(moneyToProto(source.getTotalPrice()));
        if (source.getYandexFee() != null) {
            fareBuilder.setYandexFee(moneyToProto(source.getYandexFee()));
        }
        result.setPrice(fareBuilder.build());
        if (source.getPassenger().getSeatPartnerId() != null) {
            result.setSeat(source.getPassenger().getSeatPartnerId());
        }
        result.setTicketType(ticketTypeToSpec(source.getPassenger().getTicketType()));
        result.setPassenger(convertPassenger(source.getPassenger()));
        return result.build();
    }

    private int getGeoIdByCode(String code) {
        try {
            return countryDataProvider.getByCodeId(code).getGeoId();
        } catch (NoSuchElementException ignored) {
            return countryDataProvider.getByCodeId(code.toUpperCase()).getGeoId();
        }
    }

    private Passenger convertPassenger(BusesPassenger passenger) {
        var result = Passenger.newBuilder();
        result.setBirthDate(localDateToProto(passenger.getBirthday()));
        result.setDocumentNumber(passenger.getDocumentNumber());
        result.setDocumentType(documentTypeToSpec(passenger.getDocumentType()));
        result.setFirstName(passenger.getFirstName());
        result.setPatronymic(Strings.nullToEmpty(passenger.getMiddleName()));
        result.setLastName(passenger.getLastName());
        if (passenger.getGender() != null) { // TODO BUSES-1522 report when this happens
            result.setSex(genderToSpec(passenger.getGender()));
        }
        if (passenger.getCitizenship() != null) { // TODO BUSES-1522 report when this happens
            String citizenshipCode = passenger.getCitizenship();
            int citizenshipGeoId = getGeoIdByCode(citizenshipCode);
            result.setCitizenship(citizenshipGeoId);
        }
        return result.build();
    }

    public Point convertPoint(BusPointInfo source) {
        if (source == null) {
            return null;
        }
        var result = Point.newBuilder();
        consumeNotNull(t -> result.setType(convertPointType(t)), source.getType());
        result.setSupplierDescription(source.getSupplierDescription());
        consumeNonEmpty(result::setTimezone, source.getTimezone());
        consumeNonEmpty(result::setTitle, source.getTitle());
        consumeNonEmpty(result::setAddress, source.getAddress());
        if (source.getLatitude() != null && source.getLongitude() != null) {
            result.setCoordinates(Coordinates.newBuilder()
                    .setLatitude(source.getLatitude())
                    .setLongitude(source.getLongitude())
                    .build());
        }
        if (source.getSource() != null) {
            result.setSource(POINT_SOURCE_MAP.getOrDefault(source.getSource(), PointSource.Enum.INVALID));
        }
        return result.build();
    }

    // TODO move to bus-parsers

    @NotNull
    public BusPointInfo makePointInfo(@NotNull String supplierDescription, @NotNull TPointKey pointKey) {
        var builder = BusPointInfo.builder()
                .supplierDescription(supplierDescription);
        try {
            if (ProtoUtils.hasFields(pointKey)) {
                switch (pointKey.getType()) {
                    case POINT_KEY_TYPE_STATION:
                        return buildPointInfo(builder, stationDataProvider.getById(pointKey.getId()));
                    case POINT_KEY_TYPE_SETTLEMENT:
                        return buildPointInfo(builder, settlementDataProvider.getById(pointKey.getId()));
                    default:
                        throw new RuntimeException(String.format("Unexpected point type %s", pointKey.getType()));
                }
            }
        } catch (NoSuchElementException ex) {
            log.warn("Unknown bus point", ex);
        }
        return builder.build();
    }

    @NotNull
    private BusPointInfo buildPointInfo(
            @NotNull BusPointInfo.BusPointInfoBuilder builder, @NotNull TStation station) {
        return builder
                .type(EPointKeyType.POINT_KEY_TYPE_STATION)
                .id(station.getId())
                .pointKey(PointProviderHelper.buildStationPointKey(station.getId()))
                .title(station.getTitleDefault())
                .timezone(getTimeZoneCode(station.getTimeZoneId()))
                .address(station.getLocalAddress())
                .longitude(station.getLongitude())
                .latitude(station.getLatitude())
                .build();
    }

    @NotNull
    private BusPointInfo buildPointInfo(
            @NotNull BusPointInfo.BusPointInfoBuilder builder, @NotNull TSettlement settlement) {
        return builder
                .type(EPointKeyType.POINT_KEY_TYPE_SETTLEMENT)
                .id(settlement.getId())
                .pointKey(PointProviderHelper.buildSettlementPointKey(settlement.getId()))
                .title(settlement.getTitleDefault())
                .timezone(getTimeZoneCode(settlement.getTimeZoneId()))
                .longitude(settlement.getLongitude())
                .latitude(settlement.getLatitude())
                .build();
    }

    @Data
    private static class StationWithSettlement {
        private final TStation station;
        private final TSettlement settlement;

        private boolean hasParent() {
            return getStation() != null && getSettlement() != null;
        }
    }

    @NotNull
    private StationWithSettlement getStationWithSettlement(@NotNull TPointKey pointKey) {
        switch (pointKey.getType()) {
            case POINT_KEY_TYPE_STATION:
                var station = stationDataProvider.getById(pointKey.getId());
                int settlementId = station.getSettlementId();
                if (settlementId != 0) {
                    return new StationWithSettlement(station, settlementDataProvider.getById(settlementId));
                }
                return new StationWithSettlement(station, null);
            case POINT_KEY_TYPE_SETTLEMENT:
                return new StationWithSettlement(null, settlementDataProvider.getById(pointKey.getId()));
            default:
                throw new RuntimeException(String.format("Unexpected point type %s", pointKey.getType()));
        }
    }

    @NotNull
    public BusPointInfo makeTitlePointInfo(
            @NotNull String supplierDescription, @NotNull TPointKey pointKey, @NotNull TPointKey queryPointKey) {
        var builder = BusPointInfo.builder()
                .supplierDescription(supplierDescription);
        var queryStationWithSettlement = getStationWithSettlement(queryPointKey);

        if (ProtoUtils.hasFields(pointKey)) {
            var stationWithSettlement = getStationWithSettlement(pointKey);
            if (stationWithSettlement.getSettlement() != null) {
                builder.source(stationWithSettlement.hasParent() ?
                        BusPointInfoSource.MATCHING_PARENT :
                        BusPointInfoSource.MATCHING);
                return buildPointInfo(builder, stationWithSettlement.getSettlement());
            }

            if (queryStationWithSettlement.getSettlement() != null) {
                builder.source(queryStationWithSettlement.hasParent() ?
                        BusPointInfoSource.QUERY_PARENT :
                        BusPointInfoSource.QUERY);
                return buildPointInfo(builder, queryStationWithSettlement.getSettlement());
            }

            builder.source(BusPointInfoSource.MATCHING);
            return buildPointInfo(builder, stationWithSettlement.getStation());
        }

        if (queryStationWithSettlement.getSettlement() != null) {
            builder.source(queryStationWithSettlement.hasParent() ?
                    BusPointInfoSource.QUERY_PARENT :
                    BusPointInfoSource.QUERY);
            return buildPointInfo(builder, queryStationWithSettlement.getSettlement());
        }

        builder.source(BusPointInfoSource.QUERY);
        return buildPointInfo(builder, queryStationWithSettlement.getStation());
    }

    private String getTimeZoneCode(Integer timeZoneId) {
        return timeZoneDataProvider.getById(timeZoneId).getCode();
    }
}
