package ru.yandex.travel.api.services.orders;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.generic_booking_flow.model.ContactInfoDTO;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.CreateOrderUserInfo;
import ru.yandex.travel.api.services.common.ApiSpecProtoUtils;
import ru.yandex.travel.api.services.dictionaries.country.CountryDataProvider;
import ru.yandex.travel.api.spec.buses.CreateBusesServiceParams;
import ru.yandex.travel.api.spec.buses.CreateBusesTicket;
import ru.yandex.travel.bus.model.BusBenefitType;
import ru.yandex.travel.bus.model.BusDocumentType;
import ru.yandex.travel.bus.model.BusGenderType;
import ru.yandex.travel.bus.model.BusLegalEntity;
import ru.yandex.travel.bus.model.BusRegisterType;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.model.BusRide;
import ru.yandex.travel.bus.model.BusTicketType;
import ru.yandex.travel.bus.model.BusesPassenger;
import ru.yandex.travel.buses.backend.proto.EDocumentType;
import ru.yandex.travel.buses.backend.proto.EGenderType;
import ru.yandex.travel.buses.backend.proto.ERegisterType;
import ru.yandex.travel.buses.backend.proto.ETicketType;
import ru.yandex.travel.buses.backend.proto.TCarrier;
import ru.yandex.travel.buses.backend.proto.TCitizenship;
import ru.yandex.travel.buses.backend.proto.TDocumentType;
import ru.yandex.travel.buses.backend.proto.TGender;
import ru.yandex.travel.buses.backend.proto.TPointKey;
import ru.yandex.travel.buses.backend.proto.TRide;
import ru.yandex.travel.buses.backend.proto.TSeat;
import ru.yandex.travel.buses.backend.proto.TSupplier;
import ru.yandex.travel.buses.backend.proto.TTicketType;
import ru.yandex.travel.buses.backend.proto.api.TOffer;
import ru.yandex.travel.commons.proto.ProtoUtils;

import static ru.yandex.travel.api.services.common.ApiSpecProtoUtils.consumeNotNull;

@Service
@RequiredArgsConstructor
@Slf4j
public class BusOfferMapService {
    private final TrainReservationMapService trainReservationMapService;
    private final CountryDataProvider countryDataProvider;
    private final BusModelMapService busModelMapService;

    public BusReservation createBusReservation(TOffer offer, CreateBusesServiceParams busService,
                                               CreateOrderUserInfo userInfo, ContactInfoDTO contactInfo) {
        var payload = new BusReservation();
        UUID offerId = UUID.fromString(offer.getId());
        payload.setOfferId(offerId);
        payload.setEmail(contactInfo.getEmail());
        payload.setPhone(contactInfo.getPhone());
        payload.setMoscowRegion(trainReservationMapService.isMoscowRegion(userInfo.getGeoId()));

        payload.setRide(convertRide(
                offer.getRide(), offer.getBookParams().getTicketTypesList(), offer.getQueryFrom(), offer.getQueryTo()));

        payload.setRequestPassengers(convertRequestPassengers(busService.getTicketsList(), offer));
        return payload;
    }

    private List<BusesPassenger> convertRequestPassengers(List<CreateBusesTicket> createBusesTicketList, TOffer offer) {
        Map<EGenderType, String> genderPartnerIdMap = offer.getBookParams().getGendersList().stream()
                .collect(Collectors.toMap(TGender::getId, TGender::getPartnerId, (a, b) -> {
                    log.error(String.format("Duplicate values (%s, %s) of genders. Offer id: %s", a, b,
                            offer.getId()));
                    return a;
                }));
        Map<String, String> citizenshipPartnerIdMap = offer.getBookParams().getCitizenshipsList().stream()
                .collect(Collectors.toMap(TCitizenship::getId, TCitizenship::getPartnerId, (a, b) -> {
                    log.error(String.format("Duplicate values (%s, %s) of citizenship. Offer id: %s", a, b,
                            offer.getId()));
                    return a;
                }));
        Map<EDocumentType, String> documentTypePartnerIdMap = offer.getBookParams().getDocumentsList().stream()
                .collect(Collectors.toMap(TDocumentType::getId, TDocumentType::getPartnerId, (a, b) -> {
                    log.error(String.format("Duplicate values (%s, %s) of document type. Offer id: %s", a, b,
                            offer.getId()));
                    return a;
                }));
        Map<ETicketType, String> ticketTypePartnerIdMap = offer.getBookParams().getTicketTypesList().stream()
                .collect(Collectors.toMap(TTicketType::getId, TTicketType::getPartnerId, (a, b) -> {
                    log.error(String.format("Duplicate values (%s, %s) of ticket type. Offer id: %s", a, b,
                            offer.getId()));
                    return a;
                }));
        Map<String, String> seatPartnerIdMap = offer.getBookParams().getSeatsList().stream()
                .collect(Collectors.toMap(TSeat::getId, TSeat::getPartnerId, (a, b) -> {
                    log.error(String.format("Duplicate values (%s, %s) of seats. Offer id: %s", a, b,
                            offer.getId()));
                    return a;
                }));
        var result = new ArrayList<BusesPassenger>();
        for (var createTicket : createBusesTicketList) {
            var passenger = new BusesPassenger();
            passenger.setFirstName(createTicket.getPassenger().getFirstName());
            passenger.setMiddleName(createTicket.getPassenger().getPatronymic());
            passenger.setLastName(createTicket.getPassenger().getLastName());
            passenger.setBirthday(ApiSpecProtoUtils.localDateFromProto(createTicket.getPassenger().getBirthDate()));
            BusGenderType gender = BusModelMapService.genderFromSpec(createTicket.getPassenger().getSex());
            passenger.setGender(gender);
            passenger.setGenderPartnerId(genderPartnerIdMap.get(gender.getProtoValue()));
            int citizenshipGeoId = createTicket.getPassenger().getCitizenship();
            String citizenshipCode = countryDataProvider.getByGeoId(citizenshipGeoId).getCode();
            passenger.setCitizenship(citizenshipCode);
            passenger.setCitizenshipPartnerId(citizenshipPartnerIdMap.get(citizenshipCode));
            BusDocumentType documentType =
                    BusModelMapService.documentTypeFromSpec(createTicket.getPassenger().getDocumentType());
            passenger.setDocumentType(documentType);
            passenger.setDocumentTypePartnerId(documentTypePartnerIdMap.get(documentType.getProtoValue()));
            passenger.setDocumentNumber(createTicket.getPassenger().getDocumentNumber());
            BusTicketType ticketType = BusModelMapService.ticketTypeFromSpec(createTicket.getTicketType());
            passenger.setTicketType(ticketType);
            passenger.setTicketTypePartnerId(ticketTypePartnerIdMap.get(ticketType.getProtoValue()));
            passenger.setSeatId(createTicket.getSeat());
            passenger.setSeatPartnerId(seatPartnerIdMap.get(createTicket.getSeat()));
            result.add(passenger);
        }
        return result;
    }

    private BusRide convertRide(TRide source, List<TTicketType> ticketTypes, TPointKey queryFrom, TPointKey queryTo) {
        var result = new BusRide();
        result.setRideId(source.getId());
        result.setSupplierId(source.getSupplierId());
        if (source.hasSupplier()) {
            result.setSupplier(convertLegalEntity(source.getSupplier()));
        }
        result.setCarrierCode(source.getCarrierCode());
        if (source.hasCarrier()) {
            result.setCarrier(convertLegalEntity(source.getCarrier()));
        } else { // TODO BUSES-1579 remove else
            result.setCarrier(buildCarrier(source.getCarrierName()));
        }
        result.setDepartureTime(convertSeconds(source.getDepartureTime()));
        result.setArrivalTime(convertSeconds(source.getArrivalTime()));
        result.setDuration(source.getDuration());
        result.setPointFrom(
                busModelMapService.makePointInfo(source.getFromDesc(), source.getFrom()));
        result.setPointTo(
                busModelMapService.makePointInfo(source.getToDesc(), source.getTo()));
        result.setTitlePointFrom(
                busModelMapService.makeTitlePointInfo(source.getFromDesc(), source.getFrom(), queryFrom));
        result.setTitlePointTo(
                busModelMapService.makeTitlePointInfo(source.getToDesc(), source.getTo(), queryTo));
        result.setPrice(ProtoUtils.fromTPrice(source.getPrice()));
        result.setFee(ProtoUtils.fromTPrice(source.getFee()));
        result.setYandexFee(ProtoUtils.fromTPrice(source.getYandexFee()));
        result.setTicketTypeToFee(ticketTypes.stream()
                .collect(Collectors.toMap(
                        TTicketType::getId,
                        tt -> ProtoUtils.fromTPrice(tt.hasPartnerFee() ? tt.getPartnerFee() : source.getFee()),
                        Money::add
                )));
        result.setFreeSeats(source.getFreeSeats());
        result.setTicketLimit(source.getTicketLimit());
        result.setBus(source.getBus());
        result.setRouteName(source.getRouteName());
        result.setRouteNumber(source.getRouteNumber());
        result.setBenefits(source.getBenefitsList().stream()
                .map(BusBenefitType.BY_PROTO::getByValue).collect(Collectors.toList()));
        result.setCanPayOffline(source.getCanPayOffline());
        result.setBookOnly(source.getBookOnly());
        result.setOnlineRefund(source.getOnlineRefund());
        result.setRefundConditions(source.getRefundConditions());
        return result;
    }

    private Instant convertSeconds(long seconds) {
        return Instant.ofEpochSecond(seconds);
    }

    @NonNull
    private BusLegalEntity buildCarrier(@NonNull String name) {
        var result = new BusLegalEntity();
        result.setName(name);
        result.setLegalName(name);
        result.setRegisterNumber("");
        return result;
    }

    private BusLegalEntity convertLegalEntity(TCarrier source) {
        var result = new BusLegalEntity();
        if (source.getRegisterType() != ERegisterType.REGISTER_TYPE_UNKNOWN) {
            result.setRegisterType(BusRegisterType.BY_PROTO.getByValue(source.getRegisterType()));
        }
        consumeNotNull(result::setRegisterNumber, source.getRegisterNumber());
        consumeNotNull(result::setName, source.getName());
        consumeNotNull(result::setLegalName, source.getLegalName());
        consumeNotNull(result::setLegalAddress, source.getLegalAddress());
        consumeNotNull(result::setActualAddress, source.getActualAddress());
        consumeNotNull(result::setTaxationNumber, source.getInn());
        consumeNotNull(result::setTimetable, source.getTimetable());
        return result;
    }

    private BusLegalEntity convertLegalEntity(TSupplier source) {
        var result = new BusLegalEntity();
        if (source.getRegisterType() != ERegisterType.REGISTER_TYPE_UNKNOWN) {
            result.setRegisterType(BusRegisterType.BY_PROTO.getByValue(source.getRegisterType()));
        }
        consumeNotNull(result::setRegisterNumber, source.getRegisterNumber());
        consumeNotNull(result::setName, source.getName());
        consumeNotNull(result::setLegalName, source.getLegalName());
        consumeNotNull(result::setLegalAddress, source.getLegalAddress());
        consumeNotNull(result::setActualAddress, source.getActualAddress());
        consumeNotNull(result::setTaxationNumber, source.getInn());
        consumeNotNull(result::setTimetable, source.getTimetable());
        return result;
    }
}
