package ru.yandex.travel.api.services.orders;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.booking_flow.model.CurrentPaymentDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.OrderPriceInfo;
import ru.yandex.travel.api.endpoints.booking_flow.model.PaymentErrorCode;
import ru.yandex.travel.api.endpoints.booking_flow.model.PaymentErrorDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.PromoCodeApplicationResult;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.Mir2020PromoCampaignDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.PromoCampaignsDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.Taxi2020PromoCampaignDto;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.CancellationReason;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.ContactInfoDTO;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.GenericOrderState;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.PaymentDTO;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.PaymentReceiptItem;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.PaymentReceiptItemType;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.ServiceDTO;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.ServiceState;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.ServiceType;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.refund.RefundDto;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.refund.RefundPartCtx;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.refund.RefundPartInfo;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.refund.RefundPartState;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.refund.RefundPartType;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.refund.RefundState;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.refund.RefundType;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderStateBatchRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderStateRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.OrderAggregateStateForBatch;
import ru.yandex.travel.api.infrastucture.ApiTokenEncrypter;
import ru.yandex.travel.api.models.common.PromoCodeApplicationResultType;
import ru.yandex.travel.api.services.common.ApiSpecProtoUtils;
import ru.yandex.travel.api.services.orders.suburban.SuburbanHelper;
import ru.yandex.travel.api.spec.buses.ServiceInfo;
import ru.yandex.travel.bus.service.BusPartKeyService;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.proto.EGenericOrderAggregateState;
import ru.yandex.travel.orders.proto.ETaxi2020PromoStatusEnum;
import ru.yandex.travel.orders.proto.TMir2020PromoCampaignInfo;
import ru.yandex.travel.orders.proto.TOrderAggregateState;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TOrderInvoiceInfo;
import ru.yandex.travel.orders.proto.TOrderPriceInfo;
import ru.yandex.travel.orders.proto.TOrderServiceInfo;
import ru.yandex.travel.orders.proto.TPromoCodeApplicationResult;
import ru.yandex.travel.orders.proto.TRefundPartInfo;
import ru.yandex.travel.orders.proto.TTaxi2020PromoCampaignInfo;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.service.TrainPartKeyService;

@Service
@RequiredArgsConstructor
public class GenericModelMapService {
    private final static Map<EGenericOrderAggregateState, GenericOrderState> GENERIC_ORDER_AGGREGATE_STATE_MAP =
            Map.ofEntries(
                    Map.entry(EGenericOrderAggregateState.GOAG_NEW, GenericOrderState.NEW),
                    Map.entry(EGenericOrderAggregateState.GOAG_WAITING_RESERVATION,
                            GenericOrderState.WAITING_RESERVATION),
                    Map.entry(EGenericOrderAggregateState.GOAG_RESERVED, GenericOrderState.RESERVED),
                    Map.entry(EGenericOrderAggregateState.GOAG_STARTING_PAYMENT, GenericOrderState.STARTING_PAYMENT),
                    Map.entry(EGenericOrderAggregateState.GOAG_WAITING_PAYMENT, GenericOrderState.WAITING_PAYMENT),
                    Map.entry(EGenericOrderAggregateState.GOAG_PAYMENT_FAILED, GenericOrderState.PAYMENT_FAILED),
                    Map.entry(EGenericOrderAggregateState.GOAG_WAITING_CONFIRMATION,
                            GenericOrderState.WAITING_CONFIRMATION),
                    Map.entry(EGenericOrderAggregateState.GOAG_CONFIRMED, GenericOrderState.CONFIRMED),
                    Map.entry(EGenericOrderAggregateState.GOAG_WAITING_CANCELLATION,
                            GenericOrderState.WAITING_CANCELLATION),
                    Map.entry(EGenericOrderAggregateState.GOAG_CANCELLED, GenericOrderState.CANCELLED),
                    Map.entry(EGenericOrderAggregateState.GOAG_WAITING_REFUND, GenericOrderState.WAITING_REFUND),
                    Map.entry(EGenericOrderAggregateState.GOAG_REFUNDED, GenericOrderState.REFUNDED)
            );
    private final TrainModelMapService trainModelMapService;
    private final BusModelMapService busModelMapService;
    private final ApiTokenEncrypter apiTokenEncrypter;
    private final SuburbanHelper suburbanHelper;

    public GetGenericOrderStateRspV1 getStateFromAggregate(TOrderAggregateState aggregateState) {
        var result = new GetGenericOrderStateRspV1();
        result.setVersionHash(aggregateState.getVersionHash());
        result.setState(GENERIC_ORDER_AGGREGATE_STATE_MAP.get(aggregateState.getGenericOrderAggregateState()));
        return result;
    }

    public GetGenericOrderStateBatchRspV1 getStateFromAggregateBatch(List<TOrderAggregateState> batchAggregateState) {
        var result = new GetGenericOrderStateBatchRspV1();
        result.setOrders(new ArrayList<>());
        for (TOrderAggregateState ags : batchAggregateState) {
            OrderAggregateStateForBatch s = new OrderAggregateStateForBatch();
            s.setOrderId(UUID.fromString(ags.getOrderId()));
            s.setState(GENERIC_ORDER_AGGREGATE_STATE_MAP.get(ags.getGenericOrderAggregateState()));
            s.setVersionHash(ags.getVersionHash());
            result.getOrders().add(s);
        }
        return result;
    }

    public GetGenericOrderRspV1 getOrderFromInfo(TOrderInfo source) {
        GetGenericOrderRspV1 result = new GetGenericOrderRspV1();
        UUID orderId = UUID.fromString(source.getOrderId());
        result.setId(orderId);
        result.setPrettyId(source.getPrettyId());
        GetGenericOrderStateRspV1 state = getStateFromAggregate(source.getOrderAggregateState());
        result.setState(state.getState());
        result.setDisplayState(source.getDisplayOrderState());
        result.setCancellationReason(CancellationReason.BY_PROTO.getByValueOrNull(source.getCancellationReason()));
        result.setContactInfo(new ContactInfoDTO());
        result.getContactInfo().setEmail(source.getOwner().getEmail());
        result.getContactInfo().setPhone(source.getOwner().getPhone());
        if (source.hasExpiresAt()) {
            result.setExpiresAt(ProtoUtils.toInstant(source.getExpiresAt()));
        }
        if (source.hasServicedAt()) {
            result.setServicedAt(ProtoUtils.toInstant(source.getServicedAt()));
        }
        if (source.hasPriceInfo()) {
            result.setOrderPriceInfo(convertOrderPriceInfo(source.getPriceInfo()));
        }
        if (source.hasCurrentInvoice()) {
            result.setPayment(getPaymentInfoFromInvoice(source.getCurrentInvoice()));
        }
        result.setServices(new ArrayList<>());
        for (TOrderServiceInfo service : source.getServiceList()) {
            result.getServices().add(getServiceFromInfo(service, source));
        }
        result.setReloadOrderAt(source.getServiceList().stream()
                .map(this::getServiceReloadAt)
                .filter(Objects::nonNull)
                .min(Comparator.naturalOrder())
                .orElse(null));
        updateRefundParts(result, source.getRefundPartsList());
        return result;
    }

    private void updateRefundParts(GetGenericOrderRspV1 result, List<TRefundPartInfo> refundParts) {
        if (refundParts == null || refundParts.size() == 0) {
            return;
        }
        Map<String, TRefundPartInfo> refundPartsMap =
                refundParts.stream().collect(Collectors.toMap(TRefundPartInfo::getKey, x -> x));
        result.setRefundPartInfo(getRefundPartInfo(refundPartsMap.get("")));
        for (ServiceDTO service : result.getServices()) {
            if (service.getTrainInfo() != null) {
                service.setRefundPartInfo(getRefundPartInfo(refundPartsMap.get(TrainPartKeyService.getServiceKey(service.getId()))));
                for (var passenger : service.getTrainInfo().getPassengers()) {
                    if (passenger.getCustomerId() != null) {
                        passenger.setRefundPartInfo(getRefundPartInfo(refundPartsMap.get(
                                TrainPartKeyService.getPassengerKey(service.getId(), passenger.getCustomerId()))));
                    }
                }
            } else if (service.getBusInfo() != null) {
                service.setRefundPartInfo(getRefundPartInfo(refundPartsMap.get(BusPartKeyService.getServiceKey(service.getId()))));
                ServiceInfo.Builder builder = service.getBusInfo().toBuilder();
                for (var ticket : builder.getTicketsBuilderList()) {
                    ticket.setRefundPartInfo(ApiSpecProtoUtils.refundPartInfoToProto(
                            refundPartsMap.get(BusPartKeyService.getTicketKey(service.getId(), ticket.getId())),
                            apiTokenEncrypter));
                }
                service.setBusInfo(builder.build());
            }
        }
    }

    // TODO(ganintsev): use ApiProtoUtils.refundPartInfoToProto
    private RefundPartInfo getRefundPartInfo(TRefundPartInfo source) {
        if (source == null) {
            return null;
        }
        var rp = new RefundPartInfo();
        rp.setState(RefundPartState.BY_PROTO.getByValue(source.getState()));
        rp.setContext(new RefundPartCtx());
        rp.setType(RefundPartType.BY_PROTO.getByValue(source.getType()));
        rp.getContext().setInfo(source.getContext());
        if (source.hasRefund()) {
            rp.setRefund(new RefundDto());
            rp.getRefund().setId(ProtoUtils.fromStringOrNull(source.getRefund().getId()));
            rp.getRefund().setType(RefundType.BY_PROTO.get(source.getRefund().getRefundType()));
            rp.getRefund().setState(RefundState.fromState(source.getRefund().getState()));
            if (source.getRefund().hasRefundAmount()) {
                rp.getRefund().setRefundAmount(ProtoUtils.fromTPrice(source.getRefund().getRefundAmount()));
            }
            if (source.getRefund().hasRefundBlankToken()) {
                rp.getRefund().setRefundBlankToken(apiTokenEncrypter.toDownloadBlankToken(source.getRefund().getRefundBlankToken()));
            }
            rp.getRefund().setPaymentRefundReceiptUrls(source.getRefund().getPaymentRefundReceiptUrlsList().stream()
                    .map(ApiSpecProtoUtils::convertReceiptUrlToPdf).collect(Collectors.toList()));
        }
        return rp;
    }

    private ServiceDTO getServiceFromInfo(TOrderServiceInfo source, TOrderInfo orderInfo) {
        var result = new ServiceDTO();
        result.setId(UUID.fromString(source.getServiceId()));
        result.setState(getServiceStateGeneric(source.getServiceInfo().getGenericOrderItemState()));
        switch (source.getServiceType()) {
            case PT_TRAIN:
                boolean insurancePricingFailed = orderInfo.getServiceList().stream()
                        .filter(s -> s.getServiceType() == EServiceType.PT_TRAIN)
                        .map(s -> s.getServiceInfo().getPayload())
                        .map(p -> ProtoUtils.fromTJson(p, TrainReservation.class))
                        .anyMatch(p -> p.getInsuranceStatus() == InsuranceStatus.PRICING_FAILED);
                result.setServiceType(ServiceType.TRAIN);
                UUID orderId = UUID.fromString(orderInfo.getOrderId());
                result.setTrainInfo(trainModelMapService.convertTrainServiceInfo(source, orderId, insurancePricingFailed));
                break;
            case PT_BUS:
                result.setServiceType(ServiceType.BUS);
                result.setBusInfo(busModelMapService.buildBusServiceInfo(source, orderInfo.getDocumentUrl(), orderInfo.getOrderId()));
                break;
            case PT_SUBURBAN:
                result.setServiceType(ServiceType.SUBURBAN);
                result.setSuburbanInfo(suburbanHelper.buildSuburbanServiceInfo(source));
                break;
            case PT_BNOVO_HOTEL:
            case PT_DOLPHIN_HOTEL:
            case PT_EXPEDIA_HOTEL:
                result.setServiceType(ServiceType.HOTEL);
                // TODO: result.setHotelInfo();
                // break;
            default:
                throw new RuntimeException(String.format("Unsupported service type %s", source.getServiceType()));
        }
        return result;
    }

    private Instant getServiceReloadAt(TOrderServiceInfo source) {
        switch (source.getServiceType()) {
            case PT_TRAIN:
                return trainModelMapService.getServiceReloadAt(source);
            case PT_SUBURBAN:
            case PT_BNOVO_HOTEL:
            case PT_DOLPHIN_HOTEL:
            case PT_EXPEDIA_HOTEL:
            default:
                return null;
        }
    }

    private ServiceState getServiceStateGeneric(EOrderItemState genericOrderItemState) {
        switch (genericOrderItemState) {
            case IS_CANCELLING:
            case IS_CANCELLED:
                return ServiceState.CANCELLED;
            case IS_RESERVED:
                return ServiceState.RESERVED;
            case IS_CONFIRMED:
                return ServiceState.CONFIRMED;
            case IS_REFUNDED:
                return ServiceState.REFUNDED;
            default:
                return ServiceState.IN_PROGRESS;
        }
    }

    private PaymentDTO getPaymentInfoFromInvoice(TOrderInvoiceInfo source) {
        CurrentPaymentDto current = CurrentPaymentDto.builder()
                .paymentUrl(source.getPaymentUrl())
                .purchaseToken(source.getPurchaseToken())
                .build();
        PaymentErrorDto error = null;
        if (!Strings.isNullOrEmpty(source.getAuthorizationErrorCode())) {
            error = PaymentErrorDto.builder()
                    .code(PaymentErrorCode.fromErrorCode(source.getAuthorizationErrorCode()))
                    .build();
        }

        var builder = PaymentDTO.builder()
                .current(current)
                .error(error);

        builder.receipts(source.getFiscalReceiptList().stream()
                .map(r -> new PaymentReceiptItem(
                        ApiSpecProtoUtils.convertReceiptUrlToPdf(r.getUrl()),
                        PaymentReceiptItemType.BY_PROTO.getByValue(r.getType())
                )).collect(Collectors.toList()));
        return builder.build();
    }

    private OrderPriceInfo convertOrderPriceInfo(TOrderPriceInfo source) {
        var orderPriceInfo = new OrderPriceInfo();
        orderPriceInfo.setOriginalPrice(ProtoUtils.fromTPrice(source.getOriginalPrice()));
        orderPriceInfo.setDiscountAmount(ProtoUtils.fromTPrice(source.getDiscountAmount()));
        orderPriceInfo.setPrice(ProtoUtils.fromTPrice(source.getPrice()));

        if (source.hasPromoCampaignsInfo()) {
            orderPriceInfo.setPromoCampaigns(new PromoCampaignsDto());
            TTaxi2020PromoCampaignInfo taxi2020 = source.getPromoCampaignsInfo().getTaxi2020();
            orderPriceInfo.getPromoCampaigns().setTaxi2020(new Taxi2020PromoCampaignDto());
            orderPriceInfo.getPromoCampaigns().getTaxi2020().setEligible(taxi2020.getStatus() == ETaxi2020PromoStatusEnum.OTPS_ELIGIBLE);
            if (source.getPromoCampaignsInfo().hasMir2020()) {
                TMir2020PromoCampaignInfo mir2020 = source.getPromoCampaignsInfo().getMir2020();
                Mir2020PromoCampaignDto mirDto = new Mir2020PromoCampaignDto();
                orderPriceInfo.getPromoCampaigns().setMir2020(mirDto);
                if (mir2020.getEligible()) {
                    mirDto.setEligible(true);
                    mirDto.setCashbackAmount(mir2020.getCashbackAmount().getValue());
                } else {
                    mirDto.setEligible(false);
                }
            }
        }

        if (source.getPromoCodeApplicationResultsCount() > 0) {
            orderPriceInfo.setPromoCodeApplicationResults(new ArrayList<>());
            source.getPromoCodeApplicationResultsList().forEach(
                    ar -> orderPriceInfo.getPromoCodeApplicationResults().add(convertProtoCodeApplicationResult(ar))
            );

        }
        return orderPriceInfo;
    }

    private PromoCodeApplicationResult convertProtoCodeApplicationResult(TPromoCodeApplicationResult v) {
        PromoCodeApplicationResult p = new PromoCodeApplicationResult();
        p.setCode(v.getCode());
        if (v.hasDiscountAmount()) {
            p.setDiscountAmount(ProtoUtils.fromTPrice(v.getDiscountAmount()));
        }
        p.setType(PromoCodeApplicationResultType.fromProto(v.getType()));
        return p;
    }
}
