package ru.yandex.travel.api.services.orders;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import io.grpc.Status;
import io.opentracing.Span;
import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.generic_booking_flow.model.CreateTrainServiceData;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.GetOrderRequestSource;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.refund.RefundPartCtx;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.CalculateRefundAmountReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.CalculateRefundAmountRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.CancelGenericOrderReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.CreateGenericOrderReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.DownloadBlankReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GenericOrderAddServiceReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GenericOrderAddServiceRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderStateBatchReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderStateBatchRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderStateReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderStateRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.StartGenericOrderPaymentReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.StartRefundReqV1;
import ru.yandex.travel.api.infrastucture.ApiTokenEncrypter;
import ru.yandex.travel.api.infrastucture.MetricUtils;
import ru.yandex.travel.api.services.common.PdfDownloadService;
import ru.yandex.travel.api.services.orders.suburban.SuburbanHelper;
import ru.yandex.travel.api.services.train.TrainOfferService;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.service.BusesService;
import ru.yandex.travel.buses.backend.proto.api.GetOfferRequest;
import ru.yandex.travel.buses.backend.proto.api.TOffer;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.experiments.KVExperiments;
import ru.yandex.travel.commons.experiments.OrderExperiments;
import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryStrategyBuilder;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TAddServiceReq;
import ru.yandex.travel.orders.proto.TCalculateRefundReqV2;
import ru.yandex.travel.orders.proto.TCheckoutReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TDownloadBlankToken;
import ru.yandex.travel.orders.proto.TGetOrderAggregateStateBatchReq;
import ru.yandex.travel.orders.proto.TGetOrderAggregateStateReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderServiceInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartCancellationReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.tracing.TracerHelpers;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.ImClientRetryableException;
import ru.yandex.travel.train.partners.im.model.OrderReservationBlankRequest;


@Service
@RequiredArgsConstructor
@Slf4j
@EnableConfigurationProperties(GenericOrdersServiceProperties.class)
public class GenericOrdersService {
    private final GenericOrdersServiceProperties genericOrdersServiceProperties;
    private final OrchestratorClientFactory orchestratorClientFactory;
    private final GenericModelMapService modelMapService;
    private final TrainReservationMapService trainReservationMapService;
    private final BusOfferMapService busOfferMapService;
    private final SuburbanHelper suburbanHelper;
    private final Meters meters;
    private final Tracer tracer;
    private final Retry retryHelper;
    private final TrainOfferService offerService;
    private final ApiTokenEncrypter apiTokenEncrypter;
    private final ImClient imClient;
    private final BusesService busesService;
    private final PdfDownloadService pdfDownloadService;

    public CompletableFuture<GetGenericOrderRspV1> createOrder(CreateGenericOrderReqV1 request,
                                                               UserCredentials userCredentials,
                                                               Map<String, String> experiments) {
        try {
            return createOrderInner(request, userCredentials, experiments);
        } catch (Exception e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private CompletableFuture<GetGenericOrderRspV1> createOrderInner(CreateGenericOrderReqV1 request,
                                                                     UserCredentials userCredentials,
                                                                     Map<String, String> experiments) {
        Span span = tracer.activeSpan();
        TUserInfo.Builder userInfo = TUserInfo.newBuilder()
                .setIp(Strings.nullToEmpty(request.getUserInfo().getIp()))
                .setEmail(request.getContactInfo().getEmail())
                .setPhone(request.getContactInfo().getPhone())
                .setGeoId(request.getUserInfo().getGeoId() != null ? request.getUserInfo().getGeoId() : 0)
                .setYandexUid(userCredentials.getYandexUid())
                .setPassportId(Strings.nullToEmpty(userCredentials.getPassportId()))
                .setLogin(Strings.nullToEmpty(userCredentials.getLogin()));

        TCreateOrderReq.Builder reqBuilder = TCreateOrderReq.newBuilder()
                .setOwner(userInfo)
                .setOrderType(EOrderType.OT_GENERIC)
                .setDeduplicationKey(request.getDeduplicationKey().toString())
                .setCurrency(ECurrency.C_RUB);
        // TODO remove after TRAVELBACK-2252
        if (request.getLabel() != null) {
            reqBuilder.setLabel(request.getLabel());
        }
        if (!Strings.isNullOrEmpty(request.getPaymentTestContextToken())) {
            reqBuilder.setPaymentTestContext(
                    apiTokenEncrypter.fromPaymentTestContextToken(request.getPaymentTestContextToken()));
            reqBuilder.setMockPayment(true);
        }
        CompletableFuture<TCreateOrderReq.Builder> future = CompletableFuture.completedFuture(reqBuilder);

        if (request.getTrainServices() != null) {
            List<CreateTrainServiceData> trainServices = request.getTrainServices();
            for (int i = 0; i < trainServices.size(); i++) {
                CreateTrainServiceData trs = trainServices.get(i);
                var itemIndex = i;
                future = future.thenCompose(builder -> offerService.get(trs.getOfferId()).thenApply(offer -> {
                    TrainReservation payload = trainReservationMapService.createTrainReservation(trs,
                            request.getUserInfo().isMobile(), request.getUserInfo().getGeoId(), offer);
                    payload.setPartnerItemIndex(itemIndex);
                    payload.setPartnerMultiOrder(trainServices.size() > 1);
                    TJson payloadJson = ProtoUtils.toTJson(payload);
                    log.info("Creating train service with payload: {}", payloadJson.getValue());
                    TCreateServiceReq.Builder serviceReqBuilder = TCreateServiceReq.newBuilder()
                            .setServiceType(EServiceType.PT_TRAIN)
                            .setSourcePayload(payloadJson);
                    if (!Strings.isNullOrEmpty(trs.getTrainTestContextToken())) {
                        var testContext = apiTokenEncrypter.fromTrainTestContextToken(trs.getTrainTestContextToken());
                        serviceReqBuilder.setTrainTestContext(testContext);
                    }
                    builder.addCreateServices(serviceReqBuilder);
                    return builder;
                }));
            }
        }
        if (request.getBusesServices() != null) {
            for (var busService : request.getBusesServices()) {
                future = future.thenCompose(builder -> getBusOffer(busService.getOfferId()).thenApply(offer -> {
                    BusReservation payload = busOfferMapService.createBusReservation(offer, busService,
                            request.getUserInfo(), request.getContactInfo());
                    TJson payloadJson = ProtoUtils.toTJson(payload);
                    log.info("Creating bus service with payload: {}", payloadJson.getValue());
                    TCreateServiceReq.Builder serviceReqBuilder = TCreateServiceReq.newBuilder()
                            .setServiceType(EServiceType.PT_BUS)
                            .setSourcePayload(payloadJson);
                    if (!Strings.isNullOrEmpty(busService.getBusTestContextToken())) {
                        var testContext =
                                apiTokenEncrypter.fromBusTestContextToken(busService.getBusTestContextToken());
                        serviceReqBuilder.setBusTestContext(testContext);
                    }
                    builder.addCreateServices(serviceReqBuilder);
                    return builder;
                }));
            }
        }

        reqBuilder.addAllCreateServices(suburbanHelper.buildSuburbanServicesReqs(request.getSuburbanServices()));
        reqBuilder.addAllKVExperiments(KVExperiments.toProto(experiments));

        // TODO: add hotelServices

        return future.thenCompose(
                req -> createAndReserveOrder(req.build(), request.getOrderHistory().toString(), span)
        );
    }

    private CompletableFuture<TOffer> getBusOffer(String offerId) {
        return busesService.getOffer(GetOfferRequest.newBuilder().setOfferId(offerId).build()).thenApply(rsp -> {
            if (Strings.isNullOrEmpty(rsp.getError())) {
                return rsp.getOffer();
            }
            throw new RuntimeException(rsp.getError());
        });
    }

    public CompletableFuture<GetGenericOrderStateRspV1> getOrderState(GetGenericOrderStateReqV1 req) {
        return FutureUtils.buildCompletableFuture(
                withSlavePreferred().getOrderAggregateState(TGetOrderAggregateStateReq.newBuilder()
                        .setOrderId(req.getOrderId().toString()).build())
        ).thenApply(aggState -> modelMapService.getStateFromAggregate(aggState.getOrderAggregateState()));
    }

    public CompletableFuture<GetGenericOrderStateBatchRspV1> getOrderStateBatch(GetGenericOrderStateBatchReqV1 req) {
        return FutureUtils.buildCompletableFuture(
                withSlavePreferred().getOrderAggregateStateBatch(
                        TGetOrderAggregateStateBatchReq.newBuilder()
                                .addAllOrderId(req.getOrderIds().stream().map(UUID::toString).collect(Collectors.toUnmodifiableList()))
                                .build()
                )
        ).thenApply(response -> modelMapService.getStateFromAggregateBatch(response.getOrderAggregateStateList()));
    }

    public CompletableFuture<GetGenericOrderRspV1> getOrderInfo(GetGenericOrderReqV1 req,
                                                                HttpServletRequest httpServletRequest) {
        try (var ignoredMdc = NestedMdc.forEntityId(req.getOrderId().toString())) {
            boolean updateOrderOnTheFly = req.getSource() == GetOrderRequestSource.ORDER_PAGE;
            log.info("Getting order info with update on the fly: {}", updateOrderOnTheFly);
            var client = withClient();
            if (updateOrderOnTheFly) {
                // TODO (mbobrov): think of propagating time remaining through retries
                client = client.withDeadlineAfter(
                        genericOrdersServiceProperties.getGetActualizedOrderInfoDuration().toMillis(),
                        TimeUnit.MILLISECONDS
                );
            }
            return FutureUtils.buildCompletableFuture(
                            client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(req.getOrderId().toString())
                                    .setUpdateOrderOnTheFly(updateOrderOnTheFly).build()
                            )
                    ).thenApply(orderRsp -> modelMapService.getOrderFromInfo(orderRsp.getResult()))
                    .handle((rsp, t) -> {
                        var errorCode = Optional.ofNullable(t)
                                .map(Status::trailersFromThrowable)
                                .map(x -> x.get(ServerUtils.METADATA_ERROR_KEY))
                                .map(TError::getCode)
                                .filter(x -> x == EErrorCode.EC_IM_RETRYABLE_ERROR || x == EErrorCode.EC_CALL_TO_IM_OVERLOADED);
                        MetricUtils.addRequestMetricTag(httpServletRequest, "partner_error",
                                Boolean.toString(errorCode.isPresent()), true);
                        if (t != null) {
                            return CompletableFuture.<GetGenericOrderRspV1>failedFuture(t);
                        }
                        return CompletableFuture.completedFuture(rsp);
                    })
                    .thenCompose(x -> x);
        }
    }

    private CompletableFuture<GetGenericOrderRspV1> createAndReserveOrder(TCreateOrderReq req,
                                                                          String orderHistory, Span span) {
        return FutureUtils.buildCompletableFuture(
                        withClient().createOrder(req))
                .thenCompose(TracerHelpers.applyWithActiveSpanScope(tracer, span,
                        createResp -> FutureUtils.buildCompletableFuture(
                                withClient().reserve(TReserveReq.newBuilder()
                                        .setOrderId(createResp.getNewOrder().getOrderId())
                                        .setCallId(CallIdHelper.callIdForMethod("reserveOrder"))
                                        .build()
                                )).thenApply(reserveRsp -> createResp)))
                .thenApply(createResp -> {
                    try (var ignoredMdc = NestedMdc.forEntityId(createResp.getNewOrder().getOrderId())) {
                        log.info("Created order");
                        log.info(orderHistory);
                    }
                    meters.getGenericOrdersCreated().increment();
                    return modelMapService.getOrderFromInfo(createResp.getNewOrder());
                });
    }

    public CompletableFuture<Void> payOrder(StartGenericOrderPaymentReqV1 request, OrderExperiments experiments) {
        try (var ignoredMdc = NestedMdc.forEntityId(request.getOrderId().toString())) {
            log.info("Starting payment with url \"{}\" and source \"{}\"", request.getReturnUrl(), request.getSource());
            var future = FutureUtils.buildCompletableFuture(withClient().checkout(
                            TCheckoutReq.newBuilder()
                                    .setOrderId(request.getOrderId().toString()).build()))
                    .thenCompose(checkoutRsp -> FutureUtils.buildCompletableFuture(withClient().startPayment(
                            TStartPaymentReq.newBuilder()
                                    .setCallId(CallIdHelper.callIdForMethod("startPayment"))
                                    .setInvoiceType(EInvoiceType.IT_TRUST)
                                    .setOrderId(request.getOrderId().toString())
                                    .setReturnUrl(request.getReturnUrl())
                                    .setSource(request.getSource())
                                    .setNewCommonPaymentWebForm(experiments.isNewCommonPaymentWebForm())
                                    .build()
                    )));
            return future.thenApply(ignored -> null);
        }
    }

    private OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub withClient() {
        return orchestratorClientFactory.createFutureStubForTrains();
    }

    private OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub withSlavePreferred() {
        return orchestratorClientFactory.createFutureStubForTrainsReadOnly();
    }

    public CompletableFuture<Void> cancelOrder(CancelGenericOrderReqV1 request) {
        return FutureUtils.buildCompletableFuture(
                withClient().startCancellation(
                        TStartCancellationReq.newBuilder()
                                .setCallId(CallIdHelper.callIdForMethod("cancelOrder"))
                                .setOrderId(request.getOrderId().toString())
                                .build())
        ).thenApply(ignored -> null);
    }

    public CompletableFuture<GenericOrderAddServiceRspV1> addService(GenericOrderAddServiceReqV1 request) {
        var builder = TAddServiceReq.newBuilder()
                .setOrderId(request.getOrderId().toString())
                .setDeduplicationKey(request.getDeduplicationKey().toString());
        var serviceCount = (request.getTrainService() != null ? 1 : 0) +
                (request.getHotelService() != null ? 1 : 0);
        Preconditions.checkArgument(serviceCount == 1, "Request must contains only one service to add, but %s found",
                serviceCount);

        if (request.getTrainService() != null) {
            throw new UnsupportedOperationException("Adding train service is not supported");
        } else if (request.getHotelService() != null) {
            throw new UnsupportedOperationException("Adding hotel service is not supported");
        } else {
            throw new UnsupportedOperationException("Nothing added");
        }
//        return FutureUtils.buildCompletableFuture(
//                withClient().addService(builder.build())
//        ).thenApply(addServiceRsp -> {
//            var rsp = new GenericOrderAddServiceRspV1();
//            rsp.setServiceId(UUID.fromString(addServiceRsp.getServiceId()));
//            return rsp;
//        });
    }

    public CompletableFuture<CalculateRefundAmountRspV1> calculateRefundAmount(CalculateRefundAmountReqV1 request) {
        var req = TCalculateRefundReqV2.newBuilder().setOrderId(request.getOrderId().toString());
        for (RefundPartCtx ctx : request.getRefundPartContexts()) {
            req.addContext(ctx.getInfo());
        }
        return FutureUtils.buildCompletableFuture(withClient().calculateRefundV2(req.build()))
                .thenApply(calculateRefundRsp -> {
                    var result = new CalculateRefundAmountRspV1();
                    result.setExpiresAt(ProtoUtils.toInstant(calculateRefundRsp.getExpiresAt()));
                    result.setRefundToken(calculateRefundRsp.getRefundToken());
                    result.setRefundAmount(ProtoUtils.fromTPrice(calculateRefundRsp.getRefundAmount()));
                    result.setPenaltyAmount(ProtoUtils.fromTPrice(calculateRefundRsp.getPenaltyAmount()));
                    return result;
                });
    }

    public CompletableFuture<Void> startRefund(StartRefundReqV1 request) {
        return FutureUtils.buildCompletableFuture(
                        withClient().startRefund(TStartRefundReq.newBuilder()
                                .setCallId(CallIdHelper.callIdForMethod("startRefund"))
                                .setOrderId(request.getOrderId().toString())
                                .setRefundToken(request.getRefundToken()).build()))
                .thenApply(ignored -> {
                    meters.getGenericOrdersRefunded().increment();
                    return null;
                });
    }

    public CompletableFuture<ResponseEntity<byte[]>> downloadBlank(DownloadBlankReqV1 request) {
        Span span = tracer.activeSpan();
        TDownloadBlankToken token = apiTokenEncrypter.fromDownloadBlankToken(request.getToken());
        try (var ignoredMdc = NestedMdc.forEntityId(token.getOrderId())) {
            switch (token.getOneOfDownloadBlankParamsCase()) {
                case TRAINDOWNLOADBLANKPARAMS:
                    UUID orderItemId = UUID.fromString(token.getTrainDownloadBlankParams().getServiceId());
                    Integer operationId = token.getTrainDownloadBlankParams().getOperationId() == 0 ? null :
                            token.getTrainDownloadBlankParams().getOperationId();
                    log.info("Downloading train blank with operationId {}", operationId);
                    return FutureUtils.buildCompletableFuture(
                                    withClient().getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(token.getOrderId()).build()))
                            .thenApply(orderInfo -> createImBlankRequest(orderInfo, orderItemId, operationId))
                            .thenCompose(FutureUtils.withMdc(rq ->
                                    retryHelper.withRetry("GenericOrderService::downloadBlank",
                                            TracerHelpers.applyWithActiveSpanScope(
                                                    tracer, span, blankRequest ->
                                                            imClient.orderReservationBlankAsync(
                                                                    blankRequest,
                                                                    genericOrdersServiceProperties.getDownloadBlankDuration()
                                                            )
                                            ),
                                            rq,
                                            new RetryStrategyBuilder<byte[]>()
                                                    .retryOnException(e -> e instanceof ImClientRetryableException)
                                                    .setNumRetries(3)
                                                    .setTimeout(Duration.ofMillis(100))
                                                    .build()
                                    )
                            ))
                            .thenApply(bytes -> ResponseEntity
                                    .status(HttpStatus.OK)
                                    .contentType(MediaType.APPLICATION_PDF)
                                    .body(bytes));
                default:
                    return FutureUtils.buildCompletableFuture(
                                    withClient().getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(token.getOrderId()).build()))
                            .thenApply(orderInfo -> orderInfo.getResult().getDocumentUrl())
                            .thenCompose(FutureUtils.withMdc(TracerHelpers.applyWithActiveSpanScope(
                                    tracer, span,
                                    pdfDownloadService::downloadDocumentAsBytesAsync
                            )));
            }
        }
    }

    private OrderReservationBlankRequest createImBlankRequest(TGetOrderInfoRsp orderInfo, UUID orderItemId,
                                                              Integer operationId) {
        TJson payloadJson;
        TOrderServiceInfo serviceInfo = orderInfo.getResult().getServiceList().stream()
                .filter(s -> s.getServiceId().equals(orderItemId.toString()))
                .findFirst().orElseThrow(() -> new NoSuchElementException("No such service: " + orderItemId));
        payloadJson = serviceInfo.getServiceInfo().getPayload();
        TrainReservation payload = ProtoUtils.fromTJson(payloadJson, TrainReservation.class);
        return new OrderReservationBlankRequest(payload.getPartnerOrderId(), operationId);
    }
}
