package ru.yandex.travel.api.services.orders;

import java.time.Duration;
import java.util.EnumMap;

import com.google.common.collect.ImmutableMap;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Timer;
import lombok.Getter;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.services.hotels_booking_flow.AbstractPartnerBookingProvider;
import ru.yandex.travel.api.services.hotels_booking_flow.BookingFlowContext;
import ru.yandex.travel.commons.metrics.MetricsUtils;
import ru.yandex.travel.hotels.proto.EPartnerId;

@Component
public class Meters {

    private final ImmutableMap<BookingFlowContext.Stage, ImmutableMap<EPartnerId, OfferMeters>> offerMeters;

    @Getter
    private final Counter hotelOrdersCreated =
            Counter.builder("orders.hotels.created").register(Metrics.globalRegistry);
    @Getter
    private final Counter hotelOrdersRefunded =
            Counter.builder("orders.hotels.refunded").register(Metrics.globalRegistry);
    @Getter
    private final Counter trainOrdersCreated = Counter.builder("orders.train.created").register(Metrics.globalRegistry);
    @Getter
    private final Counter genericOrdersCreated = Counter.builder("orders.generic.created").register(Metrics.globalRegistry);
    @Getter
    private final Counter trainOrdersRefunded =
            Counter.builder("orders.train.refunded").register(Metrics.globalRegistry);
    @Getter
    private final Counter genericOrdersRefunded =
            Counter.builder("orders.generic.refunded").register(Metrics.globalRegistry);
    private final ImmutableMap<EPartnerId, Counter> hotelLegalDataMissing;

    public Meters() {
        //Initialize all counters for all possible stages and partners
        EnumMap<BookingFlowContext.Stage, ImmutableMap<EPartnerId, OfferMeters>> offerMetersBuilder =
                new EnumMap<>(BookingFlowContext.Stage.class);
        for (BookingFlowContext.Stage stage : BookingFlowContext.Stage.values()) {
            EnumMap<EPartnerId, OfferMeters> metersMapBuilder = new EnumMap<>(EPartnerId.class);
            for (EPartnerId partnerId : EPartnerId.values()) {
                metersMapBuilder.put(partnerId, new OfferMeters(stage, partnerId));
            }
            offerMetersBuilder.put(stage, ImmutableMap.copyOf(metersMapBuilder));
        }
        offerMeters = ImmutableMap.copyOf(offerMetersBuilder);

        EnumMap<EPartnerId, Counter> hotelLegalDataMissingBuilder = new EnumMap<>(EPartnerId.class);
        for (EPartnerId partnerId : EPartnerId.values()) {
            hotelLegalDataMissingBuilder.put(partnerId,
                    Counter.builder("hotel.legalDataMissing")
                            .tag("partner", partnerId.toString())
                            .register(Metrics.globalRegistry));
        }
        hotelLegalDataMissing = ImmutableMap.copyOf(hotelLegalDataMissingBuilder);
    }

    public OfferMeters getOfferMeters(BookingFlowContext.Stage stage) {
        return getOfferMeters(stage, EPartnerId.UNRECOGNIZED);
    }

    public OfferMeters getOfferMeters(BookingFlowContext.Stage stage, EPartnerId partnerId) {
        return offerMeters.get(stage).get(partnerId);
    }

    public void incrementHotelLegalDataMissing(EPartnerId partnerId) {
        hotelLegalDataMissing.get(partnerId).increment();
    }

    public enum PriceMismatchType {
        UP, DOWN;
    }

    @Getter
    public static class OfferMeters {
        private final BookingFlowContext.Stage stage;
        private final EPartnerId partnerId;

        private final Counter malformedToken;
        private final Counter expiredToken;
        private final Counter tokenStorageErrors;
        private final Counter geosearchHotelNotFoundErrors;
        private final Counter missingToken;
        private final Counter contentMismatch;
        private final ImmutableMap<PriceMismatchType, Counter> priceMismatch;
        private final ImmutableMap<AbstractPartnerBookingProvider.BookingProviderMethod, Counter> bookingProviderError;
        private final Counter successfulOffers;
        private final Counter soldOut;
        private final Counter failedOffers;
        private final Timer ttl;

        private OfferMeters(BookingFlowContext.Stage stage, EPartnerId partnerId) {
            this.stage = stage;
            this.partnerId = partnerId;
            malformedToken = Counter.builder("orders.hotels.malformedToken")
                    .tag("stage", stage.toString())
                    .tag("partner", partnerId.toString())
                    .register(Metrics.globalRegistry);
            expiredToken = Counter.builder("orders.hotels.expiredToken")
                    .tag("stage", stage.toString())
                    .tag("partner", partnerId.toString())
                    .register(Metrics.globalRegistry);
            missingToken = Counter.builder("orders.hotels.missingToken")
                    .tag("stage", stage.toString())
                    .tag("partner", partnerId.toString())
                    .register(Metrics.globalRegistry);
            tokenStorageErrors = Counter.builder("orders.hotels.tokenStorageErrors")
                    .tag("stage", stage.toString())
                    .tag("partner", partnerId.toString())
                    .register(Metrics.globalRegistry);
            geosearchHotelNotFoundErrors = Counter.builder("orders.hotels.geosearchHotelNotFoundErrors")
                    .tag("stage", stage.toString())
                    .tag("partner", partnerId.toString())
                    .register(Metrics.globalRegistry);
            successfulOffers = Counter.builder("orders.hotels.successfulOffers")
                    .tag("stage", stage.toString())
                    .tag("partner", partnerId.toString())
                    .register(Metrics.globalRegistry);
            if (stage == BookingFlowContext.Stage.CREATE_ORDER || stage == BookingFlowContext.Stage.ESTIMATE_DISCOUNT) {
                contentMismatch = Counter.builder("orders.hotels.contentMismatch")
                        .tag("stage", stage.toString())
                        .tag("partner", partnerId.toString())
                        .register(Metrics.globalRegistry);
            } else {
                contentMismatch = null;
            }
            EnumMap<PriceMismatchType, Counter> priceMismatchBuilder = new EnumMap<>(PriceMismatchType.class);
            for (PriceMismatchType mismatchType : PriceMismatchType.values()) {
                priceMismatchBuilder.put(mismatchType, Counter.builder("orders.hotels.priceMismatch")
                        .tag("stage", stage.toString())
                        .tag("partner", partnerId.toString())
                        .tag("direction", mismatchType.toString())
                        .register(Metrics.globalRegistry));
            }
            priceMismatch = ImmutableMap.copyOf(priceMismatchBuilder);
            EnumMap<AbstractPartnerBookingProvider.BookingProviderMethod, Counter> bookingProviderErrorBuilder =
                    new EnumMap<>(AbstractPartnerBookingProvider.BookingProviderMethod.class);
            for (AbstractPartnerBookingProvider.BookingProviderMethod method :
                    AbstractPartnerBookingProvider.BookingProviderMethod.values()) {
                bookingProviderErrorBuilder.put(method, Counter.builder("orders.hotels.bookingProviderError")
                        .tag("stage", stage.toString())
                        .tag("partner", partnerId.toString())
                        .tag("method", method.getMethodName())
                        .register(Metrics.globalRegistry));
            }
            bookingProviderError = ImmutableMap.copyOf(bookingProviderErrorBuilder);
            soldOut = Counter.builder("orders.hotels.soldOut")
                    .tag("stage", stage.toString())
                    .tag("partner", partnerId.toString())
                    .register(Metrics.globalRegistry);
            failedOffers = Counter.builder("orders.hotels.failedOffers")
                    .tag("stage", stage.toString())
                    .tag("partner", partnerId.toString())
                    .register(Metrics.globalRegistry);
            ttl = Timer.builder("orders.hotels.tokenTTL")
                    .tag("stage", stage.toString())
                    .tag("partner", partnerId.toString())
                    .serviceLevelObjectives(Duration.ofSeconds(10),
                            Duration.ofSeconds(30),
                            Duration.ofMinutes(1),
                            Duration.ofMinutes(5),
                            Duration.ofMinutes(10),
                            Duration.ofMinutes(30),
                            Duration.ofHours(1),
                            Duration.ofHours(3),
                            Duration.ofHours(6),
                            Duration.ofHours(12)
                    )
                    .publishPercentiles(MetricsUtils.higherPercentiles())
                    .register(Metrics.globalRegistry);
        }

        public Counter getPriceMismatch(PriceMismatchType mismatchType) {
            return priceMismatch.get(mismatchType);
        }

        public Counter getBookingProviderErrorCounter(AbstractPartnerBookingProvider.BookingProviderMethod method) {
            return bookingProviderError.get(method);
        }
    }


}
