package ru.yandex.travel.api.services.orders;

import java.util.Set;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.booking_flow.model.DisplayableOrderStatus;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.orders.commons.proto.TOffsetPage;
import ru.yandex.travel.orders.proto.TListOrdersReq;
import ru.yandex.travel.orders.proto.TListOrdersRsp;

import static java.util.stream.Collectors.toList;

@Service
@RequiredArgsConstructor
public class OrderListService implements TravelOrderSubtypeListFetcher {
    private final OrchestratorClientFactory orchestratorClientFactory;
    private final TravelOrderListItemMapper travelOrderListItemMapper;


    public CompletableFuture<TravelOrderSubtypeList> listOrders(int offset, int limit, Set<OrderType> orderTypes,
                                                                DisplayableOrderStatus displayableOrderStatus,
                                                                String searchTerm) {
        Preconditions.checkArgument(!orderTypes.isEmpty(), "Provided order types must be not empty");
        TListOrdersReq.Builder rqBuilder = TListOrdersReq.newBuilder()
                .setPage(TOffsetPage.newBuilder().setOffset(offset).setLimit(limit).build());

        orderTypes.forEach(ot -> rqBuilder.addTypes(ot.getOrderCommonType()));

        rqBuilder.setSearchString(Strings.nullToEmpty(searchTerm));
        if (displayableOrderStatus != null) {
            rqBuilder.addAllDisplayOrderStates(displayableOrderStatus.getProtoStates());
            rqBuilder.setDisplayOrderState(displayableOrderStatus.getProtoStates().get(0));
        }

//        .setDisplayOrderState()
        CompletableFuture<TListOrdersRsp> futureOrders =
                FutureUtils.buildCompletableFuture(orchestratorClientFactory.createRoundRobinStub().listOrders(rqBuilder.build()));
        return futureOrders.thenApply(orders -> {
            TravelOrderSubtypeList res = new TravelOrderSubtypeList();
            res.setLimit(orders.getPage().getLimit());
            res.setOffset(orders.getPage().getOffset());
            res.setHasMoreOrders(orders.getHasMoreOrders());
            res.setOrders(orders.getOrderInfoList().stream().map(travelOrderListItemMapper::map).collect(toList())
            );
            return res;
        });
    }

    @Override
    public CompletableFuture<TravelOrderSubtypeList> listOrders(int offset, int limit,
                                                                DisplayableOrderStatus displayableOrderStatus,
                                                                String searchTerm) {
        return listOrders(offset, limit, Set.of(OrderType.HOTEL), displayableOrderStatus, searchTerm);
    }
}
