package ru.yandex.travel.api.services.orders;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.booking_flow.model.OrderStatus;
import ru.yandex.travel.orders.commons.proto.EPromoCodeApplicationResultType;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TOrderInvoiceInfo;
import ru.yandex.travel.orders.proto.TPaymentInfo;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.workflow.EWorkflowState;

@Service
public class OrderStatusMappingService {
    private static final Set<ETrustInvoiceState> PAID_INVOICE_STATES =
            ImmutableSet.of(ETrustInvoiceState.IS_CANCELLED, ETrustInvoiceState.IS_REFUNDED);
    private static final Set<EPaymentState> CURRENT_PAYMENT_STATES = Set.of(
            EPaymentState.PS_INVOICE_PENDING,
            EPaymentState.PS_PAYMENT_IN_PROGRESS,
            EPaymentState.PS_PARTIALLY_PAID);


    /** @Deprecated
     * Subject to remove. Use order aggregate state instead
     */
    public OrderStatus getStatus(TOrderInfo orderInfo) {
        EWorkflowState workflowState = orderInfo.getWorkflowState();
        EHotelOrderState orderState = orderInfo.getHotelOrderState();
        boolean actionScheduled = orderInfo.getUserActionScheduled();
        List<ETrustInvoiceState> invoiceStates = orderInfo.getInvoiceList().stream().map(
                TOrderInvoiceInfo::getTrustInvoiceState
        ).collect(Collectors.toUnmodifiableList());
        Map<String, TOrderInvoiceInfo> invoiceMap = orderInfo.getInvoiceList().stream()
                .collect(Collectors.toMap(TOrderInvoiceInfo::getInvoiceId, Function.identity()));
        ETrustInvoiceState currentInvoiceState = ETrustInvoiceState.IS_UNKNOWN;
        if (orderInfo.hasCurrentInvoice()) {
            currentInvoiceState = orderInfo.getCurrentInvoice().getTrustInvoiceState();
        }
        boolean promoCodesEmptyOrSuccess = isPromoCodesEmptyOrSuccess(orderInfo);
        boolean retryPaymentAvailable = orderInfo.getPaymentRetryEnabled();

        switch (workflowState) {
            case WS_CRASHED:
                switch (orderState) {
                    case OS_WAITING_INVOICE_REFUND:
                    case OS_WAITING_SERVICE_REFUND:
                    case OS_REFUNDED:
                        return OrderStatus.REFUND_FAILED;
                    case OS_WAITING_PAYMENT:
                        return OrderStatus.PAYMENT_FAILED;
                    default:
                        return OrderStatus.FAILED;
                }
            default:
                if (actionScheduled) {
                    return OrderStatus.IN_PROGRESS;
                } else {
                    switch (orderState) {
                        case OS_CANCELLED:
                            if (invoiceStates.stream().anyMatch(PAID_INVOICE_STATES::contains)) {
                                return OrderStatus.CANCELLED_WITH_REFUND;
                            } else {
                                if (invoiceStates.stream().anyMatch(s -> s == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED)) {
                                    return OrderStatus.PAYMENT_FAILED;
                                } else {
                                    return OrderStatus.CANCELLED;
                                }
                            }
                        case OS_WAITING_PAYMENT:
                            return mapWaitingPayment(invoiceStates, promoCodesEmptyOrSuccess,
                                    retryPaymentAvailable, currentInvoiceState);
                        case OS_WAITING_EXTRA_PAYMENT:
                        case OS_CONFIRMED:
                            return mapConfirmed(orderInfo, invoiceMap);
                        case OS_REFUNDED:
                            return OrderStatus.REFUNDED;
                        case OS_MANUAL_PROCESSING:
                            return OrderStatus.FAILED;
                        default:
                            return OrderStatus.IN_PROGRESS;
                    }
                }
        }
    }

    private OrderStatus mapConfirmed(TOrderInfo orderInfo, Map<String, TOrderInvoiceInfo> invoiceMap) {
        // confirmed order may have pending invoices - for extra or deferred payments. We need to map their statuses
        // into what front-end accepts

        Optional<TPaymentInfo> currentPendingInvoice = orderInfo.getPaymentsList().stream()
                .filter(pi -> CURRENT_PAYMENT_STATES.contains(pi.getState()))
                .findFirst();
        if (currentPendingInvoice.isEmpty()) {
            // no current pending invoice: we have a fully paid confirmed order
            return OrderStatus.CONFIRMED;
        } else {
            return mapCurrentPaymentStateToConfirmedOrderState(currentPendingInvoice.get());
        }
    }

    private OrderStatus mapCurrentPaymentStateToConfirmedOrderState(TPaymentInfo currentPaymentInfo) {
        switch (currentPaymentInfo.getState()) {
            case PS_INVOICE_PENDING:
                // We have a pending, but not started invoice
                if (Strings.isNullOrEmpty(currentPaymentInfo.getActivePaymentErrorCode())) {
                    // No active errors - the invoice was never started
                    return OrderStatus.CONFIRMED;
                } else {
                    // There is an active error - the invoice was started and failed on payment
                    return OrderStatus.PAYMENT_FAILED;
                }
            case PS_PAYMENT_IN_PROGRESS:
                // we have a pending and started invoice:
                if (Strings.isNullOrEmpty(currentPaymentInfo.getActivePaymentErrorCode()) &&
                        Strings.isNullOrEmpty(currentPaymentInfo.getActivePaymentUrl())) {
                    // no active payment url yet: we are in progress
                    return OrderStatus.IN_PROGRESS;
                } else if (!Strings.isNullOrEmpty(currentPaymentInfo.getActivePaymentUrl())) {
                    // Started invoice has a trust basket waiting for payment
                    return OrderStatus.AWAITS_PAYMENT;
                } else {
                    return OrderStatus.PAYMENT_FAILED;
                }
            case PS_PARTIALLY_PAID:
                // we have a partially paid payment schedule, so we need to look at its "next" payments
                Optional<TPaymentInfo> nextPendingPayment = currentPaymentInfo.getNextPaymentsList().stream()
                        .filter(pi -> CURRENT_PAYMENT_STATES.contains(pi.getState()))
                        .findFirst();
                if (nextPendingPayment.isEmpty()) {
                    // next payment is not started yet - we are in progress
                    return OrderStatus.IN_PROGRESS;
                } else {
                    return mapCurrentPaymentStateToConfirmedOrderState(nextPendingPayment.get());
                }
            default:
                // we should not be here, as this switch covers all possible values of CURRENT_PAYMENT_STATES
                throw new IllegalStateException();
        }
    }

    private OrderStatus mapWaitingPayment(Collection<ETrustInvoiceState> invoiceStates,
                                          boolean promoCodesEmptyOrSuccess, boolean retryPaymentAvailable,
                                          ETrustInvoiceState currentInvoiceState) {
        if (retryPaymentAvailable) {
            if (currentInvoiceState == ETrustInvoiceState.IS_UNKNOWN) {
                if (promoCodesEmptyOrSuccess) {
                    return OrderStatus.RESERVED;
                } else {
                    return OrderStatus.RESERVED_WITH_RESTRICTIONS;
                }
            } else if (currentInvoiceState == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT) {
                return OrderStatus.AWAITS_PAYMENT;
            } else if (currentInvoiceState == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED) {
                return OrderStatus.PAYMENT_FAILED;
            } else {
                return OrderStatus.IN_PROGRESS;
            }
        } else {
            // we don't touch this logic to preserve existing behaviour
            if (invoiceStates.isEmpty()) {
                if (promoCodesEmptyOrSuccess) {
                    return OrderStatus.RESERVED;
                } else {
                    return OrderStatus.RESERVED_WITH_RESTRICTIONS;
                }
            } else if (invoiceStates.stream().anyMatch(is -> is == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT)) {
                return OrderStatus.AWAITS_PAYMENT;
            } else {
                return OrderStatus.IN_PROGRESS;
            }
        }
    }

    private boolean isPromoCodesEmptyOrSuccess(TOrderInfo protoOrder) {
        return protoOrder.getPriceInfo().getPromoCodeApplicationResultsCount() == 0 ||
                protoOrder.getPriceInfo().getPromoCodeApplicationResultsList().stream().allMatch(
                        aResult -> aResult.getType() == EPromoCodeApplicationResultType.ART_SUCCESS
                );
    }
}
