package ru.yandex.travel.api.services.orders;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.admin.TravelOrdersAdminMapper;
import ru.yandex.travel.api.endpoints.admin.req_rsp.AdminGetOrderReqV1;
import ru.yandex.travel.api.endpoints.booking_flow.DtoMapper;
import ru.yandex.travel.api.endpoints.booking_flow.model.HotelOrderDto;
import ru.yandex.travel.api.models.admin.Order;
import ru.yandex.travel.api.models.admin.OrderForBusinessTripDoc;
import ru.yandex.travel.api.services.hotels_booking_flow.AggregateOrderStatusProperties;
import ru.yandex.travel.api.services.hotels_booking_flow.HotelOrdersService;
import ru.yandex.travel.api.services.hotels_booking_flow.models.HotelOrder;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.orders.admin.proto.TAdminOrderInfo;
import ru.yandex.travel.orders.admin.proto.TAdminOrderInvoiceInfo;
import ru.yandex.travel.orders.admin.proto.TGetOrderReq;
import ru.yandex.travel.orders.commons.proto.ESnippet;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;

import static ru.yandex.travel.commons.concurrent.FutureUtils.buildCompletableFuture;

@Service
@RequiredArgsConstructor
@Slf4j
public class OrdersNoAuthService {
    private final DtoMapper dtoConverter;
    private final HotelOrdersService hotelOrdersService;
    private final OrchestratorClientFactory orchestratorClientFactory;
    private final AggregateOrderStatusProperties aggregateOrderStatusProperties;
    private final TravelOrdersAdminMapper adminMapper;

    public CompletableFuture<HotelOrderDto> getHotelOrderInfo(String id) {
        return getHotelOrderByIdNoAuth(id).thenApply(order -> dtoConverter.buildOrderDto(order, true));
    }

    public CompletableFuture<HotelOrder> getHotelOrderByIdNoAuth(String id) {
        return buildCompletableFuture(orchestratorClientFactory.createOrderNoAuthFutureStub()
                .getOrderInfo(TGetOrderInfoReq.newBuilder()
                        .setCalculateAggregateStateOnTheFly(aggregateOrderStatusProperties.isCalculateOnTheFly())
                        .setOrderId(id)
                        .build()))
                .thenApply(res -> hotelOrdersService.getOrderFromProto(res.getResult()));
    }

    public CompletableFuture<Order> getOrder(AdminGetOrderReqV1 request) {
        return buildCompletableFuture(orchestratorClientFactory.createOrderNoAuthFutureStub()
                .getOrder(TGetOrderReq.newBuilder()
                        .setOrderId(request.getOrderId())
                        .addAllSnippet(request.getSnippets())
                        .build()))
                .thenApply(rsp -> adminMapper.getOrderFromProto(rsp, request.getSnippets()));
    }

    public CompletableFuture<OrderForBusinessTripDoc> getOrderForBusinessTripDoc(String orderId) {
        return buildCompletableFuture(orchestratorClientFactory.createOrderNoAuthFutureStub()
                .getOrder(TGetOrderReq.newBuilder()
                        .setOrderId(orderId)
                        .addSnippet(ESnippet.S_PRIVATE_INFO)  // to get user email
                        .addSnippet(ESnippet.S_RECEIPTS_DATA)  // to get receipts data from Trust
                        .build()))
                .thenApply(rsp -> {
                    TAdminOrderInfo order = rsp.getOrderInfo();
                    HotelItinerary itinerary = ProtoUtils.fromTJson(
                            order.getOrderItem(0).getPayload(), HotelItinerary.class);
                    OrderDetails orderDetails = itinerary.getOrderDetails();

                    // Дата обращения нам не важна, т.к.юзер сам жмёт кнопку заказа документа.
                    // Чем позже, тем лучше - поэтому просто берем текущую.
                    LocalDate applicationDate = LocalDate.now();

                    List<String> receiptNumbers = order.getInvoiceList().stream()
                            .map(TAdminOrderInvoiceInfo::getFiscalReceiptList)
                            .flatMap(recList -> recList.stream().map(
                                    rec -> String.valueOf(rec.getDocumentIndex())))
                            .collect(Collectors.toList());

                    return OrderForBusinessTripDoc.builder()
                            .guestNames(
                                    itinerary.getGuests().stream().map(
                                            guest -> String.format("%s %s", guest.getFirstName(), guest.getLastName())
                                    ).collect(Collectors.toList()))
                            .userEmail(order.getOwner().getEmail())
                            .displayApplicationDate(DateTimeFormatter.ISO_LOCAL_DATE.format(applicationDate))
                            .displayCheckinDate(DateTimeFormatter.ISO_LOCAL_DATE.format(orderDetails.getCheckinDate()))
                            .displayCheckoutDate(DateTimeFormatter.ISO_LOCAL_DATE.format(orderDetails.getCheckoutDate()))
                            .hotelName(orderDetails.getHotelName())
                            .hotelAddress(orderDetails.getAddressByYandex())
                            .totalPrice(ProtoUtils.fromTPrice(order.getOrderPriceInfo().getPrice()))
                            .receiptNumbers(receiptNumbers)
                            .build();
                });
    }
}
