package ru.yandex.travel.api.services.orders;

import java.io.IOException;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutDeleteReqV1;
import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutDeleteRspV1;
import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutResultReqV1;
import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutResultRspV1;
import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutStartReqV1;
import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutStartRspV1;
import ru.yandex.travel.api.endpoints.takeout.req_rsp.TakeoutStatusRspV1;
import ru.yandex.travel.api.models.takeout.TakeoutStatus;
import ru.yandex.travel.api.services.dictionaries.avia.AviaAirportDictionary;
import ru.yandex.travel.api.services.dictionaries.avia.AviaSettlementDictionary;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.dicts.avia.TAirport;
import ru.yandex.travel.dicts.avia.TSettlement;
import ru.yandex.travel.orders.takeout.proto.ETakeoutJobState;
import ru.yandex.travel.orders.takeout.proto.ETakoutJobType;
import ru.yandex.travel.orders.takeout.proto.TCheckOrdersExistenceReq;
import ru.yandex.travel.orders.takeout.proto.TFindJobReq;
import ru.yandex.travel.orders.takeout.proto.TStartJobReq;
import ru.yandex.travel.takeout.models.AviaTakeoutOrder;
import ru.yandex.travel.takeout.models.AviaTakeoutSegment;
import ru.yandex.travel.takeout.models.TakeoutOrder;
import ru.yandex.travel.takeout.models.TakeoutResponse;

import static java.util.stream.Collectors.toList;
import static ru.yandex.travel.commons.jackson.CommonJsonUtils.createCommonObjectMapper;

@Service
@RequiredArgsConstructor
@Slf4j
public class TakeoutOrdersService {
    private final OrchestratorTakeoutClientFactory clientFactory;
    private final AviaAirportDictionary aviaAirportDictionary;
    private final AviaSettlementDictionary aviaSettlementDictionary;

    /**
     * Takeout category slug id. Used by passport to indicate we work with orders.
     * For now it is the only category in Yandex.Travel
     */
    private final static String SLUG_ID = "orders";

    private ObjectMapper objectMapper = createCommonObjectMapper();

    public CompletableFuture<TakeoutStartRspV1> start(TakeoutStartReqV1 request) {
        var client = clientFactory.createTakeoutFutureStub();
        return FutureUtils.buildCompletableFuture(client.startTakeoutJob(TStartJobReq.newBuilder()
                .setPassportId(request.getUid()).setJobType(ETakoutJobType.TT_GET).build()))
                .thenApply(orderRsp -> TakeoutStartRspV1.create(ProtoUtils.fromStringOrNull(orderRsp.getJobUid())));
    }

    public CompletableFuture<TakeoutResultRspV1> result(TakeoutResultReqV1 request) {
        var client = clientFactory.createTakeoutFutureStub();
        return FutureUtils.buildCompletableFuture(client.findJob(TFindJobReq.newBuilder()
                .setJobUid(ProtoUtils.toStringOrEmpty(request.getJob_id())).build()))
                .thenApply(job -> {
                    var rsp = TakeoutResultRspV1.builder();
                    if (job.getState() == ETakeoutJobState.TS_NEW) {
                        rsp.status(TakeoutStatus.PENDING);
                        return rsp.build();
                    }

                    TakeoutResponse payload = ProtoUtils.fromTJson(job.getPayload(), TakeoutResponse.class);
                    if (job.getState() == ETakeoutJobState.TS_FAILED) {
                        rsp.status(TakeoutStatus.ERROR);
                        rsp.error(payload.getError());
                        return rsp.build();
                    } else if (job.getState() == ETakeoutJobState.TS_DONE) {
                        if (payload.isEmpty()) {
                            rsp.status(TakeoutStatus.NO_DATA);
                            return rsp.build();
                        }
                        try {
                            List<Object> orders = Stream
                                    .of(payload.getTrainOrders(), payload.getAviaOrders(), payload.getHotelOrders(), payload.getGenericOrders())
                                    .flatMap(list -> list != null ? list.stream() : Stream.empty())
                                    .peek(this::enrichDictionaryData)
                                    .collect(toList());
                            String ordersJson = objectMapper.writeValueAsString(orders);
                            rsp.status(TakeoutStatus.OK);
                            rsp.data(new TakeoutResultRspV1.TakeoutData(ordersJson));
                            return rsp.build();
                        } catch (IOException e) {
                            throw new IllegalArgumentException("Failed to generate JSON", e);
                        }
                    }
                    throw new IllegalStateException(String.format("Unknown job state %s", job.getState()));
                });
    }

    public CompletableFuture<TakeoutStatusRspV1> status(String requestId) {
        log.info("Requesting orders takeout status with id {}", requestId);
        var client = clientFactory.createTakeoutFutureStub();
        String passportId = UserCredentials.get().getPassportId();
        var existFuture = FutureUtils.buildCompletableFuture(
                client.checkOrdersExistence(TCheckOrdersExistenceReq.newBuilder()
                        .setPassportId(passportId)
                        .build())
        );
        var findJobFuture = FutureUtils.buildCompletableFuture(
                client.findJob(TFindJobReq.newBuilder()
                        .setPassportId(passportId)
                        .setJobType(ETakoutJobType.TT_DELETE)
                        .build())
        );
        return findJobFuture.thenCombine(existFuture, (job, existence) -> {
            var result = TakeoutStatusRspV1.builder();
            boolean jobExists = Strings.isNotEmpty(job.getJobUid());
            var data = TakeoutStatusRspV1.TakeoutData.builder();
            data.id(SLUG_ID);
            data.slug(SLUG_ID);
            if (jobExists) {
                data.updateDate(ProtoUtils.toInstant(job.getUpdatedAt()).atZone(ZoneId.of("UTC")).format(
                        DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'")
                ));
            }

            if (jobExists && job.getState() == ETakeoutJobState.TS_NEW) {
                data.state(TakeoutStatusRspV1.TakeoutDataState.IN_PROGRESS);
            } else if (existence.getOrdersExist()) {
                data.state(TakeoutStatusRspV1.TakeoutDataState.READY);
            } else {
                data.state(TakeoutStatusRspV1.TakeoutDataState.EMPTY);
            }
            result.data(List.of(data.build()));
            result.status(TakeoutStatus.OK);

            return result.build();
        });
    }

    public CompletableFuture<TakeoutDeleteRspV1> delete(String requestId, TakeoutDeleteReqV1 request) {
        log.info("Requesting orders takeout deletion with id {}", requestId);
        if (request.getId().isEmpty() || request.getId().stream().noneMatch(s -> s.equals(SLUG_ID))) {
            return CompletableFuture.completedFuture(new TakeoutDeleteRspV1(TakeoutStatus.OK, List.of()));
        }
        var client = clientFactory.createTakeoutFutureStub();
        return FutureUtils.buildCompletableFuture(client.startTakeoutJob(TStartJobReq.newBuilder()
                        .setPassportId(UserCredentials.get().getPassportId())
                        .setJobType(ETakoutJobType.TT_DELETE)
                        .build()))
                .thenApply(orderRsp -> new TakeoutDeleteRspV1(TakeoutStatus.OK, List.of()));
    }

    private void enrichDictionaryData(Object takeoutOrder) {
        if (takeoutOrder instanceof TakeoutOrder) {
            TakeoutOrder order = (TakeoutOrder) takeoutOrder;
            order.getAviaItems().stream()
                    .flatMap(x -> x.getSegments().stream())
                    .flatMap(Collection::stream)
                    .forEach(this::enrichAviaDictionaryData);
        } else if (takeoutOrder instanceof AviaTakeoutOrder) {
            AviaTakeoutOrder aviaOrder = (AviaTakeoutOrder) takeoutOrder;
            if (aviaOrder.getSegments() == null) {
                return;
            }
            for (List<AviaTakeoutSegment> leg : aviaOrder.getSegments()) {
                for (AviaTakeoutSegment segment : leg) {
                    enrichAviaDictionaryData(segment);
                }
            }
        }
    }

    private void enrichAviaDictionaryData(AviaTakeoutSegment segment) {
        // from
        TAirport fromAirport = aviaAirportDictionary.getByIataCode(segment.getDepartureAirport());
        segment.setDepartureAirportName(fromAirport.getTitle());
        TSettlement fromSettlement = aviaSettlementDictionary.getById(fromAirport.getSettlementId());
        segment.setDepartureCityName(fromSettlement.getTitle());
        // to
        TAirport toAirport = aviaAirportDictionary.getByIataCode(segment.getArrivalAirport());
        segment.setArrivalAirportName(toAirport.getTitle());
        TSettlement toSettlement = aviaSettlementDictionary.getById(toAirport.getSettlementId());
        segment.setArrivalCityName(toSettlement.getTitle());
    }
}
