package ru.yandex.travel.api.services.orders;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.models.train.Settlement;
import ru.yandex.travel.api.models.train.Station;
import ru.yandex.travel.api.services.dictionaries.train.settlement.TrainSettlementDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.station.TrainStationDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.station_code.TrainStationCodeDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.station_express_alias.TrainStationExpressAliasDataProvider;
import ru.yandex.travel.api.services.dictionaries.train.time_zone.TrainTimeZoneDataProvider;
import ru.yandex.travel.dicts.rasp.proto.TSettlement;
import ru.yandex.travel.dicts.rasp.proto.TStation;
import ru.yandex.travel.dicts.rasp.proto.TTimeZone;

@Slf4j
@Service
@RequiredArgsConstructor
public class TrainDictionaryMapService {

    private final TrainSettlementDataProvider trainSettlementDataProvider;
    private final TrainStationDataProvider trainStationDataProvider;
    private final TrainStationCodeDataProvider trainStationCodeDataProvider;
    private final TrainTimeZoneDataProvider trainTimeZoneDataProvider;
    private final TrainStationExpressAliasDataProvider trainStationExpressAliasDataProvider;

    public Station createStationInfoByCode(String stationCode) {
        Integer stationId = trainStationCodeDataProvider.getStationIdByCode(stationCode);
        return createStationInfoById(stationId);
    }

    public String getStationTitleByExpressName(String expressName) {
        if (!Strings.isNullOrEmpty(expressName)) {
            Integer id = trainStationExpressAliasDataProvider.getStationIdByExpressName(expressName);
            if (id != null) {
                TStation protoStation = trainStationDataProvider.getById(id);
                return protoStation.getTitleDefault();
            }
        }
        return expressName;
    }

    public Station createStationInfoById(Integer stationId) {
        TStation protoStation = trainStationDataProvider.getById(stationId);
        TTimeZone protoRailwayTimeZone = trainTimeZoneDataProvider.getById(protoStation.getRailwayTimeZoneId());
        TTimeZone protoTimeZone = trainTimeZoneDataProvider.getById(protoStation.getTimeZoneId());
        Station result = Station.builder()
                .id(protoStation.getId())
                .build();
        if (Strings.isNullOrEmpty(protoStation.getPopularTitleDefault())) {
            result.setTitle(protoStation.getTitleDefault());
        } else {
            result.setTitle(protoStation.getPopularTitleDefault());
        }
        result.setPopularTitle(result.getTitle());
        if (protoStation.getSettlementId() != 0) {
            TSettlement protoSettlement = trainSettlementDataProvider.getById(protoStation.getSettlementId());
            result.setSettlementId(protoStation.getSettlementId());
            result.setSettlementTitle(protoSettlement.getTitleDefault());
            result.setSettlementGeoId(protoSettlement.getGeoId());
            // not sure if the title could ever be empty but we have agreed to set the field only when it's not
            if (!Strings.isNullOrEmpty(result.getSettlementTitle())) {
                result.setSettlement(Settlement.builder()
                        .id(result.getSettlementId())
                        .geoId(result.getSettlementGeoId())
                        .title(result.getSettlementTitle())
                        .titleAccusative(protoSettlement.getTitleRuAccusativeCase())
                        .titleGenitive(protoSettlement.getTitleRuGenitiveCase())
                        .titlePrepositional(protoSettlement.getTitleRuPrepositionalCase())
                        .titleLocative(protoSettlement.getTitleRuPrepositionalCase())
                        .preposition(protoStation.getTitleRuPreposition())
                        .build());
            }
        }
        result.setRailwayTimezone(protoRailwayTimeZone.getCode());
        result.setTimezone(protoTimeZone.getCode());
        return result;
    }

}
