package ru.yandex.travel.api.services.orders;

import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.protobuf.Timestamp;
import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.booking_flow.model.PaymentErrorCode;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.train.PassengerDTO;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.train.TrainServiceInfoDTO;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.OrderInfoRspV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.OrderStatusRspV1;
import ru.yandex.travel.api.infrastucture.ApiTokenEncrypter;
import ru.yandex.travel.api.models.train.ApiErrorCode;
import ru.yandex.travel.api.models.train.DisplayReservationPlaceType;
import ru.yandex.travel.api.models.train.ErrorInfo;
import ru.yandex.travel.api.models.train.ErrorType;
import ru.yandex.travel.api.models.train.Insurance;
import ru.yandex.travel.api.models.train.InsuranceStatus;
import ru.yandex.travel.api.models.train.Passenger;
import ru.yandex.travel.api.models.train.PlaceWithType;
import ru.yandex.travel.api.models.train.Refund;
import ru.yandex.travel.api.models.train.RefundStatus;
import ru.yandex.travel.api.models.train.RefundTicket;
import ru.yandex.travel.api.models.train.SegmentInfo;
import ru.yandex.travel.api.models.train.TariffInfo;
import ru.yandex.travel.api.models.train.Ticket;
import ru.yandex.travel.api.models.train.TicketPayment;
import ru.yandex.travel.api.models.train.TrainInfo;
import ru.yandex.travel.api.models.train.TrainOrderMaps;
import ru.yandex.travel.api.models.train.TrainOrderStatus;
import ru.yandex.travel.api.models.train.TrainRefundType;
import ru.yandex.travel.api.models.train.WarningInfo;
import ru.yandex.travel.api.models.train.WarningMessageCode;
import ru.yandex.travel.api.services.orders.model.TrainOrderListInfo;
import ru.yandex.travel.api.services.orders.model.TrainOrderListInfoPassenger;
import ru.yandex.travel.api.services.orders.model.TrainOrderListSegmentInfo;
import ru.yandex.travel.api.services.orders.model.TrainOrderListTrainInfo;
import ru.yandex.travel.commons.proto.EFiscalReceiptType;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TFiscalReceipt;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.proto.EGenericOrderAggregateState;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.proto.EOrderRefundType;
import ru.yandex.travel.orders.proto.ETrainOrderAggregateState;
import ru.yandex.travel.orders.proto.TDownloadBlankToken;
import ru.yandex.travel.orders.proto.TOrderAggregateState;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TOrderRefund;
import ru.yandex.travel.orders.proto.TOrderServiceInfo;
import ru.yandex.travel.orders.proto.TServiceInfo;
import ru.yandex.travel.orders.proto.TTrainOrderAggregateErrorInfo;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.train.proto.TTrainDownloadBlankParams;
import ru.yandex.travel.train.model.Direction;
import ru.yandex.travel.train.model.PassengerCategory;
import ru.yandex.travel.train.model.RoutePolicy;
import ru.yandex.travel.train.model.TrainModelHelpers;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicket;
import ru.yandex.travel.train.model.refund.TicketRefund;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.BoardingSystemType;

import static java.util.stream.Collectors.toList;

@Service
@RequiredArgsConstructor
public class TrainModelMapService {
    private static final Set<EOrderRefundType> TICKET_REFUND_TYPES =
            Set.of(EOrderRefundType.RT_TRAIN_OFFICE_REFUND,
                    EOrderRefundType.RT_TRAIN_USER_REFUND,
                    EOrderRefundType.RT_GENERIC_USER_REFUND);
    private static final ZoneId ZONE_UTC = ZoneId.of("UTC");
    private static final Duration SIX_HOURS_PERIOD = Duration.ofHours(6);
    private static final Duration FIFTEEN_MINUTES_PERIOD = Duration.ofMinutes(15);
    private static final Set<EOrderRefundState> ORDER_REFUND_REFUNDED_STATES =
            Set.of(EOrderRefundState.RS_WAITING_INVOICE_REFUND, EOrderRefundState.RS_REFUNDED);

    private final TrainDictionaryMapService trainDictionaryMapService;
    private final TrainOrderStatusMappingService statusMappingService;
    private final TrainOrdersServiceProperties properties;
    private final ApiTokenEncrypter apiTokenEncrypter;

    public static ru.yandex.travel.api.models.train.Direction mapTrainDirection(Direction payloadDirection) {
        Direction direction = payloadDirection != null ? payloadDirection : Direction.FORWARD;
        return TrainOrderMaps.DIRECTION_DIRECTION_MAP.get(direction);
    }

    public OrderInfoRspV1 convert(TOrderInfo orderInfo, boolean withNewSegments) {
        OrderInfoRspV1 result = new OrderInfoRspV1();

        if (properties.isAggregateStatusEnabled() && orderInfo.hasOrderAggregateState()) {
            fillStatusResponseFromAggregate(result, orderInfo.getOrderAggregateState());
        } else {
            fillStatusResponse(result, orderInfo);
        }

        if (orderInfo.hasCurrentInvoice()) {
            String receiptUrl = orderInfo.getCurrentInvoice().getFiscalReceiptList().stream()
                    .filter(fr -> fr.getType() == EFiscalReceiptType.FRT_ACQUIRE)
                    .map(fr -> convertReceiptUrlToPdf(fr.getUrl())).findFirst().orElse(null);
            result.setPaymentReceiptUrl(receiptUrl);
        }

        Map<String, String> refundReceiptUrls = orderInfo.getCurrentInvoice().getFiscalReceiptList().stream()
                .filter(x -> !(Strings.isNullOrEmpty(x.getOrderRefundId()) || Strings.isNullOrEmpty(x.getUrl())))
                .collect(Collectors.toMap(TFiscalReceipt::getOrderRefundId, TFiscalReceipt::getUrl));
        result.setRefunds(new ArrayList<>());
        for (TOrderRefund orderRefund : orderInfo.getRefundList()) {
            var refund = new Refund();
            result.getRefunds().add(refund);
            refund.setTickets(new ArrayList<>());
            refund.setCreatedAt(ProtoUtils.toInstant(orderRefund.getCreatedAt()));
            refund.setStatus(RefundStatus.fromState(orderRefund.getState()));
            refund.setType(TrainRefundType.fromProto(orderRefund.getRefundType()));
            if (!Strings.isNullOrEmpty(orderRefund.getPayload().getValue())) {
                TicketRefund ticketRefund = ProtoUtils.fromTJson(orderRefund.getPayload(), TicketRefund.class);
                refund.setTickets(ticketRefund.getItems().stream().map(x -> {
                    var refundTicket = new RefundTicket();
                    refundTicket.setBlankId(x.getBlankId());
                    refundTicket.setAmount(x.calculateActualRefundSum());
                    return refundTicket;
                }).collect(toList()));
            }
            String refundReceiptUrl = refundReceiptUrls.get(orderRefund.getId());
            if (!Strings.isNullOrEmpty(refundReceiptUrl)) {
                //TODO mbobrov YATRAVEL-548
                refund.setPaymentReceiptUrl(convertReceiptUrlToPdf(refundReceiptUrl));
            }
        }

        result.setCustomerEmail(orderInfo.getOwner().getEmail());
        result.setCustomerPhone(orderInfo.getOwner().getPhone());

        List<WarningInfo> warnings = new ArrayList<>();
        List<SegmentInfo> segments = new ArrayList<>();
        Preconditions.checkArgument(orderInfo.getServiceCount() >= 1,
                "At least of train service is expected! Order id %s", orderInfo.getOrderId());
        for (TOrderServiceInfo orderServiceInfo : orderInfo.getServiceList()) {
            Preconditions.checkArgument(orderServiceInfo.getServiceType() == EServiceType.PT_TRAIN,
                    "Only train services are expected in a train order but got %s for service %s",
                    orderServiceInfo.getServiceType(), orderServiceInfo.getServiceId());
            TServiceInfo serviceInfo = orderServiceInfo.getServiceInfo();
            TrainReservation payload = ProtoUtils.fromTJson(serviceInfo.getPayload(), TrainReservation.class);
            SegmentInfo segment = convertSegment(payload);
            segment.setWarnings(extractFirstSegmentWarning(
                    orderServiceInfo.getConfirmedAt(), serviceInfo.getGenericOrderItemState(), payload));
            segments.add(segment);
            warnings.addAll(segment.getWarnings());

        }
        if (withNewSegments) {
            result.setSegments(segments);
        }
        result.setWarnings(extractFirstWarning(warnings));

        if (segments.size() == 1) {
            initLegacySingleSegmentFields(result, segments.get(0));
        }

        return result;
    }

    public SegmentInfo convertSegment(TrainReservation payload) {
        SegmentInfo result = new SegmentInfo();

        result.setPassengers(new ArrayList<>());
        result.setDirection(TrainModelMapService.mapTrainDirection(payload.getDirection()));
        result.setPartnerOrderId(payload.getPartnerOrderId());
        result.setReservationNumber(payload.getReservationNumber());

        Instant departure = payload.getDepartureTime();
        if (departure == null) {
            departure = payload.getReservationRequestData().getDepartureTime();
        }
        for (TrainPassenger sourcePassenger : payload.getPassengers()) {
            result.getPassengers().add(getPassenger(sourcePassenger, payload, departure));
        }

        result.setDeparture(departure);
        result.setArrival(payload.getArrivalTime());
        result.setCoachType(TrainOrderMaps.CAR_TYPE_TO_STR.get(payload.getCarType()));
        result.setCoachNumber(payload.getCarNumber());
        result.setCoachOwner(payload.getCarrier());
        result.setCompanyTitle(payload.getCompanyTitle());
        if (payload.getReservationRequestData().getCabinGenderKind() != null) {
            result.setCompartmentGender(TrainOrderMaps.CABIN_GENDER_KIND_TO_STR.get(
                    payload.getReservationRequestData().getCabinGenderKind()));
        }
        // TODO (mbobrov,ganintsev): remove this hack in favor of some more robust solution
        result.setStationFrom(trainDictionaryMapService.createStationInfoByCode(payload.getReservationRequestData().getStationFromCode()));
        result.setStationTo(trainDictionaryMapService.createStationInfoByCode(payload.getReservationRequestData().getStationToCode()));
        result.setPartner("im");
        result.setSpecialNotice(payload.getSpecialNotice());
        result.setTrainInfo(getTrainInfo(payload));
        return result;
    }

    private List<WarningInfo> extractFirstSegmentWarning(Timestamp serviceConfirmedAt, EOrderItemState state,
                                                         TrainReservation payload) {
        Instant confirmedAt = ProtoUtils.toInstantFromSafe(serviceConfirmedAt);
        // skip warnings for in-progress (changing) or refunded services
        if (state != EOrderItemState.IS_NEW &&
                state != EOrderItemState.IS_RESERVING &&
                state != EOrderItemState.IS_REFUNDED) {
            return getOrderWarnings(payload, confirmedAt, true);
        }
        return List.of();
    }

    private void initLegacySingleSegmentFields(OrderInfoRspV1 order, SegmentInfo segment) {
        order.setPartnerOrderId(segment.getPartnerOrderId());
        order.setReservationNumber(segment.getReservationNumber());
        order.setStationFrom(segment.getStationFrom());
        order.setStationTo(segment.getStationTo());
        order.setTrainInfo(segment.getTrainInfo());
        order.setCarType(segment.getCoachType());
        order.setCompartmentGender(segment.getCompartmentGender());
        order.setArrival(segment.getArrival());
        order.setDeparture(segment.getDeparture());
        order.setPartner(segment.getPartner());
        order.setCarNumber(segment.getCoachNumber());
        order.setSpecialNotice(segment.getSpecialNotice());
        order.setTwoStorey(segment.isTwoStorey());
        order.setCoachOwner(segment.getCoachOwner());
        order.setCompanyTitle(segment.getCompanyTitle());
        order.setPassengers(segment.getPassengers());
    }

    public TrainOrderListInfo convertToListInfo(TOrderInfo orderInfo) {
        TrainOrderListInfo result = new TrainOrderListInfo();
        result.setSegments(orderInfo.getServiceList().stream()
                .map(serviceInfo -> convertToListInfo(serviceInfo.getServiceInfo()))
                .collect(toList()));
        fillAggregateOrderInfo(orderInfo, result);
        fillLegacySingleSegmentInfo(result);
        return result;
    }

    private TrainOrderListSegmentInfo convertToListInfo(TServiceInfo orderItem) {
        var payload = ProtoUtils.fromTJson(orderItem.getPayload(), TrainReservation.class);
        TrainOrderListSegmentInfo result = new TrainOrderListSegmentInfo();
        result.setDirection(mapTrainDirection(payload.getDirection()));

        result.setReservationNumber(payload.getReservationNumber());
        result.setPartnerOrderId(payload.getPartnerOrderId());
        result.setPassengers(new ArrayList<>());
        Instant departure = payload.getDepartureTime();
        if (departure == null) {
            departure = payload.getReservationRequestData().getDepartureTime();
        }
        Money totalAmount = Money.zero(ProtoCurrencyUnit.RUB);
        for (TrainPassenger sourcePassenger : payload.getPassengers()) {
            var passenger = new TrainOrderListInfoPassenger();
            passenger.setFirstName(sourcePassenger.getFirstName());
            passenger.setLastName(sourcePassenger.getLastName());

            if (sourcePassenger.getTicket() != null) {
                totalAmount = totalAmount.add(
                        TrainModelHelpers.calculateTotalCostForPassenger(payload, sourcePassenger)
                );

                if (sourcePassenger.getTicket().getImBlankStatus() != null) {
                    passenger.setTicketRzhdStatus(sourcePassenger.getTicket().getImBlankStatus().name());
                }
                passenger.setCategory(sourcePassenger.getCategory());
            }

            result.getPassengers().add(passenger);
        }

        result.setDeparture(departure);
        result.setArrival(payload.getArrivalTime());
        result.setCarNumber(payload.getCarNumber());
        result.setCarType(payload.getCarType());
        // TODO (mbobrov,ganintsev): remove this hack in favor of some more robust solution
        result.setStationFrom(trainDictionaryMapService.createStationInfoByCode(payload.getReservationRequestData().getStationFromCode()));
        result.setStationTo(trainDictionaryMapService.createStationInfoByCode(payload.getReservationRequestData().getStationToCode()));
        result.setTrainInfo(new TrainOrderListTrainInfo());
        result.getTrainInfo().setSuburban(payload.isSuburban());
        result.getTrainInfo().setBrandTitle(payload.getReservationRequestData().getBrandTitle());
        result.getTrainInfo().setTrainTitle(payload.getReservationRequestData().getTrainTitle());
        result.getTrainInfo().setTrainNumber(payload.getReservationRequestData().getTrainNumber());
        result.getTrainInfo().setTrainTicketNumber(payload.getReservationRequestData().getTrainTicketNumber());
        result.getTrainInfo().setStartSettlementTitle(payload.getReservationRequestData().getStartSettlementTitle());
        result.getTrainInfo().setEndSettlementTitle(payload.getReservationRequestData().getEndSettlementTitle());

        result.setCanChangeElectronicRegistrationTill(TrainModelHelpers.calculateCanChangeElectronicRegistrationTill(payload));
        result.setTotalAmount(totalAmount);
        return result;
    }

    private void fillAggregateOrderInfo(TOrderInfo orderInfo, TrainOrderListInfo result) {
        Preconditions.checkNotNull(result.getSegments(), "No segments computed");
        Money totalAmount = result.getSegments().stream()
                .map(TrainOrderListSegmentInfo::getTotalAmount)
                .reduce(Money::add)
                .orElseThrow(() -> new IllegalArgumentException(
                        "No train order segments provided: " + orderInfo.getOrderId()));
        Money refundAmount = Money.zero(ProtoCurrencyUnit.RUB);
        int refundedTicketsCount = 0;
        for (TOrderRefund orderRefund : orderInfo.getRefundList()) {
            if (ORDER_REFUND_REFUNDED_STATES.contains(orderRefund.getState())
                    && TICKET_REFUND_TYPES.contains(orderRefund.getRefundType())) {
                Preconditions.checkState(orderRefund.hasPayload(), "Order refund must have payload");
                TicketRefund refundPayload = ProtoUtils.fromTJson(orderRefund.getPayload(), TicketRefund.class);
                Money partialRefundAmount = refundPayload.calculateActualRefundSum();

                refundAmount = refundAmount.add(partialRefundAmount);
                refundedTicketsCount += refundPayload.getItems().size();
            }
        }

        result.setTotalAmount(totalAmount);
        result.setRefundAmount(refundAmount);
        result.setRefundedTicketsCount(refundedTicketsCount);

        result.setCustomerEmail(orderInfo.getOwner().getEmail());
        result.setCustomerPhone(orderInfo.getOwner().getPhone());
    }

    // todo(tlg-13,ganintsev): legacy structure should be remove in TRAVELBACK-1828
    private void fillLegacySingleSegmentInfo(TrainOrderListInfo orderInfo) {
        List<TrainOrderListSegmentInfo> segments = orderInfo.getSegments();
        Preconditions.checkArgument(segments != null && segments.size() > 0, "No segments provided");
        if (segments.size() == 1) {
            // support for simple train orders only, the new ones should be handled by the new UI code
            TrainOrderListSegmentInfo segment = segments.get(0);
            orderInfo.setReservationNumber(segment.getReservationNumber());
            orderInfo.setPartnerOrderId(segment.getPartnerOrderId());
            orderInfo.setArrival(segment.getArrival());
            orderInfo.setDeparture(segment.getDeparture());
            orderInfo.setStationFrom(segment.getStationFrom());
            orderInfo.setStationTo(segment.getStationTo());
            orderInfo.setTrainInfo(segment.getTrainInfo());
            orderInfo.setCarType(segment.getCarType());
            orderInfo.setCarNumber(segment.getCarNumber());
            orderInfo.setCanChangeElectronicRegistrationTill(segment.getCanChangeElectronicRegistrationTill());
            orderInfo.setPassengers(segment.getPassengers());
        }
    }

    void fillStatusResponseFromAggregate(OrderStatusRspV1 result, TOrderAggregateState aggregateState) {

        result.setId(UUID.fromString(aggregateState.getOrderId()));
        result.setPrettyId(aggregateState.getOrderPrettyId());
        if (aggregateState.hasReservedTo()) {
            result.setReservedTo(ProtoUtils.toInstant(aggregateState.getReservedTo()));
        }

        result.setPaymentUrl(Strings.emptyToNull(aggregateState.getPaymentUrl()));
        result.setPaymentError(PaymentErrorCode.BY_PROTO.getByValueOrNull(aggregateState.getPaymentError()));

        if (aggregateState.getTrainOrderAggregateState() != ETrainOrderAggregateState.TOAG_UNKNOWN) {
            result.setStatus(TrainOrderStatus.BY_PROTO.getByValue(aggregateState.getTrainOrderAggregateState()));
        } else if (aggregateState.getGenericOrderAggregateState() != EGenericOrderAggregateState.GOAG_UNKNOWN) {
            result.setStatus(TrainOrderStatus.BY_PROTO_GENERIC.getByValue(
                    aggregateState.getGenericOrderAggregateState()));
        } else {
            throw new IllegalArgumentException("No supported state field: " + aggregateState);
        }

        if (aggregateState.hasTrainAggregateExtInfo() && aggregateState.getTrainAggregateExtInfo().hasErrorInfo()) {
            TTrainOrderAggregateErrorInfo protoError = aggregateState.getTrainAggregateExtInfo().getErrorInfo();
            result.setError(new ErrorInfo());
            result.getError().setType(ErrorType.BY_PROTO.getByValueOrNull(protoError.getErrorType()));
            result.getError().setCode(ApiErrorCode.BY_PROTO.getByValueOrNull(protoError.getMessageCode()));
            result.getError().setMessage(Strings.emptyToNull(protoError.getMessage()));
            result.getError().setMessageParams(protoError.getMessageParamList());
        }
        result.setInsuranceStatus(
                InsuranceStatus.BY_PROTO.getByValueOrNull(aggregateState.getTrainAggregateExtInfo().getInsuranceState())
        );
    }

    void fillStatusResponse(OrderStatusRspV1 result, TOrderInfo orderInfo) {
        result.setReservedTo(ProtoUtils.toInstant(orderInfo.getExpiresAt()));
        result.setId(UUID.fromString(orderInfo.getOrderId()));
        result.setPrettyId(orderInfo.getPrettyId());

        if (orderInfo.hasCurrentInvoice()) {
            if (!Strings.isNullOrEmpty(orderInfo.getCurrentInvoice().getPaymentUrl())) {
                result.setPaymentUrl(orderInfo.getCurrentInvoice().getPaymentUrl());
            }
            if (!Strings.isNullOrEmpty(orderInfo.getCurrentInvoice().getAuthorizationErrorCode())) {
                result.setPaymentError(PaymentErrorCode.fromErrorCode(orderInfo.getCurrentInvoice().getAuthorizationErrorCode()));
            }
        }
        result.setStatus(statusMappingService.getStatus(orderInfo));

        List<TrainReservation> segmentPayloads = orderInfo.getServiceList().stream()
                .map(service -> ProtoUtils.fromTJson(service.getServiceInfo().getPayload(), TrainReservation.class))
                .collect(toList());
        result.setMaxPendingTill(segmentPayloads.stream()
                .map(TrainReservation::getMaxPendingTill)
                .filter(Objects::nonNull)
                .min(Comparator.naturalOrder())
                .orElse(null));
        // backward compatibility, for complex orders the client should reload the order and check its data instead
        if (orderInfo.getServiceCount() == 1) {
            var payloadJson = orderInfo.getService(0).getServiceInfo().getPayload();
            var payload = ProtoUtils.fromTJson(payloadJson, TrainReservation.class);
            result.setError(statusMappingService.getErrorInfo(orderInfo, payload.getErrorInfo()));
            result.setInsuranceStatus(InsuranceStatus.fromModelValue(payload.getInsuranceStatus()));
        }
    }

    public List<WarningInfo> getOrderWarnings(TrainReservation payload, Instant confirmedAt,
                                              boolean getFirstActualWarning) {
        List<WarningInfo> result = new ArrayList<>();
        var now = Instant.now();
        Instant departure = payload.getDepartureTime();
        if (departure == null) {
            departure = payload.getReservationRequestData().getDepartureTime();
        }
        Instant maxWarningTime = departure.plus(properties.getAfterDepartureMaxWarningTime());

        if (payload.getPassengers().stream().allMatch(p -> p.getTicket() != null
                && p.getTicket().getImBlankStatus() == ImBlankStatus.STRICT_BOARDING_PASS)) {
            result.add(new WarningInfo(WarningMessageCode.TICKETS_TAKEN_AWAY, null, maxWarningTime));
        }

        if (confirmedAt != null
                && payload.getInsuranceStatus() == ru.yandex.travel.train.model.InsuranceStatus.AUTO_RETURN) {
            var insuranceWarningActualTill = confirmedAt.plus(properties.getInsuranceAutoReturnWarningTime());
            if (now.isBefore(insuranceWarningActualTill)) {
                result.add(new WarningInfo(WarningMessageCode.INSURANCE_AUTO_RETURN,
                        confirmedAt,
                        insuranceWarningActualTill));
            }
        }

        var hasTicketsWithEr = TrainModelHelpers.hasTicketsWithBlankStatus(payload, ImBlankStatus.REMOTE_CHECK_IN);
        var hasTicketsWithoutEr = TrainModelHelpers.hasTicketsWithBlankStatus(payload,
                ImBlankStatus.NO_REMOTE_CHECK_IN);

        var returnTill = TrainModelHelpers.calculateCanRefundOnServiceTill(payload);
        if (payload.isProviderP2()) {
            result.addAll(refundWarningsMovista(now, returnTill, maxWarningTime));
        } else if (payload.getBoardingSystemType() == BoardingSystemType.PASSENGER_BOARDING_CONTROL) {
            result.addAll(refundWarningsWithPassengerBoardingControl(now, payload, returnTill, maxWarningTime));
        } else if (hasTicketsWithoutEr || returnTill == null) {
            result.addAll(refundWarningsWithoutEr(now, payload, maxWarningTime));
        } else if (hasTicketsWithEr) {
            result.addAll(refundWarningsWithEr(now, payload, returnTill, maxWarningTime));
        }

        if (payload.getPassengers().stream()
                .anyMatch(p -> p.isNonRefundableTariff() && p.getTicket() != null && p.getTicket().getRefundStatus() == null)) {
            result.add(new WarningInfo(WarningMessageCode.HAS_NON_REFUNDABLE_TARIFF, null, maxWarningTime));
        }

        if (getFirstActualWarning) {
            return extractFirstWarning(result);
        }
        return result;
    }

    private List<WarningInfo> extractFirstWarning(List<WarningInfo> warnings) {
        Instant now = Instant.now();
        return warnings.stream()
                .filter(w -> (w.getFrom() == null || w.getFrom().isBefore(now))
                        && (w.getTo() == null || w.getTo().isAfter(now)))
                .sorted(Comparator.<WarningInfo, Integer>comparing(w -> w.getCode().getSeverity()).reversed())
                .limit(1)
                .collect(toList());
    }

    private Collection<WarningInfo> refundWarningsWithoutEr(Instant now, TrainReservation payload,
                                                            Instant maxWarningTime) {
        var result = new ArrayList<WarningInfo>();

        if (payload.getDepartureTime() == null) {
            return result;
        }

        var isInternationalRoute = payload.getRoutePolicy() == RoutePolicy.INTERNATIONAL;
        var sixHoursToDeparture = payload.getDepartureTime().minus(SIX_HOURS_PERIOD);

        if (isInternationalRoute) {
            if (now.isBefore(sixHoursToDeparture)) {
                result.add(new WarningInfo(WarningMessageCode.ALMOST_SIX_HOURS_TO_DEPARTURE,
                        sixHoursToDeparture.minus(properties.getElectronicRegistrationChangeWarningTime()),
                        sixHoursToDeparture));
            }

            result.add(new WarningInfo(WarningMessageCode.LESS_THEN_SIX_HOURS_TO_DEPARTURE,
                    sixHoursToDeparture, maxWarningTime));


        } else {
            if (now.isBefore(payload.getDepartureTime())) {
                result.add(new WarningInfo(WarningMessageCode.TRAIN_ALMOST_LEFT_DEPARTURE_STATION,
                        payload.getDepartureTime().minus(properties.getElectronicRegistrationChangeWarningTime()),
                        payload.getDepartureTime()));
            }

            result.add(new WarningInfo(WarningMessageCode.TRAIN_LEFT_DEPARTURE_STATION, payload.getDepartureTime(),
                    maxWarningTime));
        }

        return result;
    }

    private Collection<WarningInfo> refundWarningsWithPassengerBoardingControl(
            Instant now, TrainReservation payload, Instant returnTill, Instant maxWarningTime) {
        var result = new ArrayList<WarningInfo>();
        if (returnTill == null) {
            returnTill = payload.getDepartureTime();
        }
        if (returnTill == null) {
            return result;
        }

        if (now.isBefore(returnTill)) {
            result.add(new WarningInfo(WarningMessageCode.TRAIN_ALMOST_LEFT_DEPARTURE_STATION,
                    returnTill.minus(properties.getElectronicRegistrationChangeWarningTime()),
                    returnTill));
        }
        result.add(new WarningInfo(WarningMessageCode.TRAIN_LEFT_DEPARTURE_STATION, returnTill, maxWarningTime));
        return result;
    }

    private Collection<WarningInfo> refundWarningsMovista(Instant now, Instant returnTill, Instant maxWarningTime) {
        var result = new ArrayList<WarningInfo>();
        if (returnTill == null) {
            return result;
        }

        if (now.isBefore(returnTill)) {
            result.add(new WarningInfo(WarningMessageCode.FIFTEEN_MINUTES_TO_REFUND,
                    returnTill.minus(FIFTEEN_MINUTES_PERIOD),
                    returnTill));
        }
        result.add(new WarningInfo(WarningMessageCode.REFUND_DISABLED_BECAUSE_LESS_THAN_ONE_HOUR_TO_DEPARTURE, returnTill, maxWarningTime));
        return result;
    }

    private Collection<WarningInfo> refundWarningsWithEr(Instant now, TrainReservation payload, Instant returnTill,
                                                         Instant maxWarningTime) {
        var result = new ArrayList<WarningInfo>();
        var isInternationalRoute = payload.getRoutePolicy() == RoutePolicy.INTERNATIONAL;
        var leftStartStationAt = returnTill.plus(Duration.ofHours(1));
        var sixHoursToDeparture = payload.getDepartureTime().minus(SIX_HOURS_PERIOD);

        if (isInternationalRoute && sixHoursToDeparture.isBefore(returnTill)) {
            if (now.isBefore(sixHoursToDeparture)) {
                result.add(new WarningInfo(WarningMessageCode.ALMOST_SIX_HOURS_TO_DEPARTURE,
                        sixHoursToDeparture.minus(properties.getElectronicRegistrationChangeWarningTime()),
                        sixHoursToDeparture));
            }

            result.add(new WarningInfo(WarningMessageCode.LESS_THEN_SIX_HOURS_TO_DEPARTURE,
                    sixHoursToDeparture, maxWarningTime));
        } else {
            if (now.isBefore(returnTill)) {
                result.add(new WarningInfo(WarningMessageCode.ELECTRONIC_REGISTRATION_ALMOST_EXPIRED,
                        returnTill.minus(properties.getElectronicRegistrationChangeWarningTime()),
                        returnTill));
            }

            if (now.isBefore(leftStartStationAt)) {
                result.add(new WarningInfo(WarningMessageCode.ELECTRONIC_REGISTRATION_EXPIRED, returnTill,
                        leftStartStationAt));
            }

            result.add(new WarningInfo(WarningMessageCode.TRAIN_LEFT_START_STATION, leftStartStationAt,
                    maxWarningTime));
        }

        return result;
    }

    private String convertReceiptUrlToPdf(String receiptUrl) {
        return receiptUrl.replace("mode=mobile", "mode=pdf");
    }

    public TrainServiceInfoDTO convertTrainServiceInfo(TOrderServiceInfo orderServiceInfo, UUID orderId, boolean insurancePricingFailed) {
        var result = new TrainServiceInfoDTO();
        TServiceInfo source = orderServiceInfo.getServiceInfo();
        var payload = ProtoUtils.fromTJson(source.getPayload(), TrainReservation.class);
        result.setError(statusMappingService.getErrorInfo(null, payload.getErrorInfo()));
        if (insurancePricingFailed) {
            result.setInsuranceStatus(InsuranceStatus.PRICING_FAILED);
        } else {
            result.setInsuranceStatus(InsuranceStatus.fromModelValue(payload.getInsuranceStatus()));
        }
        result.setDirection(mapTrainDirection(payload.getDirection()));
        result.setSegmentIndex(payload.getSegmentIndex());
        result.setPartnerOrderId(payload.getPartnerOrderId());
        result.setReservationNumber(payload.getReservationNumber());
        Instant departure = payload.getDepartureTime();
        if (departure == null) {
            departure = payload.getReservationRequestData().getDepartureTime();
        }
        result.setDeparture(departure);
        result.setPassengers(new ArrayList<>());
        for (TrainPassenger sourcePassenger : payload.getPassengers()) {
            result.getPassengers().add(getPassengerDto(sourcePassenger, payload, departure, insurancePricingFailed));
        }
        result.setArrival(payload.getArrivalTime());
        result.setCarNumber(payload.getCarNumber());
        result.setCarType(TrainOrderMaps.CAR_TYPE_TO_STR.get(payload.getCarType()));
        result.setCoachOwner(payload.getCarrier());
        result.setCompanyTitle(payload.getCompanyTitle());
        if (payload.getReservationRequestData().getCabinGenderKind() != null) {
            result.setCompartmentGender(TrainOrderMaps.CABIN_GENDER_KIND_TO_STR.get(
                    payload.getReservationRequestData().getCabinGenderKind()));
        }
        result.setStationFrom(trainDictionaryMapService.createStationInfoByCode(payload.getReservationRequestData().getStationFromCode()));
        result.setStationTo(trainDictionaryMapService.createStationInfoByCode(payload.getReservationRequestData().getStationToCode()));
        result.setPartner("im");
        result.setSpecialNotice(payload.getSpecialNotice());
        result.setTrainInfo(getTrainInfo(payload));
        Instant confirmedAt = ProtoUtils.toInstantFromSafe(orderServiceInfo.getConfirmedAt());
        if (source.getGenericOrderItemState() != EOrderItemState.IS_NEW &&
                source.getGenericOrderItemState() != EOrderItemState.IS_RESERVING &&
                source.getGenericOrderItemState() != EOrderItemState.IS_REFUNDED) {
            result.setWarnings(getOrderWarnings(payload, confirmedAt, true));
        }
        if (source.getGenericOrderItemState() == EOrderItemState.IS_CONFIRMED ||
                source.getGenericOrderItemState() == EOrderItemState.IS_REFUNDING ||
                source.getGenericOrderItemState() == EOrderItemState.IS_REFUNDED) {
            TDownloadBlankToken token = TDownloadBlankToken.newBuilder()
                    .setTrainDownloadBlankParams(TTrainDownloadBlankParams.newBuilder()
                            .setServiceId(orderServiceInfo.getServiceId())
                            .build())
                    .setOrderId(orderId.toString()).build();
            result.setDownloadBlankToken(apiTokenEncrypter.toDownloadBlankToken(token));
        }
        return result;
    }

    private PassengerDTO getPassengerDto(TrainPassenger sourcePassenger, TrainReservation payload, Instant departure,
                                         boolean insurancePricingFailed) {
        var passenger = new PassengerDTO();
        long years = sourcePassenger.getBirthday().until(departure.atZone(ZONE_UTC), ChronoUnit.YEARS);
        passenger.setAge((int) years);
        passenger.setBirthDate(sourcePassenger.getBirthday());
        passenger.setCitizenship(sourcePassenger.getCitizenshipCode());
        passenger.setCustomerId(sourcePassenger.getCustomerId());
        passenger.setDocId(sourcePassenger.getDocumentNumber());
        passenger.setDocType(sourcePassenger.getDocumentType());
        passenger.setFirstName(sourcePassenger.getFirstName());
        passenger.setLastName(sourcePassenger.getLastName());
        passenger.setPatronymic(sourcePassenger.getPatronymic());
        passenger.setSex(sourcePassenger.getSex());
        passenger.setNonRefundableTariff(sourcePassenger.isNonRefundableTariff());
        TrainTicket sourceTicket = sourcePassenger.getTicket();
        if (sourceTicket != null) {
            Ticket ticket = getTicket(sourcePassenger, payload, sourceTicket);
            passenger.setTicket(ticket);
            passenger.setTotal(ticket.getAmount());
        }
        var sourceInsurance = sourcePassenger.getInsurance();
        if (sourceInsurance != null && !insurancePricingFailed) {
            var insurance = new Insurance();
            passenger.setInsurance(insurance);
            insurance.setAmount(sourceInsurance.getAmount());
            insurance.setCompensation(sourceInsurance.getCompensation());
            passenger.setTotal(passenger.getTotal().add(sourceInsurance.getAmount()));
        }
        return passenger;
    }

    private Ticket getTicket(TrainPassenger sourcePassenger, TrainReservation payload, TrainTicket sourceTicket) {
        var ticket = new Ticket();
        ticket.setAmount(sourceTicket.calculateTotalCost());
        ticket.setBlankId(sourceTicket.getBlankId());
        ticket.setPayment(new TicketPayment());
        ticket.getPayment().setAmount(sourceTicket.getTariffAmount().add(sourceTicket.getServiceAmount()));
        ticket.getPayment().setFee(sourceTicket.getFeeAmount());
        ticket.getPayment().setBeddingAmount(sourceTicket.getServiceAmount());
        ticket.setPlaces(sourceTicket.getPlaces().stream().map(trainPlace -> {
            var p = new PlaceWithType();
            p.setNumber(trainPlace.getNumber());
            p.setType(DisplayReservationPlaceType.fromReservationPlaceType(trainPlace.getType()));
            return p;
        }).collect(toList()));

        ticket.setBookedTariffCode(sourceTicket.getBookedTariffCode());
        ticket.setRawTariffTitle(sourceTicket.getRawTariffName());
        if (sourcePassenger.getCategory() == PassengerCategory.BABY) {
            ticket.setBookedTariffCode("baby");
            // TODO (ganintsev): remove after merging into bookedTariffCode
            var tariffInfo = new TariffInfo();
            tariffInfo.setCode("baby");
            tariffInfo.setTitle(sourceTicket.getRawTariffName());
            ticket.setTariffInfo(tariffInfo);
        }
        ticket.setDiscountDenied(sourcePassenger.isDiscountDenied());

        if (sourceTicket.getImBlankStatus() != null) {
            ticket.setRzhdStatus(sourceTicket.getImBlankStatus().name());
        }
        ticket.setPending(sourceTicket.isPendingElectronicRegistration());
        ticket.setCanChangeElectronicRegistrationTill(sourceTicket.calculateCanChangeElectronicRegistrationTill(
                payload.getDepartureTime(), payload.getReservationRequestData().isElectronicRegistrationEnabled()));
        ticket.setCanReturnTill(sourceTicket.calculateCanReturnTill(payload));
        return ticket;
    }

    private Passenger getPassenger(TrainPassenger sourcePassenger, TrainReservation payload, Instant departure) {
        var passenger = new Passenger();
        long years = sourcePassenger.getBirthday().until(departure.atZone(ZONE_UTC), ChronoUnit.YEARS);
        passenger.setAge((int) years);
        passenger.setBirthDate(sourcePassenger.getBirthday());
        passenger.setCitizenship(sourcePassenger.getCitizenshipCode());
        passenger.setCustomerId(sourcePassenger.getCustomerId());
        passenger.setDocId(sourcePassenger.getDocumentNumber());
        passenger.setDocType(sourcePassenger.getDocumentType());
        passenger.setFirstName(sourcePassenger.getFirstName());
        passenger.setLastName(sourcePassenger.getLastName());
        passenger.setPatronymic(sourcePassenger.getPatronymic());
        passenger.setSex(sourcePassenger.getSex());
        passenger.setNonRefundableTariff(sourcePassenger.isNonRefundableTariff());
        passenger.setTickets(new ArrayList<>());
        TrainTicket sourceTicket = sourcePassenger.getTicket();
        if (sourceTicket != null) {
            Ticket ticket = getTicket(sourcePassenger, payload, sourceTicket);
            passenger.getTickets().add(ticket);
        }
        var sourceInsurance = sourcePassenger.getInsurance();
        if (sourceInsurance != null) {
            var insurance = new Insurance();
            passenger.setInsurance(insurance);
            insurance.setAmount(sourceInsurance.getAmount());
            insurance.setCompensation(sourceInsurance.getCompensation());
        }
        return passenger;
    }

    private TrainInfo getTrainInfo(TrainReservation payload) {
        var result = TrainInfo.builder().build();
        result.setSuburban(payload.isSuburban());
        result.setTrainTitle(payload.getReservationRequestData().getTrainTitle());
        result.setStartStationTitle(trainDictionaryMapService.getStationTitleByExpressName(
                payload.getReservationRequestData().getImInitialStationName()));
        result.setEndStationTitle(trainDictionaryMapService.getStationTitleByExpressName(
                payload.getReservationRequestData().getImFinalStationName()));
        result.setStartSettlementTitle(payload.getReservationRequestData().getStartSettlementTitle());
        result.setEndSettlementTitle(payload.getReservationRequestData().getEndSettlementTitle());
        result.setBrandTitle(payload.getReservationRequestData().getBrandTitle());
        result.setTrainNumber(payload.getReservationRequestData().getTrainNumber());
        result.setTrainTicketNumber(payload.getReservationRequestData().getTrainTicketNumber());
        return result;
    }

    public Instant getServiceReloadAt(TOrderServiceInfo orderServiceInfo) {
        TServiceInfo source = orderServiceInfo.getServiceInfo();
        var payload = ProtoUtils.fromTJson(source.getPayload(), TrainReservation.class);
        Instant confirmedAt = ProtoUtils.toInstantFromSafe(orderServiceInfo.getConfirmedAt());
        var warnings = getOrderWarnings(payload, confirmedAt, false);
        Instant now = Instant.now();
        return Stream.concat(warnings.stream().flatMap(x -> Stream.of(x.getFrom(), x.getTo())),
                        payload.getPassengers().stream()
                                .map(TrainPassenger::getTicket)
                                .filter(Objects::nonNull)
                                .flatMap(x -> Stream.of(
                                        x.getCanChangeElectronicRegistrationTill(), x.calculateCanReturnTill(payload))))
                .filter(Objects::nonNull)
                .filter(now::isBefore)
                .min(Comparator.naturalOrder()).orElse(null);
    }
}
