package ru.yandex.travel.api.services.orders;

import java.time.ZoneId;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.booking_flow.model.DisplayableOrderStatus;
import ru.yandex.travel.api.models.train.Direction;
import ru.yandex.travel.api.models.train.TrainOrderMaps;
import ru.yandex.travel.api.models.travel_orders.TrainOrderListItemV2;
import ru.yandex.travel.api.services.orders.model.TrainOrderListInfo;
import ru.yandex.travel.api.services.orders.model.TrainOrderListInfoPassenger;
import ru.yandex.travel.api.services.orders.model.TrainOrderListSegmentInfo;
import ru.yandex.travel.api.services.orders.model.TrainOrderListTrainInfo;
import ru.yandex.travel.api.services.train.TrainApiProxy;
import ru.yandex.travel.api.services.train.TrainProxyOrderDTO;
import ru.yandex.travel.api.services.train.TrainProxyOrderPassengerInfo;
import ru.yandex.travel.api.services.train.TrainProxyOrderStatus;
import ru.yandex.travel.api.services.train.TrainProxyOrderTicket;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;

@Service
@Slf4j
@RequiredArgsConstructor
public class TrainOrderListServiceV2 implements TravelOrderSubtypeListFetcher {

    private static final String REFUNDED_RZHD_STATUS = "REFUNDED";
    private final static BiMap<TrainProxyOrderStatus, DisplayableOrderStatus> STATUS_MAPPING =
            ImmutableBiMap.of(TrainProxyOrderStatus.IN_PROGRESS, DisplayableOrderStatus.IN_PROGRESS,
                    TrainProxyOrderStatus.CANCELLED, DisplayableOrderStatus.CANCELLED,
                    TrainProxyOrderStatus.RESERVED, DisplayableOrderStatus.AWAITS_PAYMENT,
                    TrainProxyOrderStatus.DONE, DisplayableOrderStatus.FULFILLED);
    private final TrainApiProxy trainApiProxy;
    private final TrainDictionaryMapService trainDictionaryMapService;

    @Override
    public CompletableFuture<TravelOrderSubtypeList> listOrders(int offset, int limit,
                                                                DisplayableOrderStatus displayableOrderStatus,
                                                                String searchTerm) {
        return trainApiProxy.listTrainOrders(offset, limit,
                STATUS_MAPPING.inverse().get(displayableOrderStatus), searchTerm)
                .thenApply(list -> {
                    TravelOrderSubtypeList res = new TravelOrderSubtypeList();
                    res.setOffset(offset);
                    res.setLimit(limit);
                    res.setHasMoreOrders(list.getCount() > offset + limit);
                    res.setOrders(list.getResults().stream()
                            .map(this::convertOrderListItem)
                            .collect(Collectors.toList()));
                    return res;
                });
    }


    private TrainOrderListItemV2 convertOrderListItem(TrainProxyOrderDTO tpOrder) {
        TrainOrderListItemV2 result = new TrainOrderListItemV2();
        TrainOrderListInfo info = new TrainOrderListInfo();

        result.setId(tpOrder.getUid());
        result.setYandexOrderId(tpOrder.getOrderNumber());
        info.setReservedTo(tpOrder.getReservedTo().toInstant());
        if (tpOrder.getMaxPendingTill() != null) {
            info.setMaxPendingTill(tpOrder.getMaxPendingTill().toInstant());
        }
        info.setReservationNumber(tpOrder.getOrderNumber());

        info.setPassengers(new ArrayList<>());

        Money totalAmount = Money.zero(ProtoCurrencyUnit.RUB);
        Money totalRefundAmount = Money.zero(ProtoCurrencyUnit.RUB);
        int refundedTicketsCount = 0;

        for (TrainProxyOrderPassengerInfo sourcePassenger : tpOrder.getPassengers()) {
            var passenger = new TrainOrderListInfoPassenger();
            info.getPassengers().add(passenger);
            passenger.setFirstName(sourcePassenger.getFirstName());
            passenger.setLastName(sourcePassenger.getLastName());
            TrainProxyOrderTicket sourceTicket = sourcePassenger.getTickets().get(0);

            Money ticketAmount = Money.of(sourceTicket.getAmount(), ProtoCurrencyUnit.RUB);
            totalAmount = totalAmount.add(ticketAmount);
            if (!tpOrder.isInsuranceAutoReturn()) {
                if (sourcePassenger.getInsurance() != null && sourcePassenger.getInsurance().isOrdered()) {
                    totalAmount = totalAmount.add(Money.of(sourcePassenger.getInsurance().getAmount(),
                            ProtoCurrencyUnit.RUB));
                }
            }

            // nasty hack corresponding to current frontend logic
            if (REFUNDED_RZHD_STATUS.equals(sourceTicket.getRzhdStatus())) {
                refundedTicketsCount += 1;
                if (sourceTicket.getRefund() != null) {
                    if (sourceTicket.getRefund().getAmount() != null) {
                        totalRefundAmount = totalRefundAmount.add(Money.of(sourceTicket.getRefund().getAmount(),
                                ProtoCurrencyUnit.RUB));
                    }
                    if (sourcePassenger.getInsurance() != null && sourcePassenger.getInsurance().isOrdered()) {
                        if (sourcePassenger.getInsurance().getAmount() != null) {
                            totalRefundAmount =
                                    totalRefundAmount.add(
                                            Money.of(sourcePassenger.getInsurance().getAmount(), ProtoCurrencyUnit.RUB)
                                    );
                        }
                    }
                }
            }

            passenger.setTicketRzhdStatus(sourceTicket.getRzhdStatus());

        }

        info.setDeparture(tpOrder.getDeparture().toInstant());
        info.setArrival(tpOrder.getArrival().toInstant());
        info.setCarNumber(tpOrder.getCarNumber());
        info.setCarType(TrainOrderMaps.CAR_TYPE_FROM_STR.get(tpOrder.getCarType()));
//        result.setCoachOwner(tpOrder.get payload.getCarrier());
        info.setStationFrom(trainDictionaryMapService.createStationInfoById(tpOrder.getStationFrom().getId()));
        info.setStationTo(trainDictionaryMapService.createStationInfoById(tpOrder.getStationTo().getId()));
        info.setTrainInfo(new TrainOrderListTrainInfo());
        info.getTrainInfo().setSuburban(tpOrder.isSuburban());
        info.getTrainInfo().setBrandTitle(tpOrder.getBrandTitle());
        info.getTrainInfo().setTrainTitle(tpOrder.getTrainTitle());
        info.getTrainInfo().setTrainNumber(tpOrder.getTrainNumber());
        info.getTrainInfo().setTrainTicketNumber(tpOrder.getTrainTicketNumber());
        if (tpOrder.getStartStation() != null) {
            info.getTrainInfo().setStartSettlementTitle(tpOrder.getStartStation().getSettlementTitle());
        } else {
            info.getTrainInfo().setStartSettlementTitle(tpOrder.getStationFrom().getSettlementTitle());
        }
        if (tpOrder.getEndStation() != null) {
            info.getTrainInfo().setEndSettlementTitle(tpOrder.getEndStation().getSettlementTitle());
        } else {
            info.getTrainInfo().setEndSettlementTitle(tpOrder.getStationTo().getSettlementTitle());
        }
        info.setCustomerEmail(tpOrder.getUserInfo().getEmail());
        info.setCustomerPhone(tpOrder.getUserInfo().getPhone());
        if (tpOrder.getTicketsStatusFreezesAt() != null) {
            info.setCanChangeElectronicRegistrationTill(tpOrder.getTicketsStatusFreezesAt().toInstant());
        }
        info.setTotalAmount(totalAmount);
        info.setRefundedTicketsCount(refundedTicketsCount);
        info.setRefundAmount(totalRefundAmount);

        result.setOrderInfo(info);
        result.setStatus(STATUS_MAPPING.get(tpOrder.getTravelStatus()));
        result.setSource(OrderListSource.TRAIN_API);
        result.setOrderType(OrderType.TRAIN);

        result.setServicedAt(tpOrder.getDeparture().withZoneSameInstant(ZoneId.systemDefault()).toLocalDateTime());

        result.getOrderInfo().setSegments(List.of(convertToListInfo(tpOrder)));

        return result;
    }

    private TrainOrderListSegmentInfo convertToListInfo(TrainProxyOrderDTO order) {
        TrainOrderListSegmentInfo result = new TrainOrderListSegmentInfo();
        result.setDirection(Direction.FORWARD);
        result.setPassengers(new ArrayList<>());

        result.setReservationNumber(order.getOrderNumber());
//        result.setPartnerOrderId(order.get());
        result.setPassengers(new ArrayList<>());

        Money totalAmount = Money.zero(ProtoCurrencyUnit.RUB);

        Money totalRefundAmount = Money.zero(ProtoCurrencyUnit.RUB);
        int refundedTicketsCount = 0;

        for (TrainProxyOrderPassengerInfo sourcePassenger : order.getPassengers()) {
            var passenger = new TrainOrderListInfoPassenger();
            result.getPassengers().add(passenger);
            passenger.setFirstName(sourcePassenger.getFirstName());
            passenger.setLastName(sourcePassenger.getLastName());
            TrainProxyOrderTicket sourceTicket = sourcePassenger.getTickets().get(0);

            Money ticketAmount = Money.of(sourceTicket.getAmount(), ProtoCurrencyUnit.RUB);
            totalAmount = totalAmount.add(ticketAmount);
            if (!order.isInsuranceAutoReturn()) {
                if (sourcePassenger.getInsurance() != null && sourcePassenger.getInsurance().isOrdered()) {
                    totalAmount = totalAmount.add(Money.of(sourcePassenger.getInsurance().getAmount(),
                            ProtoCurrencyUnit.RUB));
                }
            }

            // nasty hack corresponding to current frontend logic
            if (REFUNDED_RZHD_STATUS.equals(sourceTicket.getRzhdStatus())) {
                refundedTicketsCount += 1;
                if (sourceTicket.getRefund() != null) {
                    if (sourceTicket.getRefund().getAmount() != null) {
                        totalRefundAmount = totalRefundAmount.add(Money.of(sourceTicket.getRefund().getAmount(),
                                ProtoCurrencyUnit.RUB));
                    }
                    if (sourcePassenger.getInsurance() != null && sourcePassenger.getInsurance().isOrdered()) {
                        if (sourcePassenger.getInsurance().getAmount() != null) {
                            totalRefundAmount =
                                    totalRefundAmount.add(
                                            Money.of(sourcePassenger.getInsurance().getAmount(), ProtoCurrencyUnit.RUB)
                                    );
                        }
                    }
                }
            }

            passenger.setTicketRzhdStatus(sourceTicket.getRzhdStatus());

        }

        result.setDeparture(order.getDeparture().toInstant());
        result.setArrival(order.getArrival().toInstant());
        result.setCarNumber(order.getCarNumber());
        result.setCarType(TrainOrderMaps.CAR_TYPE_FROM_STR.get(order.getCarType()));
        result.setStationFrom(trainDictionaryMapService.createStationInfoById(order.getStationFrom().getId()));
        result.setStationTo(trainDictionaryMapService.createStationInfoById(order.getStationTo().getId()));
        result.setTrainInfo(new TrainOrderListTrainInfo());
        result.getTrainInfo().setSuburban(order.isSuburban());
        result.setTrainInfo(new TrainOrderListTrainInfo());
        result.getTrainInfo().setBrandTitle(order.getBrandTitle());
        result.getTrainInfo().setTrainTitle(order.getTrainTitle());
        result.getTrainInfo().setTrainNumber(order.getTrainNumber());
        result.getTrainInfo().setTrainTicketNumber(order.getTrainTicketNumber());
        if (order.getStartStation() != null) {
            result.getTrainInfo().setStartSettlementTitle(order.getStartStation().getSettlementTitle());
        } else {
            result.getTrainInfo().setStartSettlementTitle(order.getStationFrom().getSettlementTitle());
        }
        if (order.getEndStation() != null) {
            result.getTrainInfo().setEndSettlementTitle(order.getEndStation().getSettlementTitle());
        } else {
            result.getTrainInfo().setEndSettlementTitle(order.getStationTo().getSettlementTitle());
        }

        if (order.getTicketsStatusFreezesAt() != null) {
            result.setCanChangeElectronicRegistrationTill(order.getTicketsStatusFreezesAt().toInstant());
        }
        result.setTotalAmount(totalAmount);
        return result;
    }
}
